package io.agora.avc.app.meeting

import android.view.SurfaceView
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.core.view.isInvisible
import androidx.core.view.isVisible
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.RecyclerView
import com.bumptech.glide.Glide
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.NetworkQuality
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.AvcItemMeetingBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.extensions.getVideoStats
import io.agora.avc.extensions.isMySelf
import io.agora.avc.utils.StringUtils
import io.agora.logger.LogConverter
import io.agora.logger.Logger
import io.agora.rtc.Constants

/**
 * if receive metadata no user id,but update room host
 * then user host status will not refresh
 * so when id changed,need to changed host status
 */
class MeetingViewHolder constructor(
    val binding: AvcItemMeetingBinding
) : RecyclerView.ViewHolder(binding.root) {

    private var onItemListener: MeetingAdapter.OnItemListener? = null

    val observer = Observer<Boolean> {
        binding.mediaStatics.isVisible = it
    }

    fun bindUser(user: ARoomUser, showVideoData: Boolean) {
        setupContainer(user)
        //set up video
        bindName(user)
        bindVideoStatus(user)
        bindWatermark(user)
        //set up audio
        bindAudioStatus(user)
        bindSpeaking(user)
        bindShareStatus(user)
        bindQuality(user)
        setupMediaStatics(user, showVideoData)
        setListener(user)
    }

    private fun setListener(user: ARoomUser) {
        binding.smallBg.setOnClickListener {
            onItemListener?.onClicked(user)
        }
        binding.smallBg.setOnLongClickListener {
            onItemListener?.onLongClick(user)
            true
        }
    }

    private fun setupContainer(user: ARoomUser) {
        Logger.i(TAG, "setup container for user:${user.streamId}")
        binding.container.removeAllViews()
        if (user.videoState) {
            var view: View? = null
            val app = binding.container.context.applicationContext
            if (app is MyApplication) {
                val appController = app.appContainer.appController
                user.streamId.let { streamId ->
                    view = appController?.createTextureView(app)
                    appController?.startRenderVideo(
                        streamId,
                        view,
                        Constants.RENDER_MODE_HIDDEN
                    )
                }
            }
            view?.let {
                if (it is SurfaceView) {
                    it.setZOrderMediaOverlay(true)
                }
                binding.container.addView(
                    view,
                    FrameLayout.LayoutParams(
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.MATCH_PARENT
                    )
                )
            }
        }
    }

    fun bindVideoStatus(user: ARoomUser) {
        if (user.videoState) {
            openScreenVideo(user)
        } else {
            muteScreenVideo(user)
        }
    }

    fun bindName(user: ARoomUser) {
        binding.tvName.setTextKeepState(user.getConferenceNicknameMaybeAssistant())
    }

    private fun muteScreenVideo(user: ARoomUser) {
        Logger.i(
            TAG,
            "mute user:${user.streamId} video, status:${LogConverter.muted(!user.videoState)}"
        )

        binding.container.isVisible = false
        binding.streamDefaultBg.isVisible = true
        if (StringUtils.isEmpty(user.avatar)) {
            Logger.i(TAG, "avatar is ${user.avatar}")
        } else {
            Glide.with(binding.root.context)
                .load(user.avatar)
                .placeholder(R.drawable.avc_ic_head_portrait)
                .centerCrop()
                .into(binding.ivStreamDefaultBg)
        }
    }

    private fun openScreenVideo(user: ARoomUser) {
        Logger.i(
            TAG,
            "unMute user:${user.streamId} video, status:${LogConverter.muted(!user.videoState)}"
        )

        if (binding.container.childCount > 0) {
            val app = binding.container.context.applicationContext
            if (app is MyApplication) {
                val appController = app.appContainer.appController
                appController?.startRenderVideo(
                    user.streamId,
                    binding.container.getChildAt(0),
                    Constants.RENDER_MODE_HIDDEN
                )
            }
        } else {
            setupContainer(user)
        }

        binding.container.isVisible = true
        binding.streamDefaultBg.isVisible = false
    }

    fun bindAudioStatus(user: ARoomUser) {
        binding.audioIcon.isInvisible = user.isShareStream()
        binding.audioIcon.mute = !user.audioState
        binding.audioIcon.isMySelf = user.isMySelf()
    }

    fun setItemClickListener(listener: MeetingAdapter.OnItemListener?) {
        this.onItemListener = listener
    }

    fun bindQuality(user: ARoomUser) {
        if (!user.isMySelf()
            && (user.videoState || user.audioState)
            && NetworkQuality.allowShowMile(user.quality)
        ) {
            user.quality.let {
                binding.signalIcon.setQuality(it)
            }
            binding.signalIcon.isVisible = true
        } else {
            binding.signalIcon.isVisible = false
        }
    }

    //user speaking
    fun bindSpeaking(user: ARoomUser) {
        if (user.isMySelf()) {
            binding.audioIcon.volume = user.volume
        } else {
            binding.audioIcon.speaking = user.isSpeaking
        }
    }

    fun bindShareStatus(user: ARoomUser) {
        binding.ivScreenShare.isVisible = user.isShareStream()
    }

    fun setupMediaStatics(user: ARoomUser, showVideoData: Boolean) {
        if (showVideoData) {
            binding.mediaStatics.setVideoStatics(user.getVideoStats())
            binding.mediaStatics.setVolume(user.volume)
            binding.mediaStatics.visibility = View.VISIBLE
        } else {
            binding.mediaStatics.visibility = View.GONE
        }
    }

    fun release() {
        onItemListener = null
        binding.container.removeAllViews()
    }

    fun bindWatermark(user: ARoomUser) {
        if (user.hasWatermark && user.videoState) {
            val app = binding.container.context.applicationContext
            var localUser: LocalUser? = null
            if (app is MyApplication) {
                localUser = app.appContainer.localUser
            }
            binding.watermark.setMarkText(
                localUser?.getConferenceNicknameMaybeAssistant()
                    ?: StringUtils.getString(R.string.avc_unknown_user)
            )
            binding.watermark.isVisible = true
        } else {
            binding.watermark.isVisible = false
        }
    }

    fun bindAssistant(user: ARoomUser) {
        setListener(user)
    }

    companion object {
        private const val TAG = "[UI][MeetingViewHolder]"
    }
}