package io.agora.avc.app.mine

import android.app.Application
import android.graphics.Bitmap
import android.media.ExifInterface
import androidx.appcompat.app.AppCompatDelegate
import com.bumptech.glide.load.resource.bitmap.TransformationUtils
import io.agora.avc.R
import io.agora.avc.auth.signin.SignInManager
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.Version
import io.agora.avc.manager.logupload.LogUploader
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.repository.MediaConfigRepository
import io.agora.avc.repository.UserRepository
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.ImageUtils
import io.agora.avc.utils.SDKUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.ResponseBody
import java.io.IOException
import javax.inject.Inject

class MineViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val userRepository: UserRepository,
    private val mediaConfigRepository: MediaConfigRepository,
    private val appConfigRepository: AppConfigRepository,
    private val logUploader: LogUploader,
    private val developerRepository: DeveloperRepository,
    private val signInManager: SignInManager
) : AppViewModel(application, appController) {
    val userNameLiveData: EventLiveData<Boolean> = EventLiveData()
    val uploadPortraitLiveData: EventLiveData<String> = EventLiveData()
    val cameraPicLiveData: EventLiveData<Bitmap> = EventLiveData()
    val uploadLogLiveData: EventLiveData<Boolean> = EventLiveData()
    val statusContentChangedLiveData: EventLiveData<Int> = EventLiveData()
    val versionTagEvent: EventLiveData<Version> = EventLiveData()
    val renderingEvent: EventLiveData<MineVo> = EventLiveData()

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        changeVersionDescribe(0)
        notifyUIChanged()
    }

    fun updateNickName(oldName: String?, newNick: String) {
        if (oldName == newNick) {
            Logger.w(TAG, "The new and old names are the same, no need to modify")
            return
        }
        Logger.i(TAG, "update user nickName, old:$oldName, new:$newNick")
        statusEvent.postValue(StatusEvent.Status.LOADING)
        statusContentChangedLiveData.postValue(R.string.avc_modifying_user_name)
        userRepository.updateUserName(newNick)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(it: BaseNo?) {
                    if (it != null && it.success) {
                        statusEvent.postValue(StatusEvent.Status.SUCCESS)
                    } else {
                        statusEvent.postValue(StatusEvent.Status.ERROR)
                    }
                    userNameLiveData.value = true
                }

                override fun onFail(t: BaseNo?) {
                    statusEvent.postValue(StatusEvent.Status.ERROR)
                }
            })
    }

    fun uploadLog(platform: String, deviceName: String) {
        statusEvent.postValue(StatusEvent.Status.LOADING)
        statusContentChangedLiveData.postValue(R.string.avc_uploading)
        userRepository.getLocalUser(platform, deviceName)
            .flatMap { user ->
                logUploader.uploadLog(user.streamId)
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<ResponseBody> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(it: ResponseBody) {
                    statusEvent.postValue(StatusEvent.Status.SUCCESS)
                    uploadLogLiveData.value = true
                }

                override fun onError(e: Throwable) {
                    statusEvent.postValue(StatusEvent.Status.ERROR)
                    uploadLogLiveData.value = false
                }

                override fun onComplete() {

                }
            })
    }

    fun changeTheme(darkTheme: Boolean) {
        if (darkTheme) {
            appConfigRepository.saveDarkTheme(AppCompatDelegate.MODE_NIGHT_YES)
            AppCompatDelegate.setDefaultNightMode(AppCompatDelegate.MODE_NIGHT_YES)
        } else {
            appConfigRepository.saveDarkTheme(AppCompatDelegate.MODE_NIGHT_NO)
            AppCompatDelegate.setDefaultNightMode(AppCompatDelegate.MODE_NIGHT_NO)
        }
    }

    fun handleCameraPic(currentPhotoPath: String) {
        Observable
            .create<Bitmap> {
                val maxWidth = 512
                val maxHeight = 512
                val bitmap = ImageUtils.getBitmap(currentPhotoPath, maxWidth, maxHeight)
                val result = rotateImageIfRequired(bitmap, currentPhotoPath)
                it.onNext(result)
            }
            .subscribeOn(Schedulers.io())
            .subscribe(object : io.reactivex.Observer<Bitmap> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Bitmap) {
                    cameraPicLiveData.postValue(t)
                }

                override fun onError(e: Throwable) {
                    cameraPicLiveData.postValue(null)
                }
            })
    }

    @Throws(IOException::class)
    private fun rotateImageIfRequired(img: Bitmap, path: String): Bitmap {
        val ei = ExifInterface(path)

        val orientation =
            ei.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL)

        return when (orientation) {
            ExifInterface.ORIENTATION_ROTATE_90 -> TransformationUtils.rotateImage(img, 90)
            ExifInterface.ORIENTATION_ROTATE_180 -> TransformationUtils.rotateImage(img, 180)
            ExifInterface.ORIENTATION_ROTATE_270 -> TransformationUtils.rotateImage(img, 270)
            else -> img
        }
    }

    fun logoutViaWeWork() {
        signInManager.signOut()
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                notifyUIChanged()
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(AppEvent.LOCAL_USER_CHANGED)
    }

    fun setVideoSetting(it: Boolean) {
        mediaConfigRepository.saveVideoSetting(it)
    }

    fun setAudioSetting(it: Boolean) {
        mediaConfigRepository.saveAudioSetting(it)
    }

    private fun notifyUIChanged() {
        val mineVo = MineVo(
            getLocalUser(),
            mediaConfigRepository.getVideoSetting(),
            mediaConfigRepository.getAudioSetting(),
            appConfigRepository.getDarkTheme(),
            appConfigRepository.getPictureMode(),
            mediaConfigRepository.getResolutionOption(),
            developerRepository.getDeveloperOptions()
        )
        renderingEvent.postValue(mineVo)
    }

    fun versionTagClicked(tag: Int) {
        changeVersionDescribe(tag)
    }

    private fun changeVersionDescribe(tag: Int) {
        val version = when (tag) {
            1 -> {
                Version(SDKUtils.getRTCSdkVersion(), 2)
            }
            2 -> {
                Version(SDKUtils.getRTMSdkVersion(), 0)
            }
            else -> {
                Version(AppUtils.getAppVersionName() + "." + AppUtils.getAppVersionCode(), 1)
            }
        }
        versionTagEvent.postValue(version)
    }

    fun setPictureMode(bool: Boolean) {
        appConfigRepository.savePictureMode(bool)
    }

    fun onResolutionChanged(resolution: Int) {
        mediaConfigRepository.saveResolutionOption(resolution)
    }

    fun saveDeveloperVisible(show: Boolean) {
        developerRepository.getDeveloperOptions().showDeveloper = show
        notifyUIChanged()
    }

    companion object {
        private const val TAG = "[VM][Mine]"
    }
}