package io.agora.avc.app.operation

import android.Manifest
import android.content.Context
import android.content.res.Configuration
import android.os.Bundle
import android.text.SpannableStringBuilder
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.widget.FrameLayout
import androidx.annotation.IntDef
import androidx.appcompat.app.AlertDialog
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.navigation.fragment.NavHostFragment
import androidx.navigation.fragment.findNavController
import com.agora.valoran.Constants
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.app.meeting.MeetingFragment
import io.agora.avc.bo.AssistantInfo
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Room
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.dao.PermissionDao
import io.agora.avc.databinding.AvcFragmentOperationBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.extensions.isMySelf
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.*
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.logger.Logger

class OperationFragment : BaseSheetDialogFragment<OperationViewModel, AvcFragmentOperationBinding>() {

    private var operationUser: ARoomUser? = null

    @OperationType
    private var operationType = OperationType.MEDIA_ASSISTANT

    private var localAssistState = Constants.AssistState.DISABLED.value
    private var assistantInfo: AssistantInfo? = null

    private var roomInfo: Room? = null

    private var hasMediaPermission: Boolean = true
    private var assistantActionEnable: Boolean = true

    private var requestDialog: AlertDialog? = null

    private var initMediaActionVisible = false
    private var initAssistantActionVisible = false

    private var openAudioConfirm = false

    private val textColorSecondary by lazy {
        val typedValue = TypedValue()
        context?.theme?.resolveAttribute(android.R.attr.textColorSecondary, typedValue, true)
        typedValue.data
    }

    private val primaryColor by lazy {
        val typedValue = TypedValue()
        context?.theme?.resolveAttribute(R.attr.colorPrimary, typedValue, true)
        typedValue.data
    }

    private var mainViewModel: MainViewModel? = null

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_operation
    }

    override fun allocObserver() {
        mViewModel?.remoteUserChangedLiveData?.observe(this) { user ->
            user?.let {
                this.operationUser = it
                assistantActionEnable = operationUser?.isMySelf() == false
                        && operationUser?.isAssistant == true
                if (!initAssistantActionVisible) {
                    initAssistantActionVisible = true
                    if (assistantActionEnable) {
                        setAssistantTitleAndOperationText()
                    }
                    setAssistantActionVisible(assistantActionEnable)
                }
                renderUI()
            }
        }

        mViewModel?.permissionChangedLiveData?.observe(this) {
            hasMediaPermission = it
            if (operationType == OperationType.MEDIA_ASSISTANT && !initMediaActionVisible) {
                initMediaActionVisible = true
                setMediaActionsVisible(hasMediaPermission)
            }
            renderUI()
        }

        mViewModel?.roomInfoChangedLiveData?.observe(this) {
            this.roomInfo = it
            renderUI()
        }

        mViewModel?.localAssistStateChangedLiveData?.observe(this) {
            localAssistState = it
            if (assistantActionEnable) {
                setAssistantTitleAndOperationText()
            }
        }

        mViewModel?.assistantChangedLiveData?.observe(this) {
            assistantInfo = it
            if (assistantActionEnable) {
                setAssistantTitleAndOperationText()
            }
        }

        mViewModel?.closePageEvent?.observe(this) {
            findNavController().navigateUp()
        }

        mViewModel?.attendeesCountChanged?.observe(this) {
            openAudioConfirm = it >= MeetingFragment.OPEN_AUDIO_NEED_CONFIRM_ATTENDEES_NUMBER
        }
    }

    private fun showVideoRequestConfirmDialog(user: ARoomUser, on: Boolean) {
        context?.apply {
            if (isRequestDialogShowing()) {
                requestDialog?.dismiss()
            }

            requestDialog =
                MaterialAlertDialogBuilder(this, R.style.avc_CustomMaterialAlertDialog)
                    .setTitle(R.string.avc_start_request)
                    .setCancelable(false)
                    .setMessage(
                        getMediaRequestMessage(
                            this,
                            false,
                            on,
                            user.getConferenceNicknameMaybeAssistant()
                        )
                    )
                    .setNegativeButton(R.string.avc_cancel) { _, _ -> }
                    .setPositiveButton(R.string.avc_confirm) { _, _ ->
                        Logger.i(TAG, "I request mute remote user(${user.streamId}) video")
                        mViewModel?.setUserVideo(user, on)
                        onBackPressed()
                    }
                    .show()
        }
    }

    private fun getMediaRequestMessage(
        context: Context,
        isAudio: Boolean,
        on: Boolean,
        names: CharSequence?
    ): String {
        val userName = names ?: context.getString(R.string.avc_unknown_user)
        return if (isAudio) {
            if (on) {
                StringUtils.getString(R.string.avc_peer_invite_microphone_confirm, userName)
            } else {
                StringUtils.getString(R.string.avc_peer_invite_microphone_mute_confirm, userName)
            }
        } else {
            if (on) {
                StringUtils.getString(R.string.avc_peer_invite_camera_confirm, userName)
            } else {
                StringUtils.getString(R.string.avc_peer_invite_camera_mute_confirm, userName)
            }
        }
    }

    private fun showAudioRequestConfirmDialog(user: ARoomUser, on: Boolean) {
        context?.apply {
            if (isRequestDialogShowing()) {
                requestDialog?.dismiss()
            }
            requestDialog =
                MaterialAlertDialogBuilder(this, R.style.avc_CustomMaterialAlertDialog)
                    .setTitle(R.string.avc_start_request)
                    .setCancelable(false)
                    .setMessage(
                        getMediaRequestMessage(
                            this,
                            true,
                            on,
                            user.getConferenceNicknameMaybeAssistant()
                        )
                    )
                    .setNegativeButton(R.string.avc_cancel) { _, _ -> }
                    .setPositiveButton(R.string.avc_confirm) { _, _ ->
                        Logger.i(TAG, "I request mute remote user(${user.streamId}) audio")
                        mViewModel?.setUserAudio(user, on)
                        onBackPressed()
                    }
                    .show()
        }
    }

    private fun renderUI() {
        if (operationType == OperationType.CLAIM_ASSISTANT) {
            renderClaimAssistantUI()
        } else {
            renderMediaAssistantUI()
        }
    }

    private fun renderMediaAssistantUI() {
        setTitle()
        //media
        val audioOpen = operationUser?.audioState ?: false
        val videoOpen = operationUser?.videoState ?: false
        mBinding.ivAudio.opened = audioOpen
        mBinding.ivAudio.loading = operationUser?.isAudioPending ?: false
        mBinding.tvAudio.text =
            getString(if (audioOpen) R.string.avc_audio_operation_disable_text else R.string.avc_audio_operation_enable_text)

        mBinding.ivVideo.opened = videoOpen
        mBinding.ivVideo.loading = operationUser?.isVideoPending ?: false
        mBinding.tvVideo.text =
            getString(if (videoOpen) R.string.avc_video_operation_disable_text else R.string.avc_video_operation_enable_text)

        mBinding.audioAction.isClickable = hasMediaPermission && !mBinding.ivAudio.loading
        mBinding.ivAudio.isActivated = hasMediaPermission
        mBinding.tvAudio.isActivated = hasMediaPermission
        mBinding.videoAction.isClickable = hasMediaPermission && !mBinding.ivVideo.loading
        mBinding.ivVideo.isActivated = hasMediaPermission
        mBinding.tvVideo.isActivated = hasMediaPermission
        mBinding.llKickOut.isClickable = hasMediaPermission
        mBinding.ivKickOut.isActivated = hasMediaPermission
        mBinding.tvKickOut.isActivated = hasMediaPermission

        //assistant
        mBinding.llAssistantOperation1.isClickable = assistantActionEnable
        mBinding.llAssistantOperation2.isClickable = assistantActionEnable
    }

    private fun renderClaimAssistantUI() {
        mBinding.llAssistantOperation1.isVisible = true
        mBinding.llAssistantOperation2.isVisible = true
        setAssistantTitleAndOperationText()
    }

    private fun setTitle() {
        if (assistantActionEnable) {
            setAssistantTitleAndOperationText()
        } else {
            mBinding.tvTitle.text = getUserName()
        }
    }

    private fun setAssistantTitleAndOperationText() {
        when {
            operationType == OperationType.CLAIM_ASSISTANT -> {//1、claim assistant
                mBinding.tvTitle.text = getString(R.string.avc_as_setting_title)
                mBinding.tvAssistantOperation1.text = getString(R.string.avc_as_setting_choice_en)
                mBinding.tvAssistantOperation2.text = getString(R.string.avc_as_setting_choice_cn)
            }
            localAssistState == Constants.AssistState.DISABLED.value
                    && assistantInfo?.type == Constants.TransLangType.EN.value
            -> {//2、Meeting assistant(English) not opened
                mBinding.tvTitle.text = getString(R.string.avc_as_open_title_en)
                mBinding.tvAssistantOperation1.text = getString(R.string.avc_as_open_button_original)
                mBinding.tvAssistantOperation2.text = getString(R.string.avc_as_open_button_listen)
            }
            localAssistState == Constants.AssistState.DISABLED.value
                    && assistantInfo?.type == Constants.TransLangType.ZH.value
            -> {//3、Meeting assistant(Chinese) not opened
                mBinding.tvTitle.text = getString(R.string.avc_as_open_title_cn)
                mBinding.tvAssistantOperation1.text = getString(R.string.avc_as_open_button_original)
                mBinding.tvAssistantOperation2.text = getString(R.string.avc_as_open_button_listen)
            }
            localAssistState == Constants.AssistState.ENABLE_WITH_ORIGIN_SOUND.value -> {
                mBinding.tvAssistantOperation2.text =
                    getString(R.string.avc_as_close_button_orginal_close)
            }
            localAssistState == Constants.AssistState.ENABLE_WITHOUT_ORIGIN_SOUND.value -> {
                mBinding.tvAssistantOperation2.text =
                    getString(R.string.avc_as_close_button_orginal_open)
            }
            else -> {
                mBinding.tvTitle.text = getString(R.string.avc_as_close_title)
                mBinding.tvAssistantOperation1.text = getString(R.string.avc_as_close_button_close)
                mBinding.tvAssistantOperation2.text =
                    getString(R.string.avc_as_close_button_orginal_close)
            }
        }
    }

    private fun setMediaActionsVisible(hasMediaPermission: Boolean) {
        mBinding.audioAction.isVisible = hasMediaPermission
        mBinding.videoAction.isVisible = hasMediaPermission
        mBinding.llKickOut.isVisible =
            operationUser !is LocalUser && hasMediaPermission && operationUser?.isAssistant != true
    }

    private fun setAssistantActionVisible(canOperationAssistant: Boolean) {
        mBinding.llAssistantOperation1.isVisible = canOperationAssistant
        mBinding.llAssistantOperation2.isVisible = canOperationAssistant
    }

    private fun getUserName(): SpannableStringBuilder {
        val spanUtils = SpanUtils()
        if (roomInfo?.isInternal() == true && operationUser?.isThirdPartyLoggedIn == true) {
            spanUtils
                .append(
                    operationUser?.thirdPartyName
                        ?: StringUtils.getString(R.string.avc_unknown_user)
                )
            if (operationUser?.thirdPartyAlias?.isNotEmpty() == true) {
                spanUtils
                    .append("\n")
                    .append(operationUser?.thirdPartyAlias ?: "")
                    .setFontSize(ConvertUtils.sp2px(12f))
                    .setForegroundColor(textColorSecondary)
            }
            return spanUtils.create()
        }
        spanUtils.append(operationUser?.name ?: StringUtils.getString(R.string.avc_unknown_user))
        return spanUtils.create()
    }

    private fun kickOut(user: ARoomUser) {
        context?.apply {
            if (isRequestDialogShowing()) {
                requestDialog?.dismiss()
            }
            requestDialog =
                MaterialAlertDialogBuilder(this, R.style.avc_CustomMaterialAlertDialog)
                    .setTitle(R.string.avc_kick_out)
                    .setCancelable(false)
                    .setMessage(
                        getString(
                            R.string.avc_pop_confirm_kick,
                            user.getConferenceNicknameMaybeAssistant()
                        )
                    )
                    .setPositiveButton(R.string.avc_kick_out_action) { _, _ ->
                        mViewModel?.kickOut(user)
                        onBackPressed()
                    }
                    .setNegativeButton(R.string.avc_cancel, null)
                    .show()
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        arguments?.getString(ARGUMENT_USER)?.let {
            mViewModel?.saveData(it)
        }
        arguments?.getInt(OPERATION_TYPE).let {
            if (it == null) {
                Logger.e(
                    TAG, "If the parameter operationType is not passed," +
                            " the pop-up window will be closed automatically"
                )
                NavHostFragment.findNavController(this@OperationFragment).navigateUp()
            } else {
                operationType = it
            }
        }
    }

    override fun onStart() {
        super.onStart()
        mDialog.setCanceledOnTouchOutside(true)
        onConfigurationChanged(resources.configuration)
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            switch2Landscape()
        } else {
            switch2Portrait()
        }
    }

    private fun switch2Portrait() {
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.width = ScreenUtils.getScreenWidth()
        layoutParams.gravity = Gravity.START
        mBinding.container.layoutParams = layoutParams
        mBehavior?.skipCollapsed = true
        mBehavior?.state = BottomSheetBehavior.STATE_EXPANDED
    }

    private fun switch2Landscape() {
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.width = ScreenUtils.getScreenWidth() / 2
        layoutParams.gravity = Gravity.END
        mBinding.container.layoutParams = layoutParams
        mBehavior?.skipCollapsed = true
        mBehavior?.state = BottomSheetBehavior.STATE_EXPANDED
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mainViewModel =
            getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)

        mBinding.touchOutside.setOnClickListener {
            onBackPressed()
        }

        mBinding.audioAction.setOnClickListener {
            Logger.i(TAG, "local user click audio button, operationUser:$operationUser")
            operationUser?.let {
                val on = !mBinding.ivAudio.opened
                if (it !is LocalUser) {
                    showAudioRequestConfirmDialog(it, on)
                } else {
                    openLocalAudio(it, on)
                }
            }
        }

        mBinding.videoAction.setOnClickListener {
            Logger.i(TAG, "local user click video button, operationUser:$operationUser")
            operationUser?.let {
                val on = !mBinding.ivVideo.opened
                if (it !is LocalUser) {
                    showVideoRequestConfirmDialog(it, on)
                } else {
                    openLocalVideo(it, on)
                }
            }
        }

        mBinding.llKickOut.setOnClickListener {
            Logger.i(TAG, "local user click hangup button, operationUser:$operationUser")
            operationUser?.let {
                kickOut(it)
            }
        }

        mBinding.llAssistantOperation1.setOnClickListener {
            when (mBinding.tvAssistantOperation1.text?.toString()) {
                getString(R.string.avc_as_open_button_original) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_open_button_original)}]"
                    )
                    mViewModel.enableAssist(false)
                    onBackPressed()
                }
                getString(R.string.avc_as_close_button_close) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_close_button_close)}]"
                    )
                    mViewModel.disableAssist()
                    onBackPressed()
                }
                getString(R.string.avc_as_setting_choice_en) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_setting_choice_en)}]"
                    )
                    mViewModel.applyAssistant(Constants.TransLangType.EN)
                    onBackPressed()
                }
            }
        }

        mBinding.llAssistantOperation2.setOnClickListener {
            when (mBinding.tvAssistantOperation2.text?.toString()) {
                getString(R.string.avc_as_open_button_listen) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_open_button_listen)}]"
                    )
                    mViewModel.enableAssist(true)
                    onBackPressed()
                }
                getString(R.string.avc_as_close_button_orginal_close) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_close_button_orginal_close)}]"
                    )
                    mViewModel.enableAssist(false)
                    onBackPressed()
                }
                getString(R.string.avc_as_close_button_orginal_open) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_close_button_orginal_close)}]"
                    )
                    mViewModel.enableAssist(true)
                    onBackPressed()
                }
                getString(R.string.avc_as_setting_choice_cn) -> {
                    Logger.i(
                        TAG,
                        "click assistant function[${getString(R.string.avc_as_setting_choice_cn)}]"
                    )
                    mViewModel.applyAssistant(Constants.TransLangType.ZH)
                    onBackPressed()
                }
            }
        }
    }

    private fun openLocalVideo(user: LocalUser, on: Boolean) {
        PermissionUtils
            .permission(PermissionConstants.CAMERA)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(
                            permissionsGranted,
                            getRequestPermission()
                        )
                    ) {
                        mViewModel?.setUserVideo(user, on)
                        onBackPressed()
                    }
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.CAMERA)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel?.onPermissionDenied(
                        permissionsDeniedForever,
                        permissionsDenied
                    )
                }
            })
            .request()

    }

    private fun openLocalAudio(user: LocalUser, on: Boolean) {
        if (on && openAudioConfirm) {
            context?.let {
                requestDialog?.dismiss()
                requestDialog = MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setCancelable(false)
                    .setMessage(it.getString(R.string.avc_microphone_multiplayer_request))
                    .setNegativeButton(R.string.avc_no_button) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_yes_button) { _, _ ->
                        checkPermissionOnLocalAudioClicked(user, on)
                    }
                    .show()
            }
        } else {
            checkPermissionOnLocalAudioClicked(user, on)
        }
    }

    private fun checkPermissionOnLocalAudioClicked(
        user: LocalUser,
        on: Boolean
    ) {
        PermissionUtils
            .permission(PermissionConstants.MICROPHONE)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(
                            permissionsGranted,
                            getRequestPermission()
                        )
                    ) {
                        mViewModel?.setUserAudio(user, on)
                        onBackPressed()
                    }
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.RECORD_AUDIO)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel?.onPermissionDenied(
                        permissionsDeniedForever,
                        permissionsDenied
                    )
                }
            })
            .request()
    }

    override fun onStop() {
        super.onStop()
    }

    override fun onDestroyView() {
        requestDialog?.dismiss()
        requestDialog = null
        super.onDestroyView()
    }

    private fun isRequestDialogShowing(): Boolean {
        return requestDialog?.isShowing == true
    }

    companion object {
        private const val TAG = "[UI][Operation]"
        const val ARGUMENT_USER = "argument_user"
        const val OPERATION_TYPE = "operation_type"

        fun navigateTo(
            fragment: Fragment,
            user: ARoomUser,
            @OperationType operationType: Int = OperationType.MEDIA_ASSISTANT,
        ) {
            Logger.i(
                TAG, "navigateTo," +
                        " from:${fragment::class.java.simpleName}," +
                        " user:${user.streamId}," +
                        " operationType:$operationType"
            )
            fragment.safeNavigate(
                R.id.action_global_operation,
                bundleOf(
                    ARGUMENT_USER to GsonUtils.toJson(user),
                    OPERATION_TYPE to operationType,
                )
            )
        }
    }

    @IntDef(
        OperationType.MEDIA_ASSISTANT,
        OperationType.CLAIM_ASSISTANT,
    )
    annotation class OperationType {
        companion object {
            /**
             * media + (use assistant)
             */
            const val MEDIA_ASSISTANT = 1

            /**
             * claim assistant
             */
            const val CLAIM_ASSISTANT = 2
        }
    }
}
