package io.agora.avc.app.rating.audio

import android.app.Application
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.MosScore
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.manager.rating.call.AudioScore
import io.agora.avc.manager.rating.call.CallRatingManager
import io.agora.avc.repository.RoomRepository
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Logger
import javax.inject.Inject

class AudioRatingViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val callRatingManager: CallRatingManager,
    private val roomRepository: RoomRepository
) : AppViewModel(application, appController) {

    val reportResultEvent = EventLiveData<Int>()

    var mosScore: MosScore? = null
        set(value) {
            field = value
            val data = arrayListOf<Map<String, String>>()
            value?.remote?.forEachIndexed { index, pair ->
                val map = hashMapOf<String, String>()
                map["serialNumber"] = (index + 1).toString()
                map["uid"] = pair.first.getConferenceNicknameMaybeAssistant()
                map["score"] = String.format("%.1f", pair.second / 100.0)
                data.add(map)
            }
            value?.local?.let {
                localScoreEvent.postValue(String.format("%.1f", it.second / 100.0))
            }
            if (data.isNotEmpty()) {
                userListEvent.postValue(data)
            }
        }

    val localScoreEvent = EventLiveData<String>()
    val userListEvent = EventLiveData<List<Map<String, String>>>()

    fun submit(
        feedback: Int?,
        selectedIndex: Int?,
        description: String?,
    ) {
        if (mosScore?.remote?.isEmpty() == true) {
            Logger.e(TAG, "Submit an error, there is no remote user")
            statusEvent.postValue(StatusEvent.Status.ERROR)
            return
        }
        Logger.i(
            TAG,
            "audio scoring submit, feedback=$feedback, selectedIndex=$selectedIndex, description=$description"
        )

        callRatingManager.scoringAudio(
            AudioScore(
                userList = mosScore!!.remote,
                description = description,
                rid = mosScore!!.rid,
                localScore = mosScore!!.local.second,
                selectedUid = selectedIndex?.let { mosScore?.remote?.get(selectedIndex)?.first?.streamId },
                localUid = mosScore!!.local.first.streamId,
                rating = 0,
                tags = feedback?.let { mutableListOf(feedback) }
            )
        )
    }

    override fun onEventReceived(arg: MessageEvent) {
        super.onEventReceived(arg)
        if (arg.type == AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT.ordinal) {
            val data = arg.obj
            if (data is Int) {
                reportResultEvent.postValue(data)
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT,
        )
    }

    companion object {
        private const val TAG = "[VM][AudioScoring]"
    }
}