package io.agora.avc.app.rating.call

import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import androidx.core.view.isVisible
import androidx.navigation.fragment.findNavController
import com.google.android.material.bottomsheet.BottomSheetBehavior
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.databinding.AvcFragmentCallRatingBinding
import io.agora.avc.utils.KeyboardUtils
import io.agora.avc.utils.ToastUtils
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.frame.base.livedata.StatusEvent


class CallRatingFragment :
    BaseSheetDialogFragment<CallRatingViewModel, AvcFragmentCallRatingBinding>() {

    private var chipList: List<Pair<Int, Int>>? = null

    private val mainViewModel: MainViewModel by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private val commentTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            s?.apply {
                if (length > MAX_COMMENT_WORDS) {
                    mBinding.tilFeedback.error =
                        getString(R.string.avc_bug_report_comment_words_overflow)
                } else {
                    mBinding.tilFeedback.error = null
                }
            }
            refreshBtnSubmitEnable(s?.length ?: 0, mBinding.ratingBar.rating)
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {

        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        }
    }

    override fun getLayoutId() = R.layout.avc_fragment_call_rating

    override fun onStart() {
        super.onStart()
        mDialog.setCanceledOnTouchOutside(false)
        mDialog.setCancelable(true)
        mBehavior.isHideable = true
        mBehavior.peekHeight
        mBehavior.skipCollapsed = true
        mBehavior.state = BottomSheetBehavior.STATE_EXPANDED
    }

    override fun initialize(savedInstanceState: Bundle?) {
        arguments?.getString("data")?.let {
            mViewModel.setMosScore(it)
        }

        mBinding.btnSubmit.setOnClickListener {
            submit()
        }

        mBinding.ratingBar.setOnRatingBarChangeListener { ratingBar, rating, fromUser ->
            refreshChipGroup(rating)
            refreshBtnSubmitEnable(mBinding.edtDescription.text?.length ?: 0, rating)
        }

        mBinding.clContent.setOnClickListener {
            KeyboardUtils.hideSoftInput(it)
        }
        mBinding.edtDescription.addTextChangedListener(commentTextWatcher)
        mBinding.btnSubmit.isEnabled = false
    }

    private fun refreshBtnSubmitEnable(length: Int, rating: Float) {
        mBinding.btnSubmit.isEnabled = length <= MAX_COMMENT_WORDS && rating > 0
    }

    private fun submit() {
        val description = mBinding.edtDescription.text.toString()
        if (description.length > 100) {
            return
        }

        val tag = if (mBinding.ratingBar.rating > 0 && mBinding.ratingBar.rating <= TAG_RATING) {
            mBinding.chips.getCheckedTags()
        } else {
            arrayListOf()
        }
        mViewModel.submit(
            mBinding.ratingBar.rating,
            tag,
            description = description,
        )
        findNavController().navigateUp()
    }

    private fun refreshChipGroup(rating: Float) {
        when (rating.toInt()) {
            1,
            2,
            3 -> {
                mBinding.chips.isVisible = true
            }
            else -> {
                mBinding.chips.isVisible = false
            }
        }

    }

    private fun addChips() {
        if (mBinding.chips.childCount > 0) {
            return
        }
        chipList?.let {
            mBinding.chips.addItems(it)
        }
    }

    override fun allocObserver() {
        mViewModel.statusEvent.observe(this, StatusEvent.StatusObserver {
            if (it == StatusEvent.Status.ERROR) {
                ToastUtils.showShort(R.string.avc_request_failed)
            }
        })
        mViewModel.chipDataEvent.observe(this) {
            this.chipList = it
            addChips()
            refreshChipGroup(mBinding.ratingBar.rating)
        }
        mainViewModel.callRatingRetryEvent.observe(this) {
            mBinding.btnSubmit.callOnClick()
        }
    }

    override fun onDestroyView() {
        mBinding.edtDescription.removeTextChangedListener(commentTextWatcher)
        super.onDestroyView()
    }

    companion object {
        private const val TAG = "[UI][CallRating]"
        private const val MAX_COMMENT_WORDS = 100
        private const val TAG_RATING = 3
    }
}