package io.agora.avc.app.setting

import android.os.Bundle
import android.view.View
import androidx.appcompat.app.AlertDialog
import androidx.core.view.isVisible
import androidx.lifecycle.Observer
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.operation.OperationFragment
import io.agora.avc.bo.AssistantInfo
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.Room
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.AvcFragmentRoomSettingsBinding
import io.agora.avc.extensions.*
import io.agora.avc.listener.DeveloperClickedListener
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.StringUtils
import io.agora.avc.widget.LoadingStatusTextButton
import io.agora.avc.widget.LoadingStatusTextButton.Companion.NO_TEXT
import io.agora.avc.widget.ResolutionSetting
import io.agora.frame.base.NovelFragment
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Logger

/**
 * room operate rtc request must after rtm request
 */
class RoomSettingsFragment : NovelFragment<RoomSettingsViewModel, AvcFragmentRoomSettingsBinding>() {

    private var roomInfo: Room? = null

    private var localUser: LocalUser? = null

    private var assistantInfo: AssistantInfo? = null

    private var bizConnected: Boolean = false

    private var showDeveloper: Boolean = false

    private var unifiedDialog: AlertDialog? = null

    private val developerClickedListener = object : DeveloperClickedListener(false) {
        override fun onDeveloperClicked(view: View?) {
            mViewModel?.saveDeveloperVisible(true)
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_room_settings
    }

    override fun allocObserver() {
        mViewModel?.bizConnectChangedLiveData?.observe(this, Observer {
            this.bizConnected = it
            renderUI()
        })

        mViewModel?.roomInfoChangedLiveData?.observe(this, Observer { room ->
//            if (this.roomInfo == null) {
//                initAssistantAction(room)
//            }
            this.roomInfo = room
            renderUI()
            assistantInfo?.let {
                setupAssistantAction(it.user)
            }
        })

        mViewModel?.localUserLiveData?.observe(this) { user ->
            when (this.localUser) {
                null -> {
                    this.localUser = user
                    renderUI()
                }
                else -> {
                    this.localUser = user
                }
            }
        }

        mViewModel?.uploadLogLiveData?.observe(this, Observer {
            if (it) {
                mBinding.btnUploadLog.state = LoadingStatusTextButton.State.SUCCESS
            } else {
                mBinding.btnUploadLog.state = LoadingStatusTextButton.State.FAIL
            }
        })

        mViewModel?.statusEvent?.observe(this, Observer {
            when (it) {
                StatusEvent.Status.LOADING -> {
                    mBinding.btnUploadLog.state = LoadingStatusTextButton.State.LOADING
                }
            }
        })

        mViewModel?.resolutionLiveData?.observe(this) { resolution ->
            resolution?.let {
                mBinding.resolution.setResolution(it)
            }
        }

        mViewModel?.developerOptionChangedLiveData?.observe(this) {
            this.showDeveloper = it.showDeveloper
            refreshDeveloperMenu(showDeveloper)
        }

        mViewModel?.assistantChangedLiveData?.observe(this) {
            Logger.i(TAG, "assistant changed, streamId:${it?.user?.streamId}, type:${it?.type}")
            assistantInfo = it
            setupAssistantAction(it?.user)
        }

        mViewModel?.assistantStatusLiveData?.observe(this) {
            Logger.i(TAG, "assistant button status change, it:${it}")
            setupAssistantButtonStatus(it)
        }
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.itemCamera.callback = {
            mViewModel?.changeRoomVideo(it)
        }

        mBinding.itemMicrophone.callback = {
            mViewModel?.changeRoomAudio(it)
        }

        mBinding.roomControl.callback = {
            //todo replace string
            showConfirmDialog(
                message = if (roomInfo?.audioMuted == true)
                    "所有房间内以及新加入的成员会解除静音"
                else
                    StringUtils.getString(R.string.avc_mute_all_alert)
            ) {
                mViewModel?.setRoomAudioMuteAll(!(roomInfo?.audioMuted?:false))
            }
        }

        mBinding.roomHost.callback = {
            if (mBinding.roomHost.getActionButtonText()
                == getString(R.string.avc_apply_host_button)
            ) {
                mViewModel?.changeRoomHost(true)
            } else {
                mViewModel?.changeRoomHost(false)
            }
        }

        mBinding.resolution.setOnResolutionChangedListener(object :
            ResolutionSetting.OnResolutionChangedListener {
            override fun onResolutionChanged(position: Int) {
                Logger.i(TAG, "I chose the resolution:$position")
                mViewModel?.changeVideoDimensions(position)
            }
        })

        mBinding.uploadLog.callback = {
            Logger.i(TAG, "I clicked the upload log button, state:${mBinding.btnUploadLog.state}")
            if (mBinding.btnUploadLog.canUpload()) {
                mViewModel?.uploadLog(DeviceUtils.getPlatform(), DeviceUtils.getModel())
            }
        }

        mBinding.roomAssistant.setOnClickListener {
            if (!mBinding.roomAssistant.getActionButton().isLoading()) {
                clickAssistant()
            }
        }

        setupAssistantAction(null)

        /******** Developer Menu *******/
//        mBinding.actionBar.setOnClickListener(developerClickedListener)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        activity?.setNavBarVisibility(true)
    }

    private fun renderUI() {
        mBinding.roomHost.setSecondaryText(
            roomInfo?.getConferenceNickname() ?: getString(R.string.avc_no_host)
        )
        mBinding.itemCamera.setSwitchChecked(roomInfo?.videoState == true)
        mBinding.itemMicrophone.setSwitchChecked(roomInfo?.audioState == true)
        setRoomActionEnable(roomInfo, localUser)
        if (roomInfo?.hostUid?.isNotEmpty() == true && roomInfo?.hostUid == localUser?.uid) {
            mBinding.roomHost.setActionButtonText(getString(R.string.avc_cancel_host_button))
        } else {
            mBinding.roomHost.setActionButtonText(getString(R.string.avc_apply_host_button))
        }
        //room control
        mBinding.roomControl.setActionButtonText(
            if (roomInfo?.audioMuted == true) {
                StringUtils.getString(R.string.avc_mute_all_unmute_button)
            } else {
                StringUtils.getString(R.string.avc_mute_all_button)
            }
        )
    }

    private fun setRoomActionEnable(room: Room?, user: LocalUser?) {
        if (user?.hasControlPermission(room?.hostUid) == true) {
            mBinding.itemCamera.setActionEnabled(true)
            mBinding.itemMicrophone.setActionEnabled(true)
            if (user?.hasPrivatePermission(room)) {
                mBinding.roomHost.setActionEnabled(true)
            } else {
                mBinding.roomHost.setActionEnabled(false)
            }
        } else {
            mBinding.itemCamera.setActionEnabled(false)
            mBinding.itemMicrophone.setActionEnabled(false)
            mBinding.roomHost.setActionEnabled(false)
        }
        mBinding.roomControl.isVisible = user?.isHost(room?.hostUid) == true
    }

    private fun refreshDeveloperMenu(show: Boolean) {
        if (show) {
            mBinding.tvDebugModelTitle.visibility = View.VISIBLE
            mBinding.mtlCardDebugModel.visibility = View.VISIBLE
            mBinding.developerMenu.callback = {
                safeNavigate(R.id.action_roomSettings_to_developer)
            }
        }
    }

    @Deprecated(message = "")
    private fun initAssistantAction(room: Room?) {
        if (room?.isInternal() == true) {
            mBinding.roomAssistant.isVisible = true
            if (mBinding.roomAssistant.getActionButton().onStatusChangedListener == null) {
                mBinding.roomAssistant.getActionButton().onStatusChangedListener = { _, _ ->
                    if (mBinding.roomAssistant.getActionButton()
                            .getText() == getString(R.string.avc_as_setting_apply_fail)
                    ) {
                        mBinding.roomAssistant.getActionButton().resetCountdown()
                    }
                }
            }
        }
    }

    @Deprecated(message = "")
    private fun setupAssistantAction(assistant: ARoomUser?) {
        mBinding.roomAssistant.setSecondaryText(assistant?.getConferenceNickname() ?: "")
        if (assistant == null) {
            Logger.i(TAG, "setup assistant action, not assistant")
            mBinding.roomAssistant.getActionButton().successText =
                getString(R.string.avc_as_setting_cancel)
            mBinding.roomAssistant.getActionButton().failText =
                getString(R.string.avc_as_setting_apply_fail)
            mBinding.roomAssistant.getActionButton().normalText =
                getString(R.string.avc_as_setting_apply)
            mBinding.roomAssistant.getActionButton().isVisible = true
            mBinding.roomAssistant.isClickable = true
            mBinding.roomAssistant.getActionButton().state = LoadingStatusTextButton.State.NORMAL
        } else {
            when {
                assistant.isMySelf() -> {
                    Logger.i(TAG, "setup assistant action, assistant is myself")
                    mBinding.roomAssistant.getActionButton().successText =
                        getString(R.string.avc_as_setting_cancel)
                    mBinding.roomAssistant.getActionButton().failText =
                        getString(R.string.avc_as_setting_apply_fail)
                    mBinding.roomAssistant.getActionButton().isVisible = true
                    mBinding.roomAssistant.isClickable = true
                }
                localUser?.isHost == true -> {
                    Logger.i(TAG, "setup assistant action, I'm host but not assistant")
                    mBinding.roomAssistant.getActionButton().successText =
                        getString(R.string.avc_as_setting_cancel_host)
                    mBinding.roomAssistant.getActionButton().failText = NO_TEXT
                    mBinding.roomAssistant.getActionButton().isVisible = true
                    mBinding.roomAssistant.isClickable = true
                }
                else -> {
                    Logger.i(TAG, "setup assistant action, assistant is others")
                    mBinding.roomAssistant.getActionButton().isVisible = false
                    mBinding.roomAssistant.isClickable = false
                }
            }
            mBinding.roomAssistant.getActionButton().state = LoadingStatusTextButton.State.SUCCESS
        }
    }

    @Deprecated(message = "")
    private fun setupAssistantButtonStatus(@LoadingStatusTextButton.State it: Int) {
        mBinding.roomAssistant.getActionButton().state = it
    }

    @Deprecated(message = "")
    private fun clickAssistant() {
        mBinding.roomAssistant.getActionButton().apply {
            getText()?.let { _text ->
                when (_text) {
                    getString(R.string.avc_as_setting_apply) -> {
                        localUser?.let {
                            OperationFragment.navigateTo(
                                this@RoomSettingsFragment,
                                it,
                                OperationFragment.OperationType.CLAIM_ASSISTANT
                            )
                        }
                    }
                    getString(R.string.avc_as_setting_cancel),
                    getString(R.string.avc_as_setting_cancel_host) -> {
                        mViewModel.cancelAssistant()
                    }
                }
            }
        }
    }

    private fun showConfirmDialog(message: String, action: () -> Unit) {
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setCancelable(false)
                    .setMessage(message)
                    .setNegativeButton(R.string.avc_no_button) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_yes_button) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    companion object {
        private const val TAG = "[UI][RoomSetting]"
    }
}
