package io.agora.avc.auth.signin.wework

import android.content.Context
import com.google.gson.JsonObject
import com.tencent.wework.api.IWWAPI
import com.tencent.wework.api.IWWAPIEventHandler
import com.tencent.wework.api.WWAPIFactory
import com.tencent.wework.api.model.WWAuthMessage
import io.agora.avc.BuildConfig
import io.agora.avc.R
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.push.PushManager
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.repository.UserRepository
import io.agora.avc.utils.ToastUtils
import io.agora.avc.widget.COMPANY_EASEMOB
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import javax.inject.Inject

const val AGORA_APP_ID = "ww1792ac25c99153bd"
const val EASEMOB_APP_ID = "ww6fef22f69c104461"
const val AGENT_ID_AGORA_RELEASE = "1000017"
const val AGENT_ID_AGORA_DEBUG = "1000024"
const val AGENT_ID_EASEMOB_RELEASE = "1000014"
const val AGENT_ID_EASEMOB_DEBUG = "1000015"
const val SCHEMA_AGORA_RELEASE = "wwauth1792ac25c99153bd000017"
const val SCHEMA_AGORA_DEBUG = "wwauth1792ac25c99153bd000024"
const val SCHEMA_EASEMOB_RELEASE = "wwauth6fef22f69c104461000014"
const val SCHEMA_EASEMOB_AGORA_DEBUG = "wwauth6fef22f69c104461000015"

class WeWorkManagerImpl @Inject constructor(
    private val dataRepository: IDataRepository,
    private val userRepository: UserRepository,
    private val developerRepository: DeveloperRepository,
    private val appConfigRepository: AppConfigRepository,
    private val appEventBus: AppEventBus,
    private val noticeManager: NoticeManager,
    private val roomRepository: RoomRepository,
    private val appController: AppController,
    private val pushManager: PushManager,
) : WeWorkManager {

    private var wwapi: IWWAPI? = null

    override fun initWWAPI(context: Context) {
        Logger.i(TAG, "Init wework libs")
        wwapi = WWAPIFactory.createWWAPI(context.applicationContext)
    }

    /**
     * login via WeWork
     * request weWork sdk -> call server
     */
    override fun loginViaWeWork(type: Int) {
        if (appConfigRepository.isTestServer() != BuildConfig.DEBUG) {
            ToastUtils.showShort(R.string.avc_wework_env_mismatch)
            return
        }
        val source = if (type == COMPANY_EASEMOB) {
            UserSource.EASEMOB.value
        } else {
            UserSource.AGORAN.value
        }

        requestLogin(source = source, type = type)
    }

    override fun requestLogin(source: Int, platform: String, deviceName: String, type: Int) {
        Logger.i(TAG, "Initiate weWork login")
        wwapi?.apply {
            registerApp(getSchema(type))
            val req = WWAuthMessage.Req()
            req.sch = getSchema(type)
            req.appId = getAppId(type)
            req.agentId = getAgentId(type)
            req.state = STATE

            val iwWApiEventHandler = IWWAPIEventHandler { resp ->
                Logger.i(TAG, "Received response from authentication")
                if (resp is WWAuthMessage.Resp) {
                    when (resp.errCode) {
                        WWAuthMessage.ERR_CANCEL -> {
                            Logger.i(TAG, "WeWork login cancelled")
                        }
                        WWAuthMessage.ERR_FAIL -> {
                            Logger.e(TAG, "WeWork login failed")
                            noticeManager.notice(NoticeEvent(NoticeCode.CODE_SETTING_ACCOUNT_IN_FAILED))
                        }
                        WWAuthMessage.ERR_OK -> {
                            Logger.i(TAG, "WeWork login Successfully,code:${resp.code}")
                            queryInnerSession(
                                resp.code,
                                source = source,
                                platform = platform,
                                deviceName = deviceName
                            )
                        }
                        else -> {
                            Logger.e(
                                TAG,
                                "Received an unknown response from authentication, error code: ${resp.errCode}"
                            )
                        }
                    }
                }
            }
            val status = sendMessage(req, iwWApiEventHandler)
            Logger.i(TAG, "send login by WeWork,state:$status")
        }
    }

    override fun logoutViaWeWork() {
        userRepository.getLocalUser()
            .map {
                developerRepository.loginWeWork(false)
                it.token ?: ""
            }
            .flatMap {
                delInnerSession()
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(t: BaseNo?) {
                    Logger.i(TAG, "Successfully to logout via WeWork")
                    userRepository.removeInnerInfo()
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }

                override fun onFail(t: BaseNo?) {
                    Logger.i(TAG, "Failed to logout via WeWork")
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }
            })
    }

    private fun weWorkAutoLogin(
        name: String,
        innerName: String,
        alias: String,
        department: String,
        innerSession: String,
        uid: String
    ): Observable<BaseNo> {
        val data = JsonObject()
        data.addProperty("name", name)
        data.addProperty("innerName", innerName)
        data.addProperty("alias", alias)
        data.addProperty("department", department)
        data.addProperty("wxUserId", innerSession)
        data.addProperty("uid", uid)
        return dataRepository.getRetrofitService(ApiService::class.java)
            .weWorkAutoLogin(data)
    }

    /**
     * verify session
     */
    override fun verifySession(session: String?, source: Int) {
        Logger.i(TAG, "verify session:$session, source:$source")
        if (source == UserSource.UNKNOWN.value) {
            handleTokenExpired()
            return
        }
        dataRepository.getRetrofitService(ApiService::class.java)
            .verifySession()
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<InnerSession>() {
                override fun onSuccess(resp: InnerSession?) {
                    if (resp == null) {
                        Logger.e(TAG, "failed to verify session, session is empty")
                        return
                    }
                    Logger.i(TAG, "Successfully to synchronize session:$resp")
                    pushManager.getToken()
                    if (resp.name != null) {
                        val info =
                            InnerInfo(resp.alias, resp.department, resp.name, session, source)
                        if (!developerRepository.getDeveloperOptions().weWorkLogin) {
                            userRepository.saveInnerInfo(info)
                        }
                    }
                }

                override fun onFail(resp: InnerSession?) {
                    Logger.e(TAG, "Failed to synchronize session,session invalid")
                    if (resp?.code != RESIGNED_USER &&
                        resp?.code != SESSION_INVALID &&
                        resp?.code != TOKEN_EXPIRED
                    ) {
                        return
                    }
                    handleTokenExpired()
                }
            })
    }

    private fun handleTokenExpired() {
        Logger.i(TAG, "handle token expired")
        userRepository.removeInnerInfo()
        appEventBus.notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal, true))
        if (roomRepository.getRoom() == null) {
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED))
        } else {
            appController.leaveRoom()
            noticeManager.notice(
                NoticeEvent(
                    NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED,
                    NoticeType.DIALOG
                )
            )
        }
    }

    override fun deInitWWAPI() {
        Logger.i(TAG, "deInit wework libs")
        wwapi?.unregisterApp()
    }

    override fun isWWAppInstalled(): Boolean {
        if (wwapi == null) {
            Logger.w(TAG, "Wework has not been initialized")
            return false
        }
        val ret = wwapi?.isWWAppInstalled ?: false
        Logger.i(TAG, "Detect whether wework is installed:$ret")
        return ret
    }

    /**
     * developer mode login weWork
     */
    override fun loginWeWorkForDeveloper(login: Boolean) {
        Logger.i(TAG, "Log in to WeWork automatically,login:$login")
        val userToken = userRepository.getUserInfo()
        if (userToken?.uid == null) {
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_SETTING_ACCOUNT_IN_FAILED))
            return
        }

        if (!login) {
            userRepository.saveUserInfo(userToken)
            appEventBus.notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal, true))
            return
        }

        val mockUser = mockWeWorkTester(userToken)
        weWorkAutoLogin(
            mockUser.name,
            mockUser.innerInfo?.name ?: "",
            mockUser.innerInfo?.alias ?: "",
            mockUser.innerInfo?.department ?: "",
            mockUser.innerInfo?.innerSession ?: "",
            mockUser.uid
        ).subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(t: BaseNo?) {
                    mockUser.innerInfo?.let {
                        userRepository.updateInnerInfo(it)
                    }
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }

                override fun onFail(t: BaseNo?) {
                    ToastUtils.showShort(R.string.avc_e1008)
                }
            })
    }

    override fun queryInnerSession(
        code: String,
        source: Int,
        platform: String,
        deviceName: String
    ) {
        val data = JsonObject()
        data.addProperty("code", code)
        data.addProperty("source", source)
        data.addProperty("platform", platform)
        data.addProperty("deviceName", deviceName)

        dataRepository.getRetrofitService(ApiService::class.java)
            .queryInnerSession(data)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<InnerSession>() {
                override fun onSuccess(t: InnerSession?) {
                    Logger.i(TAG, "Successfully to query inner session:$t")
                    if (t != null && t.success) {
                        val info = InnerInfo(t.alias, t.department, t.name, t.innerSession, source)
                        userRepository.saveInnerInfo(info)
                        pushManager.getToken()
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.LOCAL_USER_CHANGED.ordinal,
                                true
                            )
                        )
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.WEWORK_LOGIN.ordinal,
                                true
                            )
                        )
                    }
                }

                override fun onFail(t: InnerSession?) {
                    Logger.e(TAG, "Failed to query inner session")
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }
            })
    }

    override fun delInnerSession(): Observable<BaseNo> {
        return dataRepository.getRetrofitService(ApiService::class.java)
            .delInnerSession()
    }

    private fun mockWeWorkTester(userInfo: UserInfo): UserInfo {
        val innerInfo = InnerInfo(
            "Zhoujunjie",
            "后端业务测试（QA of Cloud Services）",
            "周俊捷",
            TEST_INNER_SESSION,
            UserSource.AGORAN.value
        )
        userInfo.innerInfo = innerInfo
        return userInfo
    }

    companion object {
        private const val TAG = "[COMM][WeWork]"
        private const val STATE = "io.agora"
        private const val TEST_INNER_SESSION = "zhoujunjie8845"

        private const val RESIGNED_USER = 1018
        private const val SESSION_INVALID = 1023
        private const val TOKEN_EXPIRED = 11003


        fun getAgentId(type: Int): String {
            return if (type == COMPANY_EASEMOB) {
                if (SPLiteProxy.isTestServer()) AGENT_ID_EASEMOB_DEBUG else AGENT_ID_EASEMOB_RELEASE
            } else {
                if (SPLiteProxy.isTestServer()) AGENT_ID_AGORA_DEBUG else AGENT_ID_AGORA_RELEASE
            }
        }

        fun getSchema(type: Int): String {
            return if (type == COMPANY_EASEMOB) {
                if (SPLiteProxy.isTestServer()) SCHEMA_EASEMOB_AGORA_DEBUG else SCHEMA_EASEMOB_RELEASE
            } else {
                if (SPLiteProxy.isTestServer()) SCHEMA_AGORA_DEBUG else SCHEMA_AGORA_RELEASE
            }
        }

        fun getAppId(type: Int): String {
            return if (type == COMPANY_EASEMOB) {
                EASEMOB_APP_ID
            } else {
                AGORA_APP_ID
            }
        }
    }

}