package io.agora.avc.biz

import io.agora.avc.bo.RoomMode
import io.agora.avc.utils.RegexUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger

class RoomChecker {
    val errorLiveData: EventLiveData<Error> = EventLiveData()

    fun checkRoom(rid: String, pwd: String?, mode: Int): Boolean {
        return checkName(rid, mode) && checkPwd(pwd)
    }

    fun isAgoraRoom(rid: String): Boolean {
        return rid.startsWith(PREFIX_AGORA_MEETING)
    }

    private fun checkName(rid: String, mode: Int): Boolean {
        var roomName = rid
        if (mode == RoomMode.AGORA.value && !rid.startsWith(PREFIX_AGORA_MEETING)) {
            Logger.e(TAG, "room mode:$mode name:$rid error,invalid name")
            errorLiveData.postValue(Error.RID_INVALID)
            return false
        }
        if (mode == RoomMode.AGORA.value) {
            roomName = rid.removePrefix(PREFIX_AGORA_MEETING)
        }
        if (!RegexUtils.isMatch("^[A-Z_0-9]*$", roomName)) {
            Logger.e(TAG, "room name:$roomName error,invalid name")
            errorLiveData.postValue(Error.RID_INVALID)
            return false
        }
        if (roomName.length < MIN_ROOM_NAME_LENGTH) {
            Logger.e(TAG, "room name:$roomName error,length too short")
            errorLiveData.postValue(Error.RID_TOO_SHORT)
            return false
        }
        if (roomName.length > MAX_ROOM_NAME_LENGTH) {
            Logger.e(TAG, "room name:$roomName error,length too long")
            errorLiveData.postValue(Error.RID_TOO_LONG)
            return false
        }
        return true
    }

    private fun checkPwd(pwd: String?): Boolean {
        if (pwd == null) return true
        if (pwd.length > MAX_ROOM_PWD_LENGTH) {
            Logger.e(TAG, "room pwd:$pwd error,length too long")
            errorLiveData.postValue(Error.PWD_INVALID)
            return false
        }
        val pwdChars = pwd.toCharArray()
        for (element in pwdChars) {
            if (element !in PWD_FILTER) {
                Logger.e(TAG, "room pwd:$pwd error,invalid pwd")
                errorLiveData.postValue(Error.PWD_TOO_LONG)
                return false
            }
        }
        return true
    }

    companion object {

        private const val TAG = "[Comm][RoomChecker]"

        const val PREFIX_AGORA_MEETING = ""

        const val PWD_FILTER =
            "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!#\$%&()+-:;<=.>?@[]^_{}|~,"

        const val MIN_ROOM_NAME_LENGTH = 3

        const val MAX_ROOM_NAME_LENGTH = 18

        const val MAX_ROOM_PWD_LENGTH = 18

        const val MIN_NICK_NAME_LENGTH = 1

    }

    enum class Error(val value: Int) {
        RID_INVALID(1),
        RID_TOO_SHORT(2),
        RID_TOO_LONG(3),
        PWD_INVALID(4),
        PWD_TOO_LONG(5);
    }
}