package io.agora.avc.di.module

import android.app.Application
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import dagger.Module
import dagger.Provides
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.manager.media.MediaPlayer
import io.agora.avc.manager.upgrade.AppUpgradeManager
import io.agora.avc.push.PushDataRepository
import io.agora.avc.push.PushManager
import io.agora.avc.push.fcm.FcmPushManagerImpl
import io.agora.avc.push.hms.HmsPushManagerImpl
import io.agora.avc.push.rtm.RtmPushManagerImpl
import io.agora.avc.push.xiaomi.MiPushManagerImpl
import io.agora.frame.di.scope.ApplicationScope
import io.agora.logger.Logger

@Module
class PushModule {

    @ApplicationScope
    @Provides
    fun providePushManager(
        application: Application,
        dataRepository: PushDataRepository,
        appEventBus: AppEventBus,
        mediaPlayer: MediaPlayer,
        appUpgradeManager: AppUpgradeManager
    ): PushManager {
        return when {
            supportHms(application.applicationContext) -> {
                Logger.i(TAG, "devices support hms")
                HmsPushManagerImpl(dataRepository, appEventBus, mediaPlayer, appUpgradeManager)
            }
            supportFcm(application.applicationContext) -> {
                Logger.i(TAG, "devices support fcm")
                FcmPushManagerImpl(dataRepository, appEventBus, mediaPlayer, appUpgradeManager)
            }
            supportXiaomi(application.applicationContext) -> {
                Logger.i(TAG, "devices support mi push")
                MiPushManagerImpl(
                    application,
                    dataRepository,
                    appEventBus,
                    mediaPlayer,
                    appUpgradeManager
                )
            }
            else -> {
                Logger.i(TAG, "devices support rtm")
                RtmPushManagerImpl(dataRepository, appEventBus, mediaPlayer, appUpgradeManager)
            }
        }
    }

    private fun supportXiaomi(context: Context): Boolean {
        val manufacturer = Build.MANUFACTURER
        if ("xiaomi".equals(manufacturer, true)) {
            return true
        }
        return false
    }

    private fun supportHms(context: Context): Boolean {
        return hasHmsCore(context) && isEMUISupport(9)
    }

    private fun hasHmsCore(context: Context): Boolean {
        val pm: PackageManager = context.packageManager
        var hwid = 0L
        try {
            val pi = pm.getPackageInfo("com.huawei.hwid", 0)
            if (pi != null) {
                hwid = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                    pi.longVersionCode
                } else {
                    pi.versionCode.toLong()
                }
                return hwid != 0L
            }
        } catch (e: PackageManager.NameNotFoundException) {

        }
        return false
    }

    private fun isEMUISupport(version: Int): Boolean {
        var emuiApiLevel = 0
        try {
            val cls = Class.forName("android.os.SystemProperties")
            val method = cls.getDeclaredMethod(
                "get", *arrayOf<Class<*>>(
                    String::class.java
                )
            )
            emuiApiLevel = (method.invoke(
                cls,
                *arrayOf<Any>("ro.build.hw_emui_api_level")
            ) as String)?.toInt()
            return emuiApiLevel >= version
        } catch (e: Exception) {

        }
        return false
    }

    private fun supportFcm(context: Context): Boolean {
//        val code = GoogleApiAvailabilityLight.getInstance()
//            .isGooglePlayServicesAvailable(context.applicationContext)
//        return code == ConnectionResult.SUCCESS
        return false
    }

    companion object {
        private const val TAG = "PushManager"
    }
}