package io.agora.avc.manager.audiodump

import com.agora.valoran.Constants
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.Attachment
import io.agora.avc.bo.AudioDumpStatus
import io.agora.avc.manager.bugreport.DumpStatus
import io.agora.avc.repository.ProblemRepository
import io.agora.avc.utils.FileUtils2
import io.agora.avc.utils.TimeUtils2
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.io.File
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean
import javax.inject.Inject

class AudioDumperImpl @Inject constructor(
    private val problemRepository: ProblemRepository,
    private val appEventBus: AppEventBus,
    private val appController: AppController,
) : AudioDumper {

    private var dumpDisposable: Disposable? = null
    private var currentDump: Attachment? = null
    private val isDumping: AtomicBoolean = AtomicBoolean(false)

    override fun startDumping() {
        if (isDumping.compareAndSet(false, true)) {
            currentDump = createAttachment()
            if (currentDump == null) {
                Logger.e(TAG, "failed to create attachment, current attachment is empty")
                return
            }
            if (dumpDisposable?.isDisposed == false) {
                dumpDisposable?.dispose()
            }
            Observable
                .interval(0, TIME_BASE, TimeUnit.MILLISECONDS)
                .take(TIME_COUNT + 1)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(object : io.reactivex.Observer<Long> {
                    override fun onComplete() {
                        Logger.i(TAG, "Complete audio dumping,attachment:$currentDump")
                        isDumping.set(false)
                        val issue = problemRepository.getIssue()
                        issue?.dumpStatus = DumpStatus.FINISHED
                        currentDump?.apply {
                            dumpingStatus = AudioDumpStatus.COMPLETED.value
                            opts.duration = TIME_COUNT.times(TIME_BASE)
                            appController.stopIssueDumping(Constants.ISSUE_DUMP_TYPE_AUDIO)
                            problemRepository.getIssue()?.addAttachment(this)
                        }
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal,
                                issue
                            )
                        )
                    }

                    override fun onSubscribe(d: Disposable) {
                        Logger.i(TAG, "Start audio dumping,attachment:$currentDump")
                        val issue = problemRepository.getIssue()
                        issue?.dumpStatus = DumpStatus.DUMPING
                        dumpDisposable = d
                        currentDump?.apply {
                            dumpingStatus = AudioDumpStatus.STARTING.value
                            this.path?.let { path ->
                                appController.startIssueDumping(
                                    Constants.ISSUE_DUMP_TYPE_AUDIO,
                                    path
                                )
                            }
                        }
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal,
                                issue
                            )
                        )
                    }

                    override fun onNext(t: Long) {
                        val times = t.times(TIME_BASE)
                        currentDump?.opts?.duration = times
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.AUDIO_DUMPING_TIME.ordinal,
                                (times / 1000f).toInt()
                            )
                        )
                    }

                    override fun onError(e: Throwable) {
                        Logger.e(TAG, "Audio dump is abnormal,attachment:$currentDump", e)
                        isDumping.set(false)
                        val issue = problemRepository.getIssue()
                        issue?.dumpStatus = DumpStatus.NONE
                        appController.stopIssueDumping(Constants.ISSUE_DUMP_TYPE_AUDIO)
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal,
                                issue
                            )
                        )
                    }
                })
        } else {
            Logger.e(TAG, "Audio is dumping, cannot start again")
        }
    }

    private fun addAttachment(attachment: Attachment) {
        Observable
            .create<Long> {
                problemRepository.getIssue()?.addAttachment(attachment)
                it.onNext(problemRepository.addAttach(attachment))
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<Long> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Long) {

                }

                override fun onError(e: Throwable) {

                }

                override fun onComplete() {

                }
            })
    }

    override fun stopDumping() {
        Logger.i(TAG, "Stop audio dumping,attachment:$currentDump")
        isDumping.set(false)
        val issue = problemRepository.getIssue()
        issue?.dumpStatus = DumpStatus.FINISHED
        dumpDisposable?.dispose()
        currentDump?.apply {
            dumpingStatus = AudioDumpStatus.STOPPED.value
            problemRepository.getIssue()?.addAttachment(this)
        }
        appController.stopIssueDumping(Constants.ISSUE_DUMP_TYPE_AUDIO)
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal,
                issue
            )
        )
    }

    override fun destroy() {
        Logger.i(TAG, "Destroy AudioDumper")
        if (dumpDisposable?.isDisposed == false) {
            appController.stopIssueDumping(Constants.ISSUE_DUMP_TYPE_AUDIO)
            dumpDisposable?.dispose()
        }
        isDumping.set(false)
        dumpDisposable = null
        currentDump = null
    }

    private fun createAttachment(): Attachment? {
        val fileName = "$FILE_NAME_RECORDING ${TimeUtils2.getNowString()}"
        val path =
            io.agora.avc.utils.FileUtils.getDirPath(DIR_NAME_RECORDING) + File.separator + fileName + File.separator
        if (!FileUtils2.createOrExistsDir(path)) {
            Logger.w(TAG, "Failed to create audio dump folder, please check the reason")
            return null
        }
        return Attachment(path = path)
    }

    companion object {
        private const val TAG = "[Comm][AudioDumper]"
        private const val DIR_NAME_RECORDING = "Recording"
        private const val FILE_NAME_RECORDING = "audio_dump"
        private const val TIME_BASE = 250L
        private const val TIME_COUNT = 240L
        private const val TIMING = 60
    }
}