package io.agora.avc.manager.bi

import android.app.Application
import io.agora.avc.BuildConfig
import io.agora.avc.MyApplication
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.bo.BIData
import io.agora.avc.bo.LocalUser
import io.agora.avc.utils.AppUtils
import io.agora.logger.Logger
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.NetworkUtils
import java.util.*
import javax.inject.Inject

class BIManagerImpl @Inject constructor(val application: Application) : BIManager {

    private var isTokenRecord = false
    private var isRtmTokenRecord = false
    private var isRtmLoginRecord = false
    private var isRtmJoinRecord = false

    private fun isRecordPermission(isSuccess: Boolean, bizType: String?, eventType: Int): Boolean {
        if ((!isSuccess && bizType == BIZ_TYPE_V1_TOKEN && !isTokenRecord) ||
            (!isSuccess && bizType == BIZ_TYPE_V1_RTM_TOKEN && !isRtmTokenRecord) ||
            (!isSuccess && eventType == EVENT_TYPE_RTM_LOGIN && !isRtmLoginRecord) ||
            (!isSuccess && eventType == EVENT_TYPE_RTM_JOIN_CHANNEL && !isRtmJoinRecord)
        ) {
            return true
        }
        return false
    }

    override fun biRecord(
        eventType: Int,
        bizType: String?,
        succeed: Boolean,
        responseCode: Int,
        elapse: Long
    ) {
        if (!isRecordPermission(succeed, bizType, eventType)) return
        if (bizType == BIZ_TYPE_V1_TOKEN && !succeed) {
            isTokenRecord = true
        } else if (bizType == BIZ_TYPE_V1_RTM_TOKEN && !succeed) {
            isRtmTokenRecord = true
        } else if (eventType == EVENT_TYPE_RTM_LOGIN && !succeed) {
            isRtmLoginRecord = true
        } else if (eventType == EVENT_TYPE_RTM_JOIN_CHANNEL && !succeed) {
            isRtmJoinRecord = true
        }

        if (application is MyApplication) {
            val successFlag = if (succeed) {
                1
            } else {
                0
            }
            val data = createBIData(
                eventType,
                bizType,
                responseCode,
                succeed,
                elapse,
                application.appContainer.localUser
            )
            if (!BuildConfig.DEBUG) {
                data.netType = NetworkUtils.getNetworkType().ordinal
            }
            val params = GsonUtils.toJson(data)

            if (BuildConfig.DEBUG) {
                Logger.i(TAG, params)
            } else {
                application.appContainer?.appController?.biRecord(
                    0,
                    successFlag,
                    responseCode,
                    elapse.toInt(),
                    params
                )
            }
        }
    }


    override fun biRecord(path: String, succeed: Boolean, responseCode: Int, elapse: Long) {
        val eventType = getEventType(path)
        val bizType = getBizType(path)
        biRecord(eventType, bizType, succeed, responseCode, elapse)
    }

    override fun biApp() {
        biRecord(EVENT_TYPE_DEVICE_INFO, null, true, 0, 0)
    }

    private fun createBIData(
        eventType: Int,
        bizType: String?,
        responseCode: Int,
        succeed: Boolean,
        elapse: Long,
        localUser: LocalUser?
    ): BIData {
        var sid: String = if (SPLiteProxy.getAvcGuid().isEmpty()) {
            val uuid = UUID.randomUUID().toString()
            SPLiteProxy.setAvcGuid(uuid)
            uuid
        } else {
            SPLiteProxy.getAvcGuid()
        }
        val avcUid = localUser?.uid ?: ""
        return BIData(
            eventType,
            bizType,
            succeed,
            responseCode,
            elapse.toInt(),
            sid,
            avcUid,
            BIData.Index(sid, avcUid),
            NetworkUtils.NetworkType.NETWORK_UNKNOWN.ordinal,
            DeviceUtils.getBrand() + " " + DeviceUtils.getModel(),
            1,
            DeviceUtils.getSDKVersionCode().toString(),
            AppUtils.getAppVersionName() + "." + AppUtils.getAppVersionCode(),
            System.currentTimeMillis(),
            null

        )
    }

    private fun getEventType(path: String): Int {
        return when (path) {
            "avcservice/v1/config" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/getToken" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "avcservice/v1/account/getRtmToken" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/update" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "avcservice/v1/rtmnode" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/portrait" -> {
                EVENT_TYPE_PORTRAIT_UPLOAD
            }
            "feedback/uploadFeedback" -> {
                EVENT_TYPE_LOG
            }
            "usrservice/v1/account/innerMeetingInfo" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/innerSession" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/linkId" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/verifySession" -> {
                EVENT_TYPE_BIZ_REQUEST
            }
            "usrservice/v1/account/generalLog"->{
                EVENT_TYPE_LOG
            }
            else -> {
                0
            }
        }
    }

    private fun getBizType(path: String): String? {
        return when (path) {
            "avcservice/v1/config" -> {
                BIZ_TYPE_V1_CONFIG
            }
            "usrservice/v1/account/getToken" -> {
                BIZ_TYPE_V1_TOKEN
            }
            "avcservice/v1/account/getRtmToken" -> {
                BIZ_TYPE_V1_RTM_TOKEN
            }
            "usrservice/v1/account/update" -> {
                BIZ_TYPE_ACCOUNT_UPDATE
            }
            "avcservice/v1/rtmnode" -> {
                BIZ_TYPE_V1_RTM_NODE
            }
            "usrservice/v1/account/portrait" -> {
                null
            }
            "feedback/uploadFeedback" -> {
                null
            }
            "usrservice/v1/account/innerMeetingInfo" -> {
                BIZ_TYPE_ACCOUNT_IMI
            }
            "usrservice/v1/account/innerSession" -> {
                BIZ_TYPE_ACCOUNT_INNER_SESSION
            }
            "usrservice/v1/account/linkId" -> {
                BIZ_TYPE_ACCOUNT_LINK_ID
            }
            "usrservice/v1/account/verifySession" -> {
                BIZ_TYPE_ACCOUNT_VERIFY_SESSION
            }
            "usrservice/v1/account/generalLog"->{
                null
            }
            else -> {
                null
            }
        }
    }


    companion object {
        private const val EVENT_TYPE_LOG = 1
        private const val EVENT_TYPE_PORTRAIT_UPLOAD = 2
        private const val EVENT_TYPE_PORTRAIT_DOWNLOAD = 3
        private const val EVENT_TYPE_BIZ_REQUEST = 4
        private const val EVENT_TYPE_RTM_LOGIN = 5
        private const val EVENT_TYPE_RTM_JOIN_CHANNEL = 6
        private const val EVENT_TYPE_RTM_PEER_MSG = 7
        private const val EVENT_TYPE_RTM_LINK_KEEP_TIME = 8
        private const val EVENT_TYPE_RTM_RECONNECT_COST_TIME = 9
        private const val EVENT_TYPE_JOIN_ROOM = 10
        private const val EVENT_TYPE_DEVICE_INFO = 11
        private const val TAG = "[Comm][BiRecord]"
        private const val BIZ_TYPE_V1_CONFIG = "v1/config"
        private const val BIZ_TYPE_V1_TOKEN = "account/getToken"
        private const val BIZ_TYPE_V1_RTM_TOKEN = "v1/account/getRtmToken"
        private const val BIZ_TYPE_ACCOUNT_UPDATE = "account/update"
        private const val BIZ_TYPE_V1_RTM_NODE = "v1/rtmnode"
        private const val BIZ_TYPE_ACCOUNT_IMI = "account/innerMeetingInfo"
        private const val BIZ_TYPE_ACCOUNT_INNER_SESSION = "account/innerSession"
        private const val BIZ_TYPE_ACCOUNT_LINK_ID = "account/linkId"
        private const val BIZ_TYPE_ACCOUNT_VERIFY_SESSION = "account/verifySession"
    }

}