package io.agora.avc.manager.bugreport

import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.Attachment
import io.agora.avc.bo.Issue
import io.agora.avc.bo.IssueNo
import io.agora.avc.manager.logupload.LogUploader
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.po.Problem
import io.agora.avc.repository.ProblemRepository
import io.agora.avc.utils.*
import io.agora.avc.widget.UploadingStatus
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.Observer
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody
import java.io.File
import javax.inject.Inject

class BugReporterImpl @Inject constructor(
    private val problemRepository: ProblemRepository,
    private val logUploader: LogUploader,
    private val appEventBus: AppEventBus,
    private val dataRepository: IDataRepository
) : BugReporter {

    private var reportIssueDisposable: Disposable? = null

    /**
     * clean Issue attachment & it self
     */
    override fun cleanAudioDumpFiles(problemId: Long) {
        Observable
            .create<Boolean> { emitter ->
                val problemWithAttach = problemRepository.getProblemWithAttach(problemId)
                val problem = problemWithAttach.problem
                //clean attachment
                problemWithAttach.attach?.forEach { attachment ->
                    if (attachment.path != null && attachment.path?.isNotEmpty() == true) {
                        FileUtils2.delete(attachment.path)
                    }
                }
                //clean zip file
                problem?.zipFilePath?.let { zipPath ->
                    if (zipPath.isNotEmpty()) {
                        FileUtils2.delete(zipPath)
                    }
                }
                emitter.onComplete()
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : Observer<Boolean> {
                override fun onComplete() {
                    Logger.i(TAG, "Successfully to delete issue attachment")
                }

                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Boolean) {

                }

                override fun onError(e: Throwable) {
                    Logger.e(TAG, "Failed to delete issue attachment", e)
                }
            })
    }

    /**
     * Report the problem to the server
     * , do file compression here
     * , and the generated file path will be saved in the database.
     * The reported status is saved in the database
     * , and subscribers are also notified through events.
     * After receiving the event, the subscriber can search for data
     * in the database according to the problemId.
     */
    override fun reportIssue(problemId: Long) {
        val problemWithAttach = problemRepository.getProblemWithAttach(problemId)
        val problem = problemWithAttach.problem
        Observable
            .create<Problem> { emitter ->
                if (StringUtils.isEmpty(problem.zipFilePath)) {
                    val attachmentZipFilePath =
                        getAttachmentZipFile(problemWithAttach.attach as ArrayList<Attachment>)
                    if (attachmentZipFilePath == null) {
                        Logger.e(TAG, "BugReport attachment zip file generation failed")
                    }
                    problem.zipFilePath = attachmentZipFilePath ?: ""
                    problemRepository.updateProblem(problem)
                }
                emitter.onNext(problem)
            }
            .flatMap {
                val fileBody = if (problem.zipFilePath.isNotEmpty()) {
                    val file = File(problem.zipFilePath)
                    val requestFile =
                        RequestBody.create("multipart/form-data".toMediaTypeOrNull(), file)
                    MultipartBody.Part.createFormData("packet", file.name, requestFile)
                } else {
                    null
                }

                val tags = GsonUtils.fromJson(problem.tags, Array<Int>::class.java)
                reportProblem(
                    problem.appVersion,
                    problem.description,
                    problem.issueName,
                    problem.issueStreamId.toString(),
                    problem.oppositeStreamId,
                    problem.platform,
                    problem.rid,
                    problem.sdkVersion,
                    tags?.toIntArray(),
                    fileBody
                )
            }
            .doOnNext {
                if (problem.zipFilePath.isNotEmpty()) {
                    FileUtils2.delete(problem.zipFilePath)
                }
            }
            .subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<IssueNo>() {
                override fun onSuccess(t: IssueNo?) {
                    Logger.i(TAG, "Successfully to report issue")
                    problem.issueId = t?.issueId
                    problem.status = UploadingStatus.COMPLETED
                    problemRepository.updateProblem(problem)

                    postEvent(problem, UploadingStatus.COMPLETED)
                }

                override fun onSubscribe(d: Disposable) {
                    super.onSubscribe(d)
                    reportIssueDisposable = d
                    problem.status = UploadingStatus.STARTING
                    problemRepository.updateProblem(problem)

                    postEvent(problem, UploadingStatus.STARTING)
                }

                override fun onFail(t: IssueNo?) {
                    Logger.i(TAG, "Failed to report issue")
                    problem.status = UploadingStatus.ERROR
                    problemRepository.updateProblem(problem)
                    postEvent(problem, UploadingStatus.ERROR)
                }
            })
    }

    override fun reportIssue(issue: Issue) {
        Observable
            .create<String> { emitter ->
                val path = issue.path
                val attachmentList = issue.attachmentList ?: arrayListOf()
                if (path == null || path.isEmpty()) {
                    val attachmentZipFilePath = getAttachmentZipFile(attachmentList)
                    if (attachmentZipFilePath == null) {
                        Logger.e(TAG, "BugReport attachment zip file generation failed")
                    }
                    issue.path = attachmentZipFilePath
                }
                if (issue.path != null) {
                    emitter.onNext(issue.path!!)
                } else {
                    emitter.onError(NullPointerException("issue path is empty"))
                }
            }
            .flatMap { zipPath ->
                val fileBody = if (zipPath.isNotEmpty()) {
                    val file = File(zipPath)
                    val requestFile =
                        RequestBody.create("multipart/form-data".toMediaTypeOrNull(), file)
                    MultipartBody.Part.createFormData("packet", file.name, requestFile)
                } else {
                    null
                }
                reportProblem(
                    issue.appVersion,
                    issue.description,
                    issue.issueName,
                    issue.issueStreamId.toString(),
                    issue.oppositeStreamId,
                    issue.platform,
                    issue.rid,
                    issue.sdkVersion,
                    issue.tags?.toIntArray(),
                    fileBody
                )
            }
            .doOnNext {
                if (it.success) {
                    issue.path?.let {
                        if (it.isNotEmpty()) {
                            FileUtils2.delete(it)
                        }
                    }
                }
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<IssueNo>() {
                override fun onSuccess(t: IssueNo?) {
                    Logger.i(TAG, "Successfully to report issue")
                    issue.issueId = t?.issueId
                    issue.status = UploadingStatus.COMPLETED
                    problemRepository.clearIssue()
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal,
                            issue
                        )
                    )
                }

                override fun onSubscribe(d: Disposable) {
                    super.onSubscribe(d)
                    reportIssueDisposable = d
                    issue.status = UploadingStatus.STARTING
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal,
                            issue
                        )
                    )
                }

                override fun onFail(t: IssueNo?) {
                    Logger.i(TAG, "Failed to report issue")
                    issue.status = UploadingStatus.ERROR
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal,
                            issue
                        )
                    )
                }
            })
    }

    override fun cancelReportIssue() {
        if (reportIssueDisposable != null && reportIssueDisposable?.isDisposed == false) {
            reportIssueDisposable?.dispose()
        }
    }

    private fun getAttachmentZipFile(attachList: ArrayList<Attachment>): String? {
        val dumpPathList = arrayListOf<String>()
        val zipPath = io.agora.avc.utils.FileUtils.getFileDir(null).absolutePath +
                File.separator +
                "$ZIP_FILE_NAME ${TimeUtils2.getNowString()}" +
                ZIP_SUFFIX
        if (attachList.isNotEmpty()) {
            attachList.forEach { _attach ->
                _attach.path?.let { _path ->
                    if (!StringUtils.isEmpty(_path)) {
                        dumpPathList.add(_path)
                    }
                }
            }
        }
        logUploader.getLogPath()?.let { logPath ->
            dumpPathList.add(logPath)
        }
        if (ZipUtils.zipFiles(dumpPathList, zipPath)) {
            return zipPath
        }
        return null
    }

    private fun postEvent(problem: Problem, status: Int) {
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal, BugReporterEvent(problem, status)
            )
        )
    }

    override fun reportProblem(
        appVersion: String,
        description: String?,
        issueName: String,
        issueStreamId: String,
        oppositeStreamId: Int?,
        platform: String,
        rid: String,
        sdkVersion: String,
        tags: IntArray?,
        file: MultipartBody.Part?
    ): Observable<IssueNo> {
        val map: HashMap<String, RequestBody> = hashMapOf()
        map["appVersion"] = RequestBody.create(MultipartBody.FORM, appVersion)
        map["issueName"] = RequestBody.create(MultipartBody.FORM, issueName)
        map["issueStreamId"] = RequestBody.create(MultipartBody.FORM, issueStreamId)
        description?.apply {
            map["description"] = RequestBody.create(MultipartBody.FORM, this)
        }
        if (oppositeStreamId != null && oppositeStreamId != -1) {
            map["oppositeStreamId"] =
                RequestBody.create(MultipartBody.FORM, oppositeStreamId.toString())
        }
        map["platform"] = RequestBody.create(MultipartBody.FORM, platform)
        map["rid"] = RequestBody.create(MultipartBody.FORM, rid)
        map["sdkVersion"] = RequestBody.create(MultipartBody.FORM, sdkVersion)
        tags?.apply {
            map["tags"] = RequestBody.create(MultipartBody.FORM, GsonUtils.toJson(this))
        }
        return dataRepository.getRetrofitService(ApiService::class.java)
            .reportIssue(map, file)
    }

    companion object {
        private const val TAG = "[COMM][BugReporter]"
        private const val ZIP_FILE_NAME = "audio_dump"
        private const val ZIP_SUFFIX = ".zip"
    }
}