package io.agora.avc.manager.link

import android.net.Uri
import io.agora.avc.R
import io.agora.avc.biz.RoomChecker
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.config.PREFIX_INTERNAL_MEETING
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.ClipboardUtils
import io.agora.avc.utils.SchemeUtils
import io.agora.avc.utils.StringUtils
import io.agora.avc.utils.ToastUtils
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import javax.inject.Inject

class LinkManagerImpl @Inject constructor(
    private val dataRepository: IDataRepository,
    private val roomRepository: RoomRepository,
    private val appEventBus: AppEventBus,
    private val noticeManager: NoticeManager,
) : LinkManager {

    override fun createShareLinkFromServer(
        rid: String,
        pwd: String?,
        nickName: String?
    ) {
        getShareLink(rid)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<Link>() {
                override fun onSuccess(t: Link?) {
                    Logger.i(TAG, "Successfully to get share link:${t?.linkId}")
                    t?.let {
                        createAgoraLink2Clipboard(rid, pwd, nickName, 1, it.linkId)
                    }
                }

                override fun onFail(t: Link?) {
                    Logger.e(TAG, "Failed to get share link")
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_INVITE_LINK_FAILED))
                }
            })
    }

    override fun createAgoraLink2Clipboard(
        rid: String,
        pwd: String?,
        nickName: String?,
        type: Int,
        code: String?
    ) {
        val link = SchemeUtils.generateAgoraLink(rid, pwd, type, code)
        val text = StringUtils.getString(
            R.string.avc_room_agora_invite_url,
            nickName,
            "$rid",
            "$link"
        )
        ClipboardUtils.copyText(CLIPBOARD_KEY_LINK, text)
        if (type == 0) {
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_AGORA_INVITE_DISALLOW))
        } else if (type == 1) {
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_AGORA_INVITE_ALLOW))
        }
    }

    override fun createShareLink2Clipboard(username: String, rid: String, pwd: String?) {
        val link = SchemeUtils.generateRoomLink(rid, pwd)
        val text = StringUtils.getString(
            R.string.avc_room_public_invite_url,
            username,
            "$rid",
            "$link"
        )
        ClipboardUtils.copyText(CLIPBOARD_KEY_LINK, text)
        ToastUtils.showShort(R.string.avc_room_link_copy_success)
    }

    override fun handleSchemeLink(data: Uri, roomInfo: Room?, localUser: LocalUser) {
        val roomChecker = RoomChecker()
        val path = SchemeUtils.getPathFromURI(data)
        if (path == null || path.isEmpty()) return
        when (path) {
            SchemeUtils.DEBUG_SCHEME_ROOM,
            SchemeUtils.SCHEME_ROOM -> {
                handleNormalSchemeLink(data, roomInfo, roomChecker)
            }
            SchemeUtils.DEBUG_SECURE_ROOM,
            SchemeUtils.SECURE_ROOM -> {
                handleAgoraSchemeLink(data, roomChecker, roomInfo, localUser)
            }
            else -> {
                ToastUtils.showLong(R.string.avc_scheme_unknown)
            }
        }
    }

    override fun handleOemLink(info: OemInfo, roomInfo: Room?) {
        if (roomInfo?.name == info?.rid) {
            Logger.i(TAG, "scheme: already in same room, not handle")
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_ALREADY_IN))
            return
        }
        Logger.i(TAG, "scheme: already in different room, pop dialog to choose")
        val room = Room(info.rid ?: "", info.pwd, RoomMode.NORMAL.value)
        appEventBus.notifyObservers(MessageEvent(AppEvent.LINK_ENTER_NEW_ROOM_EVENT.ordinal, room))
    }

    private fun handleNormalSchemeLink(data: Uri, roomInfo: Room?, roomChecker: RoomChecker) {
        val rid = SchemeUtils.getRidFromURI(data)
        if (rid == null) {
            Logger.e(TAG, "scheme: $data, not contain rid")
            return
        }
        val pwd = SchemeUtils.getPwdFromURI(data)
        val mode = if (rid != null && roomChecker.isAgoraRoom(rid)) {
            RoomMode.AGORA.value
        } else {
            RoomMode.NORMAL.value
        }
        if (!roomChecker.checkRoom(rid, pwd, mode)) return
        notifyLinkEnterEvent(roomInfo, rid, pwd, mode)
    }

    private fun handleAgoraSchemeLink(
        data: Uri,
        roomChecker: RoomChecker,
        roomInfo: Room?,
        localUser: LocalUser
    ) {
        val rid = SchemeUtils.getRidFromURI(data)
        val pwd = SchemeUtils.getPwdFromURI(data)
        if (rid == null || !rid.startsWith(PREFIX_INTERNAL_MEETING)) {
            Logger.w(TAG, "internal room name is not starts with AG-")
            return
        }
        val mode = if (rid != null && roomChecker.isAgoraRoom(rid)) {
            RoomMode.AGORA.value
        } else {
            RoomMode.NORMAL.value
        }
        if (!roomChecker.checkRoom(rid, pwd, mode)) return
        val type = SchemeUtils.getTypeFromURI(data)
        val code = SchemeUtils.getCodeFromURI(data)
        if (type == 0) {//need login
            if (roomInfo?.name == rid) {
                Logger.i(TAG, "scheme: already in same room, not handle")
                noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_ALREADY_IN))
                return
            }
            if (!localUser.isThirdPartyLoggedIn) {
                appEventBus.notifyObservers(
                    MessageEvent(
                        AppEvent.LINK_LOGIN_REQUEST_EVENT.ordinal,
                        Room(name = rid, pwd = pwd, mode)
                    )
                )
                return
            }
            notifyLinkEnterEvent(roomInfo, rid, pwd, mode)
        } else if (type == 1) {//need not login
            if (code == null) {
                Logger.w(TAG, "The code that allows unlogged users to join the room is empty")
                return
            }
            if (!localUser.isThirdPartyLoggedIn) {
                queryMeetingInfo(code, roomInfo)
                return
            }
            notifyLinkEnterEvent(roomInfo, rid, pwd, mode)
        }
    }

    private fun queryMeetingInfo(code: String, roomInfo: Room?) {
        roomRepository.queryInnerMeetingInfo(code)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<InnerMeeting>() {
                override fun onSuccess(it: InnerMeeting?) {
                    if (it != null && !it.isInner) {
                        notifyLinkEnterEvent(roomInfo, it.rid, it.pass, RoomMode.AGORA.value)
                    } else {
                        noticeManager.notice(NoticeEvent(NoticeCode.CODE_JOIN_FAIL_LINK_EXPIRED))
                    }
                }

                override fun onFail(t: InnerMeeting?) {
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_JOIN_FAIL_LINK_EXPIRED))
                }
            })
    }

    private fun notifyLinkEnterEvent(roomInfo: Room?, rid: String, pwd: String?, mode: Int) {
        if (roomInfo == null) {
            Logger.i(TAG, "scheme: I'm not in the room, enter the room")
            appEventBus.notifyObservers(
                MessageEvent(
                    AppEvent.LINK_ENTER_ROOM_EVENT.ordinal,
                    Room(name = rid, pwd = pwd, mode = mode)
                )
            )
            return
        }
        if (roomInfo?.name == rid) {
            Logger.i(TAG, "scheme: already in same room, not handle")
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_ROOM_ALREADY_IN))
            return
        }
        Logger.i(TAG, "scheme: already in different room, pop dialog to choose")
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.LINK_ENTER_NEW_ROOM_EVENT.ordinal,
                Room(name = rid, pwd = pwd, mode = mode)
            )
        )
    }

    private fun getShareLink(rid: String): Observable<Link> {
        return dataRepository.getRetrofitService(ApiService::class.java)
            .queryLinkId(rid)
    }

    companion object {
        private const val TAG = "[COMM][LinkManager]"

        const val CLIPBOARD_KEY_LINK = "USER_REQUEST"
    }
}