package io.agora.avc.manager.logupload

import android.app.Application
import android.os.Build
import android.text.TextUtils
import io.agora.avc.MyApplication
import io.agora.avc.bo.LocalUser
import io.agora.avc.extensions.getConferenceNickname
import io.agora.avc.net.api.ApiService
import io.agora.avc.utils.*
import io.agora.frame.data.IDataRepository
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody
import okhttp3.ResponseBody
import okio.ByteString.Companion.encodeUtf8
import java.io.File
import javax.inject.Inject

class LogUploaderImpl @Inject constructor(
    val application: Application,
    private val dataRepository: IDataRepository
) : LogUploader {

    override fun uploadLog(streamId: Int?): Observable<ResponseBody> {
        return Observable
            .create<File> {
                val logPath = getLogPath()
                if (logPath != null) {
                    val filePath = getZipFilePath()
                    FileUtils2.delete(filePath)
                    ZipUtils.zipFile(logPath, filePath)
                    it.onNext(File(filePath))
                } else {
                    it.onError(Throwable("no such file"))
                }
            }
            .flatMap {
                val requestFile = RequestBody.create("multipart/form-data".toMediaTypeOrNull(), it)
                val body = MultipartBody.Part.createFormData("log", it.name, requestFile)
                uploadLog(streamId, body)
            }
            .doOnNext {
                FileUtils2.delete(getZipFilePath())
            }
    }

    override fun doUploadLog(streamId: Int) {
        uploadLog(streamId)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<ResponseBody> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(it: ResponseBody) {

                }

                override fun onError(e: Throwable) {

                }

                override fun onComplete() {

                }
            })
    }

    /**
     * If the directory does not exist, create the directory
     */
    override fun getLogPath(): String? {
        val filePath = io.agora.avc.utils.FileUtils.getFileDir(null).absolutePath
        if (filePath == null || TextUtils.isEmpty(filePath)) {
            return filePath
        }

        val logFileDir = File(filePath + File.separator + "Log" + File.separator)
        if (!logFileDir.exists()) {
            return null
        }

        return logFileDir.absolutePath
    }

    override fun getValoranLogPath(): String {
        return getLogPath() + File.separator
    }

    private fun uploadLog(streamId: Int?, file: MultipartBody.Part): Observable<ResponseBody> {
        val uid = RequestBody.create(MultipartBody.FORM, streamId?.toString() ?: "")
        val platform = RequestBody.create(MultipartBody.FORM, "android")
        val appVersion = RequestBody.create(
            MultipartBody.FORM,
            AppUtils.getAppVersionName() + "-" + SDKUtils.getRTCSdkVersion() + "-" + SDKUtils.getRTMSdkVersion()
        )
        val osVersion = RequestBody.create(MultipartBody.FORM, Build.VERSION.RELEASE)
        // avoid context.getString(), because getString() may return zh character(need base64)
        val appName = RequestBody.create(MultipartBody.FORM, "Agora Video Call")
        val map = java.util.HashMap<String, RequestBody>()
        map["identifier"] = uid
        map["platform"] = platform
        map["appVersion"] = appVersion
        map["os"] = osVersion
        map["appName"] = appName
        return dataRepository.getRetrofitService(ApiService::class.java)
            .uploadLog(map, file)
    }

    private fun getZipFilePath(): String {
        var localUser: LocalUser? = null
        if (application is MyApplication) {
            localUser = application.appContainer.localUser
        }

        return io.agora.avc.utils.FileUtils.getFileDir(null).absolutePath +
                File.separator +
                "AVC" +
                AppUtils.getAppVersionName() +
                "." +
                AppUtils.getAppVersionCode() +
                "-Android-" +
                localUser?.streamId +
                "-" +
                localUser?.getConferenceNickname()?.encodeUtf8()?.base64Url() +
                "-" +
                DeviceUtils.getBrand() +
                "-" +
                DeviceUtils.getModel() +
                "-" +
                DeviceUtils.getSDKVersionName() +
                ".zip"

    }

    companion object {
        private const val TAG = "[COMM][LogUploader]"
    }
}