package io.agora.avc.manager.network

import android.app.Application
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.ConnectivityManager
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.utils.NetworkUtils
import io.agora.logger.Logger
import javax.inject.Inject

class NetWorkManagerImpl @Inject constructor(
    private val application: Application,
    private val appEventBus: AppEventBus,
) : NetWorkManager {

    private var mType: NetworkUtils.NetworkType? = null
    private var connected: Boolean? = null

    override fun install() {
        netWorkReceiver.setOnReceiver { _, intent ->
            if (ConnectivityManager.CONNECTIVITY_ACTION == intent?.action) {
                val networkType = NetworkUtils.getNetworkType()
                if (mType == networkType) return@setOnReceiver
                mType = networkType
                var reConnected: Boolean
                (networkType != NetworkUtils.NetworkType.NETWORK_NO).let {
                    reConnected = it != connected && it
                    connected = it
                }

                appEventBus.notifyObservers(
                    MessageEvent(
                        AppEvent.NET_WORK_CHANGED_EVENT.ordinal,
                        NetWorkInfo(
                            connected = networkType != NetworkUtils.NetworkType.NETWORK_NO,
                            reConnected = reConnected,
                            networkType = networkType,
                        ).apply {
                            Logger.i(TAG, "net work changed, connected = $connected, reConnected = $reConnected, netWorkType = $networkType")
                        }
                    )
                )
            }
        }
        application.registerReceiver(
            netWorkReceiver,
            IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION)
        )
    }

    override fun unInstall() {
        application.unregisterReceiver(netWorkReceiver)
        netWorkReceiver.setOnReceiver(null)
    }

    companion object {
        const val TAG = "[COMM][NetWorkManager]"
        private val netWorkReceiver = NetWorkReceiver()
    }
}

class NetWorkReceiver : BroadcastReceiver() {

    private var onReceiver: ((context: Context?, intent: Intent?) -> Unit)? = null

    override fun onReceive(context: Context?, intent: Intent?) {
        onReceiver?.invoke(context, intent)
    }

    fun setOnReceiver(onReceiver: ((context: Context?, intent: Intent?) -> Unit)?) {
        this.onReceiver = onReceiver
    }
}

data class NetWorkInfo(
    val connected: Boolean,
    val reConnected: Boolean,
    val networkType: NetworkUtils.NetworkType,
)
