package io.agora.avc.manager.notice

import android.content.res.Resources
import android.os.Bundle
import androidx.core.os.bundleOf
import com.agora.valoran.Constants
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.StringUtils
import io.agora.logger.Logger
import javax.inject.Inject

class NoticeManagerImpl @Inject constructor(
    private val appEventBus: AppEventBus,
) : NoticeManager {

    private fun getLocalUser(): LocalUser? {
        val appContext = MyApplication.appContext
        if (appContext is MyApplication) {
            return appContext.appContainer.localUser
        }
        return null
    }

    private fun getRoomInfo(): Room? {
        val appContext = MyApplication.appContext
        if (appContext is MyApplication) {
            return appContext.appContainer.room
        }
        return null
    }


    override fun notice(event: NoticeEvent) {
        val messageEvent = getNotice(event)
        if (event.code == NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE
            || event.code == NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE
        ) {
            if (!AppUtils.isAppForeground()) {
                messageEvent.type = AppEvent.NOTIFY_EVENT_APP_IN_BACKGROUND.ordinal
            }
        }
        appEventBus.notifyObservers(
            messageEvent
        )
    }

    override fun notice(code: NoticeCode) {
        notice(NoticeEvent(code = code))
    }

    override fun onNotify(type: Int, sourceName: String?, targetName: String?) {
        when (type) {
            //local event
            Constants.NOTIFY_LOCAL_AUDIO_MUTE_BY_REMOTE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_INVITE_MICROPHONE_CLOSE,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        sourceName
                    )
                )
            }
            Constants.NOTIFY_LOCAL_VIDEO_MUTE_BY_REMOTE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_INVITE_CAMERA_CLOSE,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        sourceName
                    )
                )
            }
            Constants.NOTIFY_LOCAL_AUDIO_VIDEO_AUTO_MUTE -> {
                val event = MessageEvent(
                    AppEvent.POPUP_EVENT.ordinal,
                    NoticeCode.CODE_DEFAULT_BOTH
                )
                appEventBus.notifyObservers(event)
            }
            Constants.NOTIFY_LOCAL_AUDIO_AUTO_MUTE -> {
                val event = MessageEvent(
                    AppEvent.POPUP_EVENT.ordinal,
                    NoticeCode.CODE_DEFAULT_MICROPHONE
                )
                appEventBus.notifyObservers(event)
            }
            Constants.NOTIFY_LOCAL_VIDEO_AUTO_MUTE -> {
                val event = MessageEvent(
                    AppEvent.POPUP_EVENT.ordinal,
                    NoticeCode.CODE_DEFAULT_CAMERA
                )
                appEventBus.notifyObservers(event)
            }
            Constants.NOTIFY_CAMERA_DISABLED_BY_PROMPT_ACTION -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_NETWORK_CAMERA_DONE,
                        NoticeType.TOAST
                    )
                )
            }
            Constants.NOTIFY_LOCAL_MEDIA_NEED_TO_DETERMINE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ROOM_RTM_DISCONNECTED_PREVIEW,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_5
                    )
                )
            }
            Constants.NOTIFY_LOCAL_MEDIA_NEED_TO_DETERMINE_CANCEL -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ROOM_RTM_DISCONNECTED_PREVIEW,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_5,
                        show = false
                    )
                )
            }
            Constants.NOTIFY_SERVER_REQUEST_UPLOAD_LOG -> {

            }
            Constants.NOTIFY_UX_VIDEO_PLAN_DUMP_OVER -> {

            }
            Constants.NOTIFY_MEDIA_LOST -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ROOM_RTC_DISCONNECTED,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_1,
                    )
                )
            }
            Constants.NOTIFY_MEDIA_LOST_RECOVERY -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ROOM_RTC_DISCONNECTED,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_1,
                        show = false
                    )
                )
            }
            Constants.NOTIFY_USER_TEMPORARY_LEAVE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ROOM_LEFT_TOAST,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_4,
                        sourceName
                    )
                )
            }
            Constants.NOTIFY_REMOTE_SHARE_START -> {//do nothing

            }
            //remote event
            Constants.NOTIFY_REMOTE_USER_KICKED_OUT -> {

            }
            Constants.NOTIFY_REMOTE_USER_KICKED_OUT_BY_SERVER -> {

            }
            Constants.NOTIFY_REMOTE_REFUSE_AUDIO -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_INVITE_MICROPHONE_REJECT,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_REMOTE_REFUSE_VIDEO -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_INVITE_CAMERA_REJECT,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_STRANGER_IN -> {
                // do nothing
            }
            Constants.NOTIFY_LOCAL_REQUEST_REMOTE_NOT_RESPONSE_AUDIO -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE_RESPOND,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_LOCAL_REQUEST_REMOTE_NOT_RESPONSE_VIDEO -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE_RESPOND,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_MUTE_REMOTE_AUDIO_NOT_RESPONSE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_NETWORK_OFF_MICROPHONE,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_MUTE_REMOTE_VIDEO_NOT_RESPONSE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_NETWORK_OFF_CAMERA,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_2,
                        targetName
                    )
                )
            }
            Constants.NOTIFY_ASSISTANT_JOIN -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_AS_ROOM_BACK,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_8,
                    )
                )
            }
            Constants.NOTIFY_ASSISTANT_LEAVE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_AS_ROOM_LEAVE_OUT,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_8,
                    )
                )
                appEventBus.notifyObservers(MessageEvent(AppEvent.ROOM_ASSISTANT_CHANGED.ordinal))
            }
            Constants.NOTIFY_ASSISTANT_TEMPORARY_LEAVE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_AS_ROOM_LEAVE_TEMP,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_8,
                    )
                )
            }
            Constants.NOTIFY_ASSISTANT_RESUME_FROM_TEMPORARY_LEAVE -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_AS_ROOM_BACK,
                        NoticeType.NOTIFICATION_A,
                        MeetingNotificationType.NO_8,
                    )
                )
            }
            Constants.NOTIFY_ME_KICKED_OUT -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_JOIN_ROOM_REMOVE,
                        NoticeType.DIALOG,
                        obj = sourceName
                    )
                )
            }
            Constants.NOTIFY_ME_KICKED_OUT_BY_SERVER -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_JOIN_FAIL_SEVER_KICK_OUT,
                        NoticeType.DIALOG
                    )
                )
            }
            Constants.NOTIFY_ME_KICKED_OUT_BY_ASSISTANT -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_JOIN_ROOM_REMOVE,
                        NoticeType.DIALOG,
                        obj = StringUtils.getString(R.string.avc_as_name)
                    )
                )
            }
            //room event
            Constants.NOTIFY_HOST_NEW -> {
                // do nothing
            }
            Constants.NOTIFY_HOST_GIVE_UP -> {
                // do nothing
            }
            Constants.NOTIFY_ROOM_AUDIO_MUTED -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_AVC_MUTE_ALL_RAISE_POP,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_5,
                    )
                )
            }
            Constants.NOTIFY_ROOM_AUDIO_UNMUTED -> {
                notice(
                    NoticeEvent(
                        NoticeCode.CODE_ALL_UNMUTED,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_5,
                    )
                )
            }
            else -> {
            }
        }

    }

    override fun onNotifyResult(type: Int, success: Boolean, code: Int, seq: Int) {
        var noticeEvent = when (type) {
            Constants.NOTIFY_APPLY_HOST_OPERATION_RESULT -> {
                getNoticeEventFormApplyHostOperation(code)
            }
            Constants.NOTIFY_GIVE_UP_HOST_OPERATION_RESULT -> {
                getNoticeEventFromGiveUpHostOperation(code)
            }
            Constants.NOTIFY_ROOM_AUDIO_OPERATION_RESULT -> {
                getNoticeEventFromRoomAudioOperation(code)
            }
            Constants.NOTIFY_ROOM_VIDEO_OPERATION_RESULT -> {
                getNoticeEventFromRoomVideoOperation(code)
            }
            Constants.NOTIFY_START_RECORDING_OPERATION_RESULT -> {
                getNoticeEventFromStartRecordingOperation(code)
            }
            Constants.NOTIFY_STOP_RECORDING_OPERATION_RESULT -> {
                getNoticeEventFromStopRecordingOperation(code)
            }
            Constants.NOTIFY_APPLY_ASSISTANT_OPERATION_RESULT -> {
                getApplyAssistantOperationNotice(success, code)
            }
            Constants.NOTIFY_CANCEL_ASSISTANT_OPERATION_RESULT -> {
                getCancelAssistantOperationNotice(success, code)
            }
            Constants.NOTIFY_INVITE_CONTACTS_OPERATION_RESULT -> {
                getInviteContactsOperationNotice(code)
            }
            Constants.NOTIFY_PICK_UP_INCOMING_OPERATION_RESULT -> {
                getPickUpIncomingOperationNotice(code)
            }
            Constants.NOTIFY_HANG_UP_INCOMING_OPERATION_RESULT -> {
                //do nothing
            }
            Constants.NOTIFY_APPLY_UNMUTE_LOCAL_MEDIA_RESULT -> {
                getApplyUnMuteLocalMediaOperationNotice(success, code)
            }
            else -> {
                null
            }
        }

        if (noticeEvent == null) {
            val result = getCommonOperationNoticeEvent(code)
            if (result is NoticeEvent) {
                noticeEvent = result
            }
        }
        if (noticeEvent is NoticeEvent) {
            notice(noticeEvent)
        }
    }

    private fun getApplyUnMuteLocalMediaOperationNotice(success: Boolean, code: Int): NoticeEvent? {
        return if (success) {
            NoticeEvent(NoticeCode.CODE_AVC_MUTE_ALL_AGREE_NOTIFY)
        } else {
            when (code) {
                Constants.NOTIFY_BIZ_WAIT_RESPONSE_TIMEOUT -> {
                    NoticeEvent(NoticeCode.CODE_AVC_MUTE_ALL_NO_RESPOND_NOTIFY)
                }
                Constants.NOTIFY_BIZ_REJECT_BY_USER -> {
                    NoticeEvent(NoticeCode.CODE_AVC_MUTE_ALL_REJECT_NOTIFY)
                }
                else -> {
                    null
                }
            }
        }
    }

    private fun getPickUpIncomingOperationNotice(code: Int): NoticeEvent? {
        return when (code) {
            Constants.NOTIFY_INCOMING_EXPIRED -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_INVALID
                )
            }
            Constants.NOTIFY_INCOMING_ANSWER_FAILED -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_ALERT
                )
            }
            Constants.NOTIFY_INCOMING_NOT_IN_INNER_ROOM -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_ALERT
                )
            }
            Constants.NOTIFY_INCOMING_ALREADY_IN -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_ALERT
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getInviteContactsOperationNotice(code: Int): NoticeEvent? {
        return when (code) {
            Constants.NOTIFY_INCOMING_INVITE_FAILED -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_ALERT
                )
            }
            Constants.NOTIFY_INCOMING_NOT_IN_INNER_ROOM -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_ALERT
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getCancelAssistantOperationNotice(success: Boolean, code: Int): NoticeEvent? {
        //No prompt
        return null
    }

    private fun getApplyAssistantOperationNotice(success: Boolean, code: Int): NoticeEvent? {
        return if (success) {
            if (code == Constants.NOTIFY_CODE_ASSISTANT_APPLIED_BY_HOST) {
                NoticeEvent(
                    NoticeCode.CODE_AS_ROOM_APPLY_SUCCESS,
                    NoticeType.NOTIFICATION_A,
                    MeetingNotificationType.NO_8,
                )
            } else {
                null
            }
        } else {
            when (code) {
                Constants.NOTIFY_BIZ_WAIT_RESPONSE_TIMEOUT -> {
                    NoticeEvent(NoticeCode.CODE_AS_ROOM_APPLY_TIME_OUT)
                }
                Constants.NOTIFY_BIZ_REJECT_BY_USER -> {
                    NoticeEvent(NoticeCode.CODE_AS_ROOM_APPLY_FAIL)
                }
                else -> {//default apply fail
                    NoticeEvent(NoticeCode.CODE_AS_SETTING_APPLY_FAIL)
                }
            }
        }

    }

    override fun onNetworkQualityPrompt(code: NoticeCode, action: Int, param: String?) {
        val resources = MyApplication.appContext.resources
        var bundle: Bundle? = null
        val text = when (action) {
            Constants.PromptActionType.getValue(Constants.PromptActionType.ACTION_CHECK_LOCAL) -> {
                resources.getString(R.string.avc_network_local_reminder)
            }
            Constants.PromptActionType.getValue(Constants.PromptActionType.ACTION_WAIT_REMOTE) -> {
                resources.getString(R.string.avc_network_peer_wait)
            }
            Constants.PromptActionType.getValue(Constants.PromptActionType.ACTION_CHECK_REMOTE) -> {
                resources.getString(R.string.avc_network_peer_reminder)
            }
            Constants.PromptActionType.getValue(Constants.PromptActionType.ACTION_TURN_DOWN_RESOLUTION) -> {
                bundle = bundleOf("isLink" to true)
                resources.getString(R.string.avc_network_resolution_reminder, param)
            }
            Constants.PromptActionType.getValue(Constants.PromptActionType.ACTION_CLOSE_LOCAL_VIDEO) -> {
                bundle = bundleOf("isLink" to true)
                resources.getString(R.string.avc_network_camera_reminder)
            }
            else -> {
                ""
            }
        }

        notice(
            NoticeEvent(
                code = code,
                type = NoticeType.NOTIFICATION_A,
                notificationType = MeetingNotificationType.NO_1,
                obj = text,
                data = bundle
            )
        )
    }

    private fun getNoticeEventFormApplyHostOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get apply host operation result, code:$code")
        return null
    }

    private fun getNoticeEventFromGiveUpHostOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get give up host operation result, code:$code")
        return when (code) {
            Constants.NOTIFY_RECORDING_HOST_EXISTS -> {
                NoticeEvent(
                    NoticeCode.CODE_ROOM_RTM_DISCONNECTED
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getNoticeEventFromRoomAudioOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get room audio operation result, code:$code")
        return null
    }

    private fun getNoticeEventFromRoomVideoOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get room video operation result, code:$code")
        return null
    }

    private fun getCommonOperationNoticeEvent(code: Int): NoticeEvent? {
        Logger.w(TAG, "get common operation result, code:$code")
        return when (code) {
            Constants.NOTIFY_BIZ_GENERAL_ERROR -> {
                NoticeEvent(
                    NoticeCode.CODE_SETTING_DEFAULT_ROOM_SETTING_FAIL
                )
            }
            Constants.NOTIFY_BIZ_RTM_SEND_ERROR -> {
                null
            }
            Constants.NOTIFY_BIZ_RTM_SEND_TIMEOUT -> {
                null
            }
            Constants.NOTIFY_BIZ_WAIT_RESPONSE_TIMEOUT -> {
                null
            }
            Constants.NOTIFY_BIZ_ROOM_OPERATION_NO_PERMISSION -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_HOST
                )
            }
            Constants.NOTIFY_BIZ_USER_OPERATION_NO_PERMISSION -> {
                NoticeEvent(
                    NoticeCode.CODE_INVITE_HOST
                )
            }
            Constants.NOTIFY_BIZ_ROOM_DISCONNECTED -> {
                NoticeEvent(
                    NoticeCode.CODE_SETTING_DEFAULT_ROOM_SETTING_FAIL
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getNoticeEventFromStartRecordingOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get start recording operation result, code:$code")
        return when (code) {
            Constants.NOTIFY_RECORDING_HOST_EXISTS -> {
                NoticeEvent(
                    NoticeCode.CODE_CLOUD_RECORDING_HOST
                )
            }
            Constants.NOTIFY_RECORDING_ALREADY_EXISTS -> {
                NoticeEvent(
                    NoticeCode.CODE_CLOUD_RECORDING_EXIST
                )
            }
            Constants.NOTIFY_RECORDING_ROOM_NOT_ALLOWED -> {
                NoticeEvent(
                    NoticeCode.CODE_CLOUD_RECORDING_START_FAIL
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getNoticeEventFromStopRecordingOperation(code: Int): NoticeEvent? {
        Logger.w(TAG, "get stop recording operation result, code:$code")
        return when (code) {
            Constants.NOTIFY_RECORDING_TIME_TOO_SHORT -> {
                NoticeEvent(
                    NoticeCode.CODE_CLOUD_RECORDING_SHORT
                )
            }
            Constants.NOTIFY_RECORDING_FILE_FAILURE -> {
                NoticeEvent(
                    NoticeCode.CODE_CLOUD_RECORDING_FILE_FAIL
                )
            }
            else -> {
                null
            }
        }
    }

    private fun getNotice(event: NoticeEvent): MessageEvent {
        val text = convertCodeToString(event)
        return getNotice(event, text)
    }

    private fun getNotice(event: NoticeEvent, text: String): MessageEvent {
        return when (event.type) {
            NoticeType.NOTIFICATION_A -> {
                MessageEvent(
                    AppEvent.NOTIFY_EVENT.ordinal,
                    MeetingNotification(
                        code = event.code,
                        noticeType = event.type,
                        notificationType = event.notificationType!!,
                        title = text,
                        content = event.obj?.toString(),
                        contentIsLink = event.data?.getBoolean("isLink", false) ?: false,
                        show = event.show,
                        data = event.data,
                    )
                )
            }
            NoticeType.NOTIFICATION_C -> {
                MessageEvent(
                    AppEvent.NOTIFY_EVENT.ordinal,
                    MeetingNotification(
                        code = event.code,
                        noticeType = event.type,
                        notificationType = event.notificationType!!,
                        title = text,
                        show = event.show,
                        data = event.data,
                    )
                )
            }
            NoticeType.NOTIFICATION_B -> {
                MessageEvent(
                    AppEvent.NOTIFY_EVENT.ordinal,
                    MeetingNotification(
                        code = event.code,
                        noticeType = event.type,
                        notificationType = event.notificationType!!,
                        title = text,
                        show = event.show,
                        data = event.data,
                    )
                )
            }
            NoticeType.DIALOG -> {
                MessageEvent(
                    AppEvent.DIALOG_EVENT.ordinal,
                    event,
                )
            }
            else -> {
                event.obj = text
                MessageEvent(
                    AppEvent.TOAST_EVENT.ordinal,
                    event,
                )
            }
        }
    }

    private fun convertCodeToString(event: NoticeEvent): String {

        val resources = MyApplication.appContext.resources

        return try {
            event.code.stringsId?.let {
                resources.getString(event.code.stringsId, event.obj).let {
                    Logger.i(
                        TAG,
                        "convert code to string:{string:${it}}"
                    )
                    it
                }
            } ?: resources.getString(R.string.avc_unknown_toast)
        } catch (e: Exception) {
            event.code.stringsId?.let {
                Logger.i(
                    TAG,
                    "convert code to string exception:{string:${resources.getString(it)}, arg:${event.obj}}"
                )
            }
            resources.getString(R.string.avc_unknown_toast)
        }
    }

    private fun parseUserName(resources: Resources, username: Any?): String {
        return username as? String ?: resources.getString(R.string.avc_unknown_user)
    }

    private fun parseInviter(inviter: String?): String {
        return inviter ?: ""
    }

    companion object {
        private const val TAG = "[Comm][NoticeManager]"
    }
}