package io.agora.avc.manager.quality

import io.agora.avc.bo.ConversationQuality
import io.agora.avc.bo.LocalUser
import io.agora.avc.extensions.isMySelf
import io.agora.avc.extensions.isMyShareScreen
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.TimeUtils
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import javax.inject.Inject

class QualityManagerImpl @Inject constructor(
    private val dataRepository: IDataRepository,
    private val roomRepository: RoomRepository,
    private val developerRepository: DeveloperRepository,
) : QualityManager {

    private var startTime: Long = 0

    @Volatile
    private var publisherExisted: Boolean = false

    @Volatile
    private var remoteUserExisted: Boolean = false

    @Volatile
    private var permitReport: Boolean = true

    override fun start() {
        //filter multiple start
        if (startTime != 0L) {
            return
        }
        startTime = TimeUtils.elapsedRealTime()
    }

    override fun createConversation(
        channelId: String,
        name: String,
        reportId: String,
        reportStreamId: Int,
        callId: String?,
        peerUid: Int,
        manual: Boolean
    ): ConversationQuality {
        return ConversationQuality(
            name,
            channelId,
            reportId,
            reportStreamId,
            manual = manual,
            callId = callId,
            peerUid = peerUid
        )
    }

    override fun forbidQualityReport() {
        Logger.i(TAG, "User prohibits quality reports")
        this.permitReport = false
    }

    private fun getWatchDuration(): Long {
        return TimeUtils.elapsedRealTime() - startTime
    }

    private fun isBeyondMinTime(): Boolean {
        return getWatchDuration() >= MIN_DETECT_TIME
    }

    override fun notifyUserChanged(localUser: LocalUser) {
        if (publisherExisted && remoteUserExisted) {
            return
        }

        roomRepository.getRoomUserList().forEach { user ->
            //detectPublisher
            if (!publisherExisted &&
                !user.isMySelf() &&
                !user.isMyShareScreen(localUser) &&
                (user.isMediaOn())
            ) {
                Logger.i(TAG, "Publisher:${user.streamId} successfully detected")
                publisherExisted = true
            }

            //detectRemoteUser
            if (!remoteUserExisted && !user.isMySelf() && !user.isMyShareScreen(localUser)) {
                Logger.i(TAG, "Remote user:${user.streamId} successfully detected")
                remoteUserExisted = true
            }

            if (publisherExisted && remoteUserExisted) {
                return
            }
        }
    }

    override fun reportConversationQuality(
        conversationQuality: ConversationQuality,
        appVersionName: String
    ) {
        Logger.i(
            TAG,
            "report conversation quality, conversation:${conversationQuality?.reportStreamId}"
        )
        reportQuality(conversationQuality, appVersionName)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(t: BaseNo?) {
                    Logger.i(TAG, "Successfully to report conversation quality")
                }

                override fun onFail(t: BaseNo?) {
                    Logger.i(TAG, "Failed to report conversation quality")
                }
            })
    }

    override fun needQualityReport(): Boolean {
        Logger.i(
            TAG,
            "Quality report info," +
                    "function on:${developerRepository.getDeveloperOptions().qualityReport} " +
                    "during:${getWatchDuration()} " +
                    "publisherExisted:$publisherExisted " +
                    "remoteUserExisted:$remoteUserExisted " +
                    "permitReport:$permitReport"
        )
        if (developerRepository.getDeveloperOptions().qualityReport &&
            isBeyondMinTime() &&
            publisherExisted &&
            remoteUserExisted &&
            permitReport
        ) {
            return true
        }
        return false
    }

    private fun reportQuality(
        quality: ConversationQuality,
        appVersionName: String
    ): Observable<BaseNo> {
        val map: HashMap<String, Any> = hashMapOf()
        map["audio_rating"] = quality.audioRating.toInt()
        map["audio_label"] = quality.audioLabel ?: intArrayOf()
        map["audio_remark"] = quality.audioRemark
        map["channel_name"] = quality.channelName
        map["name"] = quality.name
        map["stream_id"] = quality.reportStreamId
        map["uid"] = quality.reportId
        map["video_rating"] = quality.videoRating.toInt()
        map["video_label"] = quality.videoLabel ?: intArrayOf()
        map["video_remark"] = quality.videoRemark
        map["app_version"] = appVersionName
        map["call_id"] = quality.callId ?: ""
        map["delta"] = quality.delta
        map["peer_uid"] = quality.peerUid
        return dataRepository.getRetrofitService(ApiService::class.java)
            .reportQuality(map)
    }

    companion object {
        private const val TAG = "[COMM][QualityManager]"
        const val MIN_DETECT_TIME = 1 * 60 * 1000
    }
}