package io.agora.avc.manager.rating.call

import com.google.gson.JsonObject
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.utils.*
import io.agora.avc.widget.UploadingStatus
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import javax.inject.Inject

class CallRatingManagerImpl @Inject constructor(
    private val dataRepository: IDataRepository,
    private val appEventBus: AppEventBus,
) : CallRatingManager {
    override var score: Float? = null
    override var userList: Array<Pair<ARoomUser, Float>>? = null

    private var callRatingDisposable: Disposable? = null
    private var audioRatingDisposable: Disposable? = null

    override fun scoringAudio(audioScore: AudioScore) {
        Logger.i(TAG, "audio scoring submit")
        val jsonObject = packAudioData(audioScore)
        dataRepository.getRetrofitService(ApiService::class.java)
            .audioReport(audioReportBody = jsonObject)
            .subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSubscribe(d: Disposable) {
                    super.onSubscribe(d)
                    audioRatingDisposable = d
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT.ordinal,
                            UploadingStatus.STARTING
                        )
                    )
                }

                override fun onSuccess(t: BaseNo?) {
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT.ordinal,
                            UploadingStatus.COMPLETED
                        )
                    )
                }

                override fun onFail(t: BaseNo?) {
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.AUDIO_RATING_UPLOAD_STATUS_CHANGE_EVENT.ordinal,
                            UploadingStatus.ERROR
                        )
                    )
                }
            })
    }

    private fun packAudioData(audioScore: AudioScore): JsonObject {
        val jsonObject = JsonObject()
        jsonObject.addProperty("avmos", audioScore.rating)
        jsonObject.addProperty("cname", audioScore.rid)
        jsonObject.addProperty("uid0", audioScore.localUid)
        jsonObject.addProperty("mos0", audioScore.localScore)

        audioScore.tags?.let {
            jsonObject.addProperty("feedback", GsonUtils.toJson(it))
        }
        audioScore.selectedUid?.let {
            jsonObject.addProperty("s_uid", it)
        }

        if (!StringUtils.isTrimEmpty(audioScore.description)) {
            jsonObject.addProperty("b_name", audioScore.description)
        }

        audioScore.userList.forEachIndexed { index, aRoomUser ->
            when (index) {
                0 -> {
                    jsonObject.addProperty("uid1", aRoomUser.first.streamId)
                    jsonObject.addProperty("mos1", aRoomUser.second)
                }
                1 -> {
                    jsonObject.addProperty("uid2", aRoomUser.first.streamId)
                    jsonObject.addProperty("mos2", aRoomUser.second)
                }
                2 -> {
                    jsonObject.addProperty("uid3", aRoomUser.first.streamId)
                    jsonObject.addProperty("mos3", aRoomUser.second)
                }
            }
        }
        return jsonObject
    }

    private fun packVideoData(videoScore: VideoScore): JsonObject {
        val jsonObject = JsonObject()
        jsonObject.addProperty("avmos", videoScore.score)
        jsonObject.addProperty("cname", videoScore.rid)
        jsonObject.addProperty("uid0", videoScore.uid)
        jsonObject.addProperty("mos", videoScore.mos)

        videoScore.tags?.let {
            jsonObject.addProperty("feedback", GsonUtils.toJson(it))
        }
        videoScore.sid?.let {
            jsonObject.addProperty("sid", it)
        }
        if (!StringUtils.isTrimEmpty(videoScore.bName)) {
            jsonObject.addProperty("b_name", videoScore.bName)
        }
        return jsonObject
    }

    override fun scoringCall(callScore: CallScore) {
        val jsonObject = JsonObject()
        val jsonUser = JsonObject()
        jsonUser.addProperty("rating", callScore.userScore.score)
        callScore.userScore.tags?.let {
            jsonObject.addProperty("feedback", GsonUtils.toJson(it))
        }
        jsonUser.addProperty("feedback", callScore.userScore.description)

        jsonObject.addProperty("appVersion", AppUtils.getAppVersionName())
        jsonObject.addProperty("platform", DeviceUtils.getPlatform())
        jsonObject.addProperty("rid", callScore.audioScore.rid)
        jsonObject.addProperty("sdkVersion", SDKUtils.getRTCSdkVersion())
        jsonObject.add("user", jsonUser)
        jsonObject.add("audio", packAudioData(callScore.audioScore))
        jsonObject.add("video", packVideoData(callScore.videoScore))

        dataRepository.getRetrofitService(ApiService::class.java)
            .reportCallQuality(jsonObject)
            .subscribeOn(Schedulers.io())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSubscribe(d: Disposable) {
                    super.onSubscribe(d)
                    callRatingDisposable = d
                }

                override fun onSuccess(t: BaseNo?) {

                }

                override fun onFail(t: BaseNo?) {

                }
            })
    }

    override fun cancelAudioRating() {
        if (audioRatingDisposable != null && audioRatingDisposable?.isDisposed == false) {
            audioRatingDisposable?.dispose()
        }
    }

    override fun cancelCallRating() {
        if (callRatingDisposable != null && callRatingDisposable?.isDisposed == false) {
            callRatingDisposable?.dispose()
        }
    }

    companion object {
        private const val TAG = "[COMM][CallRating]"
    }
}