package io.agora.avc.manager.splite

import io.agora.avc.BuildConfig
import io.agora.avc.bo.*
import io.agora.avc.utils.EncryptUtils
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.SPUtils
import javax.inject.Inject

/***
 * SP 相关读写操作
 */
class SPLiteImp @Inject constructor() : SPLite {

    private fun getSPUtils() = SPUtils.getInstance(SP_NAME)

    override fun migrate() {
        val token = getSPUtils().getString(TOKEN, "")
        if (token.isNotEmpty() && getUserInfo().isEmpty()) {
            setUserInfo(token)
        }
    }

    override fun getInnerInfo(): InnerInfo? {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson(tokenJson, UserInfo::class.java)
        return userToken?.innerInfo
    }

    @Deprecated(
        "move to UserRepository",
        ReplaceWith("[UserRepository.saveInnerInfo() instead]"),
        DeprecationLevel.ERROR
    )
    override fun setInnerInfo(value: InnerInfo?) {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        userToken?.let {
            it.innerInfo = value
            setUserInfo(GsonUtils.toJson(it))
        }
    }

    override fun getLastVersion(): String {
        return getSPUtils().getString(KEY_LAST_VERSION)
    }

    override fun setLastVersion(value: String) {
        getSPUtils().put(KEY_LAST_VERSION, value)
    }

    override fun getNotice(): String {
        return getSPUtils().getString(KEY_NOTICE)
    }

    override fun setNotice(value: String) {
        getSPUtils().put(KEY_NOTICE, value)
    }

    @Deprecated(
        "move to UserRepository",
        ReplaceWith("[UserRepository.getUserName() instead]"),
        DeprecationLevel.ERROR
    )
    override fun getUserName(): String {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        return userToken?.name ?: ""
    }

    override fun setUserName(value: String) {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        userToken?.let {
            it.name = value
            setUserInfo(GsonUtils.toJson(it))
        }
    }

    @Deprecated(
        "move to UserRepository",
        ReplaceWith("[UserRepository.getPortraitId() instead]"),
        DeprecationLevel.ERROR
    )
    override fun getPortraitId(): String {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        return userToken?.portraitId ?: "1"
    }

    @Deprecated(
        "move to UserRepository",
        ReplaceWith("[UserRepository.savePortraitId() instead]"),
        DeprecationLevel.ERROR
    )
    override fun setPortraitId(value: String) {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        userToken?.let {
            it.portraitId = value
            setUserInfo(GsonUtils.toJson(it))
        }
    }

    @Deprecated(
        "move to UserRepository",
        ReplaceWith("[UserRepository.getTokenLocal() instead]"),
        DeprecationLevel.WARNING
    )
    override fun getAccessToken(): String {
        val tokenJson = getUserInfo()
        val userToken = GsonUtils.fromJson<UserInfo>(tokenJson, UserInfo::class.java)
        return userToken?.token ?: ""
    }

    @Deprecated(
        "remove",
        level = DeprecationLevel.ERROR
    )
    override fun getRtmToken(): String {
        return getSPUtils().getString(KEY_RTM_TOKEN)
    }

    @Deprecated(
        "remove",
        level = DeprecationLevel.ERROR
    )
    override fun setRtmToken(value: String) {
        getSPUtils().put(KEY_RTM_TOKEN, value)
    }

    override fun getUserInfo(): String {
        return EncryptUtils.decrypt(getSPUtils().getString(TOKEN, "")) ?: ""
    }

    override fun setUserInfo(value: String) {
        getSPUtils().put(TOKEN, EncryptUtils.encrypt(value))
    }

    override fun getAudioSetting(): Boolean {
        return getSPUtils().getBoolean(ME_AUDIO_ON, true)
    }

    override fun setAudioSetting(value: Boolean) {
        getSPUtils().put(ME_AUDIO_ON, value)
    }

    override fun getVideoSetting(): Boolean {
        return getSPUtils().getBoolean(ME_VIDEO_ON, true)
    }

    override fun setVideoSetting(value: Boolean) {
        getSPUtils().put(ME_VIDEO_ON, value)
    }

    override fun getDarkTheme(): Int {
        return getSPUtils().getInt(ME_DARK_THEME)
    }

    override fun setDarkTheme(value: Int) {
        getSPUtils().put(ME_DARK_THEME, value)
    }

    override fun getDevelopMode(): Boolean {
        return getSPUtils().getBoolean(KEY_DEVELOP_MODE, false)
    }

    override fun setDevelopMode(value: Boolean) {
        getSPUtils().put(KEY_DEVELOP_MODE, value)
    }

    override fun getShowWelcome(): Boolean {
        return getSPUtils().getBoolean(SHOW_WELCOME_PAGE, false)
    }

    override fun setShowWelcome(value: Boolean) {
        getSPUtils().put(SHOW_WELCOME_PAGE, value)
    }

    override fun privacyShowed(): Boolean {
        return getSPUtils().getBoolean(VERIFY_USAGE_TERMS, false)
    }

    override fun setShowPrivacy(value: Boolean) {
        getSPUtils().put(VERIFY_USAGE_TERMS, value)
    }

    override fun getResolutionOption(): Int {
        return getSPUtils().getInt(RESOLUTION_OPT, RESOLUTION_MEDIUM)
    }

    override fun setResolutionOption(value: Int) {
        getSPUtils().put(RESOLUTION_OPT, value)
    }

    override fun getRoomSecure(): Boolean {
        return getSPUtils().getBoolean(RTC_SECURE, true)
    }

    override fun setRoomSecure(value: Boolean) {
        getSPUtils().put(RTC_SECURE, value, true)
    }

    override fun getPictureMode(): Boolean {
        return getSPUtils().getBoolean(SUPPORT_PIP, true)
    }

    override fun setPictureMode(value: Boolean) {
        getSPUtils().put(SUPPORT_PIP, value)
    }

    override fun getRtcParameters(): Set<String> {
        return getSPUtils().getStringSet(KEY_RTC_PARAMETERS) ?: HashSet()
    }

    override fun setRtcParameters(value: Set<String>) {
        getSPUtils().put(KEY_RTC_PARAMETERS, value)
    }

    override fun isTestServer(): Boolean {
        return getSPUtils().getBoolean(TEST_SERVER, BuildConfig.DEBUG)
    }

    override fun setTestServer(value: Boolean) {
        getSPUtils().put(TEST_SERVER, value, true)
    }

    override fun getRtcMediaCopy(): Boolean {
        return getSPUtils().getBoolean(RTC_MEDIA_COPY, false)
    }

    override fun setRtcMediaCopy(value: Boolean) {
        getSPUtils().put(RTC_MEDIA_COPY, value)
    }

    override fun getAvcGuid(): String {
        return getSPUtils().getString(KEY_AVC_GUID)
    }

    override fun setAvcGuid(value: String) {
        getSPUtils().put(KEY_AVC_GUID, value)
    }

    override fun clear() {
        clearToken()
        clearAppTips()
    }

    override fun clearToken() {
        getSPUtils().remove(TOKEN, true)
    }

    override fun clearAppTips() {
        getSPUtils().remove(KEY_APP_TIPS, true)
    }

    override fun setRoomMode(mode: Int) {
        getSPUtils().put(KEY_ROOM_MODE, mode)
    }

    override fun getRoomMode(): Int {
        return getSPUtils().getInt(KEY_ROOM_MODE, RoomMode.NORMAL.value)
    }

    override fun saveAppConfig(config: String) {
        getSPUtils().put(KEY_APP_CONFIG, config)
    }

    override fun getAppConfig(): AppConfig? {
        val data = getSPUtils().getString(KEY_APP_CONFIG)
        return GsonUtils.fromJson(data, AppConfig::class.java)
    }

    override fun saveDevConfig(config: String) {
        getSPUtils().put(KEY_DEV_CONFIG, config)
    }

    override fun getDevConfig(): DevConfig? {
        val data = getSPUtils().getString(KEY_DEV_CONFIG)
        return GsonUtils.fromJson(data, DevConfig::class.java)
    }

    override fun saveOpenNotificationGuideTime(time: Long) {
        getSPUtils().put(KEY_OPEN_NOTIFICATION_GUIDE_TIME, time)
    }

    override fun getOpenNotificationGuideTime() =
        getSPUtils().getLong(KEY_OPEN_NOTIFICATION_GUIDE_TIME)

    override fun getDeviceUUID(): String? {
        return getSPUtils().getString(KEY_DEVICE_UUID)
    }

    override fun saveDeviceUUID(uuid: String) {
        getSPUtils().put(KEY_DEVICE_UUID, uuid)
    }

    override fun saveBannerNotificationGuideTime(time: Long) {
        getSPUtils().put(KEY_BANNER_NOTIFICATION_GUIDE_TIME, time)
    }

    override fun getBannerNotificationGuideTime(): Long? {
        return getSPUtils().getLong(KEY_BANNER_NOTIFICATION_GUIDE_TIME)
    }

    override fun getAppTips(): AppTips? {
        val value = getSPUtils().getString(KEY_APP_TIPS, "")
        return GsonUtils.fromJson(value, AppTips::class.java)
    }

    override fun setAppTips(appTips: AppTips?) {
        getSPUtils().put(KEY_APP_TIPS, GsonUtils.toJson(appTips))
    }

    companion object {
        private const val SP_NAME = "io.agora.vcall_preferences"

        /**
         * Key define
         */
        private const val RESOLUTION_MEDIUM = 1
        const val VERSION = "version"
        const val MSG_TIMEOUT = "MSG_TIMEOUT"
        const val REQUEST_TIMEOUT = "REQUEST_TIMEOUT"
        private const val TOKEN = "token"
        private const val RTC_SECURE = "RTC_SECURE"
        private const val ME_VIDEO_ON = "ME_VIDEO_ON"
        private const val ME_AUDIO_ON = "ME_AUDIO_ON"
        private const val ME_DARK_THEME = "ME_DARK_THEME"
        private const val RESOLUTION_OPT = "RESOLUTION_OPT"
        private const val VERIFY_USAGE_TERMS = "verify_usage_terms"
        private const val SHOW_WELCOME_PAGE = "SHOW_WELCOME_PAGE"
        private const val SUPPORT_PIP = "SUPPORT_PIP"
        private const val TEST_SERVER = "TEST_SERVER"
        private const val RTC_MEDIA_COPY = "RTC_MEDIA_COPY"
        private const val KEY_RTC_PARAMETERS = "KEY_RTC_PARAMETERS"
        private const val KEY_RTM_TOKEN = "KEY_RTM_TOKEN"
        private const val KEY_DEVELOP_MODE = "KEY_DEVELOP_MODE"
        private const val KEY_DEVELOP_OPTION = "KEY_DEVELOP_OPTION"
        private const val KEY_LAST_VERSION = "KEY_LAST_VERSION"
        private const val KEY_NOTICE = "KEY_NOTICE"
        private const val KEY_AVC_GUID = "KEY_AVC_GUID"
        private const val KEY_APP_TIPS = "APP_TIPS"
        private const val KEY_ROOM_MODE = "ROOM_MODE"
        private const val KEY_APP_CONFIG = "APP_CONFIG"
        private const val KEY_DEV_CONFIG = "DEV_CONFIG"
        private const val KEY_DEVICE_UUID = "DEVICE_UUID"
        private const val KEY_OPEN_NOTIFICATION_GUIDE_TIME = "OPEN_NOTIFICATION_GUIDE_TIME"
        private const val KEY_BANNER_NOTIFICATION_GUIDE_TIME = "OPEN_BANNER_NOTIFICATION_GUIDE_TIME"
    }
}
