package io.agora.avc.net.interceptor

import io.agora.logger.Logger
import okhttp3.*
import okhttp3.internal.http.promisesBody
import okio.Buffer
import okio.BufferedSource
import java.io.EOFException
import java.io.IOException
import java.nio.charset.Charset
import java.nio.charset.UnsupportedCharsetException
import java.util.*
import java.util.concurrent.atomic.AtomicInteger

class LogInterceptor : Interceptor {

    @Throws(IOException::class)
    override fun intercept(chain: Interceptor.Chain): Response {
        val request = chain.request()
        val url = request.url
        var path = url.encodedPath
        if (path.isNotEmpty()) {
            path = path.substring(1)
        }
        val id = ID_GENERATOR.incrementAndGet()
        val requestPrefix = "[$id request]"
        val requestBuffer = StringBuffer()
        requestBuffer.append(requestPrefix)
        requestBuffer.append(String.format("%1\$s->%2\$s", request.method, request.url))
            .append("\n")
        if (request.headers != null) {
            requestBuffer.append("Headers:" + request.headers)
        }
        if (bodyEncoded(request.headers)) {
            requestBuffer.append("END ${request.method} (encoded body omitted)")
        } else if (!logPath.contains(path) && request.body != null) {
            requestBuffer.append("RequestBody:" + bodyToString(requestPrefix, request.body))
        } else {
            requestBuffer.append("END ${request.method}")
        }
        Logger.i(TAG, requestBuffer.toString())
        val responsePrefix = "[$id response]"
        var response: Response?
        try {
            response = chain.proceed(chain.request())
        } catch (e: Exception) {
            Logger.e(TAG, "$responsePrefix HTTP FAILED:", e)
            throw e
        }
        val responseBody = response?.body
        val contentLength = responseBody?.contentLength()
        if (!response.promisesBody()) {
            Logger.i(TAG, "$responsePrefix END HTTP")
        } else if (bodyEncoded(response.headers)) {
            Logger.i(TAG, "$responsePrefix END HTTP (encoded body omitted)")
        } else {
            val source: BufferedSource? = responseBody?.source()
            source?.request(Long.MAX_VALUE) // Buffer the entire body.
            val sourceBuffer = source?.buffer()
            var charset = UTF8
            val contentType: MediaType? = responseBody?.contentType()
            if (contentType != null) {
                try {
                    charset = contentType.charset(UTF8)
                } catch (e: UnsupportedCharsetException) {
                    Logger.e(TAG, "$responsePrefix Couldn't decode the response body")
                    return response
                }
            }
            if (sourceBuffer != null && !isPlaintext(sourceBuffer)) {
                Logger.e(TAG, "$responsePrefix byte body omitted")
                return response
            }

            if (contentLength != 0L) {
                val bufferString = sourceBuffer?.clone()?.readString(charset)
                Logger.i(TAG, "$responsePrefix END HTTP $bufferString")
            } else {
                Logger.i(TAG, "$responsePrefix END HTTP (${sourceBuffer?.size}-byte body)")
            }
        }
        return response
    }

    private fun isPlaintext(buffer: Buffer): Boolean {
        return try {
            val prefix = Buffer()
            val byteCount = if (buffer.size < 64) buffer.size else 64
            buffer.copyTo(prefix, 0, byteCount)
            for (i in 0..15) {
                if (prefix.exhausted()) {
                    break
                }
                val codePoint = prefix.readUtf8CodePoint()
                if (Character.isISOControl(codePoint) && !Character.isWhitespace(codePoint)) {
                    return false
                }
            }
            true
        } catch (e: EOFException) {
            false // Truncated UTF-8 sequence.
        }
    }

    private fun bodyToString(prefix: String, request: RequestBody?): String? {
        if (request != null) {
            try {
                val copy: RequestBody = request
                val buffer = Buffer()
                copy.writeTo(buffer)
                return buffer.readUtf8()
            } catch (e: IOException) {
                Logger.e(TAG, "$prefix Failed to convert request body to string")
            }
        }
        return null
    }

    private fun bodyEncoded(headers: Headers): Boolean {
        val contentEncoding = headers["Content-Encoding"]
        return contentEncoding != null && !contentEncoding.equals("identity", ignoreCase = true)
    }

    companion object {
        private const val TAG = "[Comm][Http]"
        private val ID_GENERATOR = AtomicInteger(0)
        private val UTF8 = Charset.forName("UTF-8")
        private val logPath = ArrayList<String>()

        init {
            logPath.add("feedback/uploadFeedback")
            logPath.add("usrservice/v1/account/portrait")
            logPath.add("usrservice/v1/account/generalLog")
            logPath.add("usrservice/v1/account/generalIssue")
        }
    }
}