package io.agora.avc.push.xiaomi

import android.content.Context
import android.content.Intent
import androidx.core.os.bundleOf
import com.agora.valoran.bean.IncomingData
import com.xiaomi.mipush.sdk.*
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.push.*
import io.agora.avc.utils.JsonUtils
import io.agora.logger.Logger
import org.json.JSONObject

const val KEY_COMMAND = "command"
const val KEY_REG_ID = "reg_id"
const val KEY_DATA = "key_data"
const val COMMAND_INCOMING_CALL = "command_incoming_call"

class MiPushMessageReceiver : PushMessageReceiver() {

    private var regId: String? = null

    /**
     * Used to receive the transparent message sent by the server
     */
    override fun onReceivePassThroughMessage(ctx: Context?, message: MiPushMessage?) {
        super.onReceivePassThroughMessage(ctx, message)
    }

    /**
     * Used to receive notification bar messages from the server
     * (triggered when the user clicks on the notification bar)
     */
    override fun onNotificationMessageClicked(ctx: Context?, message: MiPushMessage?) {
        super.onNotificationMessageClicked(ctx, message)
        Logger.i(TAG, "message clicked:${message?.content}")
        val data = message?.content
        val obj = JSONObject(data)
        val avatar = JsonUtils.getString(obj, KEY_AVATAR)
        val sourceName = JsonUtils.getString(obj, KEY_INVITER)
        val rid = JsonUtils.getString(obj, KEY_RID)
        val pwd = JsonUtils.getString(obj, KEY_PWD)
        val requestId = JsonUtils.getString(obj, KEY_REQUEST_ID)
        val ticket = JsonUtils.getString(obj, KEY_TICKET)

        if (rid.isNullOrEmpty() || requestId.isNullOrEmpty() || ticket.isNullOrEmpty()) {
            Logger.e(TAG, "invalid data, rid:$rid, requestId:$requestId, ticket:$ticket")
            return
        }

        val incomingData = IncomingData().apply {
            this.rid = rid
            this.pwd = pwd
            this.ticket = ticket
            this.inviterName = sourceName
            this.inviterAvatar = avatar
            this.requestId = requestId
        }
        ctx?.startActivity(Intent(ctx, AgoraActivity::class.java).apply {
            action = ACTION_PUSH
            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            putExtras(
                bundleOf(
                    KEY_RID to incomingData.rid,
                    KEY_PWD to incomingData.pwd,
                    KEY_REQUEST_ID to incomingData.requestId,
                    KEY_TICKET to incomingData.ticket,
                    KEY_INVITER to incomingData.inviterName,
                    KEY_AVATAR to incomingData.inviterAvatar,
                )
            )
        })
    }

    /**
     * Used to receive notification bar messages from the server
     */
    override fun onNotificationMessageArrived(ctx: Context?, message: MiPushMessage?) {
        super.onNotificationMessageArrived(ctx, message)
        Logger.i(TAG, "notification message arrived:${message?.content}")

    }

    /**
     * Used to accept the response returned after the client sends a registration command message to the server
     */
    override fun onReceiveRegisterResult(ctx: Context?, message: MiPushCommandMessage?) {
        super.onReceiveRegisterResult(ctx, message)
        Logger.i(
            TAG,
            "receive register result, command:${message?.command}, result:${message?.resultCode}, reason:${message?.reason}"
        )
        //start service to report token
        val command = message?.command ?: return
        val resultCode = message?.resultCode?.toInt()
        val arguments = message.commandArguments
        val cmdArg1 = if (arguments != null && arguments.size > 0) arguments[0] else null
        val cmdArg2 = if (arguments != null && arguments.size > 1) arguments[1] else null
        when (command) {
            MiPushClient.COMMAND_REGISTER -> {
                if (resultCode == ErrorCode.SUCCESS) {
                    regId = cmdArg1
                    ctx?.startService(Intent(ctx, MiPushMessageService::class.java).apply {
                        putExtra(KEY_COMMAND, command)
                        putExtra(KEY_REG_ID, cmdArg1)
                    })
                }
            }
            else -> {
            }
        }
    }

    /**
     * Used to receive the response returned by the client after sending a command message to the server
     */
    override fun onCommandResult(ctx: Context?, message: MiPushCommandMessage?) {
        super.onCommandResult(ctx, message)
        Logger.i(
            TAG,
            "on command result, command:${message?.command}, result:${message?.resultCode}, reason:${message?.reason}"
        )
    }

    override fun onRequirePermissions(p0: Context?, p1: Array<out String>?) {
        super.onRequirePermissions(p0, p1)
        val sb = StringBuffer()
        p1?.forEach { permission ->
            sb.append(permission)
        }
        Logger.e(TAG, "Lack of permissions:$sb")
    }

    companion object {
        private const val TAG = "MiPushMessageReceiver"
    }
}