package io.agora.avc.repository.impl

import android.app.Application
import com.agora.valoran.Constants
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.bo.valoran.RoomInfo
import io.agora.avc.extensions.copy
import io.agora.avc.extensions.isMySelf
import io.agora.avc.extensions.isMyShareScreen
import io.agora.avc.net.api.ApiService
import io.agora.avc.repository.RoomRepository
import io.agora.frame.base.BaseModel
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import java.util.concurrent.CopyOnWriteArrayList
import javax.inject.Inject

/**
 * Not thread safe
 */
class RoomRepositoryImpl @Inject constructor(
    dataRepository: IDataRepository?,
    private val application: Application
) : BaseModel(dataRepository), RoomRepository {

    private val roomLock = Any()
    private var room: Room? = null

    private var majorMedia: ARoomUser? = null
    private var cloudRecordInfo: CloudRecordInfo? = null
    private var assistantInfo: AssistantInfo? = null
    private var assistState: Int = Constants.AssistState.NONE.value
    private var audioDumpingUsers: List<ARoomUser>? = null

    private val roomUsers = CopyOnWriteArrayList<ARoomUser>()
    private val mediaRoomUsers = CopyOnWriteArrayList<ARoomUser>()

    override fun createRoom(name: String, password: String?, mode: Int, resolution: Int): Room {
        Logger.i(TAG, "create room, name:${name}, password:${password}")
        synchronized(roomLock) {
            val room = Room(name = name, pwd = password, mode = mode, resolution = resolution)
            this.room = room
            return room.copy()
        }
    }

    override fun releaseRoom() {
        Logger.i(TAG, "release room, name:${room?.name}")
        synchronized(roomLock) {
            saveCloudRecordInfo(null)
            saveAudioDumpingUser(null)
            saveAssistantInfo(null)
            saveAssistState(Constants.AssistState.NONE.value)
            clearRoomUserList()
            clearMediaList()
            room = null
        }
    }

    /**
     * update room info, need not update room name & room pwd
     */
    override fun updateRoomInfo(roomInfo: RoomInfo): Room? {
        room?.let {
            synchronized(roomLock) {
                it.audioState = roomInfo.audioState
                it.videoState = roomInfo.videoState
                it.audioMuted = roomInfo.audioMuted
                it.hostUid = roomInfo.hostUid
                it.hostName = roomInfo.hostName
                it.hostThirdPartyName = roomInfo.hostThirdPartyName
                it.hostThirdPartyAlias = roomInfo.hostThirdPartyAlias
                it.elapse = roomInfo.elapsedTime
            }
        }
        return room?.copy()
    }

    override fun rtmJoinedBefore(joined: Boolean) {
        room?.rtmJoinedBefore = joined
    }

    override fun getRoom(): Room? {
        return room
    }

    override fun getRoomUserList(): CopyOnWriteArrayList<ARoomUser> {
        return CopyOnWriteArrayList(roomUsers)
    }

    override fun getOppositeSnapshot(localUser: LocalUser): CopyOnWriteArrayList<ARoomUser> {
        return CopyOnWriteArrayList(roomUsers.filter {
            !it.isMySelf() || it.isMyShareScreen(localUser)
        })
    }

    override fun getCloudRecordInfo(): CloudRecordInfo? {
        return cloudRecordInfo?.copy()
    }

    override fun getAssistantInfo(): AssistantInfo? {
        return assistantInfo?.copy()
    }

    override fun queryAssistState(): Int {
        return assistState
    }

    override fun getAudioDumpingUser(): List<ARoomUser>? {
        return audioDumpingUsers
    }

    override fun saveAudioDumpingUser(users: List<ARoomUser>?) {
        audioDumpingUsers = users
    }

    override fun saveCloudRecordInfo(info: CloudRecordInfo?) {
        Logger.i(TAG, "save cloud recording user:${info}")
        cloudRecordInfo = info
    }

    override fun saveAssistantInfo(info: AssistantInfo?) {
        Logger.i(TAG, "save assistant user:${info}")
        assistantInfo = info
    }

    override fun saveAssistState(state: Int) {
        assistState = state
    }

    override fun moveRoomUser(from: Int, to: Int, user: ARoomUser) {
        if (from in 0 until roomUsers.size) {
            roomUsers.removeAt(from)
            roomUsers.add(to, user)
        }
    }

    override fun changeRoomUserListItem(position: Int, user: ARoomUser) {
        if (position >= 0 && position < roomUsers.size) {
            roomUsers[position] = user
        } else {
            Logger.e(
                TAG,
                "change room user IndexOutOfBoundsException, index:$position, size:${roomUsers.size}",
            )
        }
    }

    override fun setupRoomUserList(users: List<ARoomUser>) {
        roomUsers.clear()
        roomUsers.addAll(users)
    }

    override fun addRoomUsers(users: List<ARoomUser>) {
        roomUsers.addAll(users)
    }

    override fun addRoomUser(user: ARoomUser) {
        roomUsers.add(user)
    }

    override fun addRoomUsers(position: Int, users: List<ARoomUser>) {
        if (position in 0..roomUsers.size) {
            roomUsers.addAll(position, users)
        } else {
            Logger.e(
                TAG,
                "add room user list IndexOutOfBoundsException, index:$position, size:${roomUsers.size}"
            )
        }
    }

    override fun addRoomUser(position: Int, user: ARoomUser) {
        if (position in 0..roomUsers.size) {
            roomUsers.add(position, user)
        } else {
            Logger.e(
                TAG,
                "add room user IndexOutOfBoundsException, index:$position, size:${roomUsers.size}"
            )
        }
    }

    override fun removeRoomUserAt(position: Int) {
        if (position in 0 until roomUsers.size) {
            roomUsers.removeAt(position)
        } else {
            Logger.e(
                TAG,
                "remove room user at IndexOutOfBoundsException, index:$position, size:${roomUsers.size}"
            )
        }
    }

    override fun removeRoomUser(uid: String) {
        roomUsers
            .find { it.uid == uid }
            .let {
                roomUsers.remove(it)
            }
    }

    override fun removeRoomUser(streamId: Int) {
        roomUsers
            .find { it.streamId == streamId }
            .let {
                roomUsers.remove(it)
            }
    }

    override fun removeRoomUser(user: ARoomUser) {
        roomUsers.remove(user)
    }

    override fun clearRoomUserList() {
        Logger.i(TAG, "clear room user list, size:${roomUsers.size}")
        roomUsers.clear()
    }

    override fun getMajorMedia(): ARoomUser? {
        return if (majorMedia is LocalUser) {
            (majorMedia as? LocalUser)?.copy()
        } else {
            majorMedia?.copy()
        }
    }

    override fun saveMajorMedia(user: ARoomUser) {
        majorMedia = user
    }

    override fun getMediaList(): CopyOnWriteArrayList<ARoomUser> {
        return CopyOnWriteArrayList(mediaRoomUsers)
    }

    override fun moveMediaItem(from: Int, to: Int, user: ARoomUser) {
        if (from in 0 until mediaRoomUsers.size) {
            mediaRoomUsers.removeAt(from)
            mediaRoomUsers.add(to, user)
        }
    }

    override fun changeMediaListItem(position: Int, user: ARoomUser) {
        if (position in 0 until mediaRoomUsers.size) {
            mediaRoomUsers[position] = user
        } else {
            Logger.e(
                TAG,
                "change media room user IndexOutOfBoundsException, index:$position, size:${mediaRoomUsers.size}"
            )
        }
    }

    override fun addMediaList(users: List<ARoomUser>) {
        mediaRoomUsers.addAll(users)
    }

    override fun addMediaList(position: Int, users: List<ARoomUser>) {
        if (position in 0..mediaRoomUsers.size) {
            mediaRoomUsers.addAll(position, users)
        } else {
            Logger.e(
                TAG,
                "add media list IndexOutOfBoundsException, index: $position, size: ${mediaRoomUsers.size}"
            )
        }
    }

    override fun addMediaItem(user: ARoomUser) {
        mediaRoomUsers.add(user)
    }

    override fun addMediaItem(position: Int, user: ARoomUser) {
        if (position in 0..mediaRoomUsers.size) {
            mediaRoomUsers.add(position, user)
        } else {
            Logger.e(
                TAG,
                "add media item IndexOutOfBoundsException, index:$position, size:${mediaRoomUsers.size}"
            )
        }
    }

    override fun removeMediaAt(position: Int) {
        if (position in 0 until mediaRoomUsers.size) {
            mediaRoomUsers.removeAt(position)
        } else {
            Logger.e(
                TAG,
                "remove media at IndexOutOfBoundsException, index:$position, size:${mediaRoomUsers.size}"
            )
        }
    }

    override fun removeMediaItem(uid: String) {
        mediaRoomUsers
            .find { it.uid == uid }
            .let {
                mediaRoomUsers.remove(it)
            }
    }

    override fun removeMediaItem(streamId: Int) {
        mediaRoomUsers
            .find { it.streamId == streamId }
            .let {
                mediaRoomUsers.remove(it)
            }
    }

    override fun removeMediaItem(user: ARoomUser) {
        mediaRoomUsers.remove(user)
    }

    override fun clearMediaList() {
        Logger.i(TAG, "clear media list, size:${mediaRoomUsers.size}")
        mediaRoomUsers.clear()
    }

    override fun findShareStream(): ARoomUser? {
        return mediaRoomUsers.find { it.isShareStream() }.apply {
            Logger.i(TAG, "find share stream user, user:${this}")
        }
    }

    override fun findAudioDumpingUser(): ARoomUser? {
        return roomUsers.find { it.isAudioDumping() }.apply {
            Logger.i(TAG, "find audio dumping user, user:${this}")
        }
    }

    override fun findParentByStreamId(uid: Int): ARoomUser? {
        return mediaRoomUsers.find { it.shareId == uid }.apply {
            Logger.i(TAG, "find parent by stream id, user:${this}")
        }
    }

    override fun queryOtherPublishers(): List<ARoomUser> {
        val mediaUsers = mediaRoomUsers.filter {
            it !is LocalUser && it.videoState
        }.apply {
            Logger.i(TAG, "query other publishers, user size:${this.size}")
        }.toMutableList()
        if (majorMedia !is LocalUser && majorMedia?.videoState == true) {
            mediaUsers.add(majorMedia)
        }
        return mediaUsers
    }

    override fun queryInnerMeetingInfo(linkId: String): Observable<InnerMeeting> {
        return getRetrofitService(ApiService::class.java)
            .queryInnerMeetingInfo(linkId)
    }

    companion object {
        private const val TAG = "[Repository][Room]"
    }
}