package io.agora.avc.repository.impl

import android.app.Application
import com.google.gson.JsonObject
import com.google.gson.JsonSyntaxException
import io.agora.avc.MyApplication
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.config.DEFAULT_AVATAR_ID
import io.agora.avc.manager.splite.SPLite
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.repository.UserRepository
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.GsonUtils
import io.agora.frame.base.BaseModel
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import javax.inject.Inject

class UserRepositoryImpl @Inject constructor(
    private val application: Application,
    private val dataRepository: IDataRepository,
    private val spLite: SPLite
) : BaseModel(dataRepository), UserRepository {

    private var localUser: LocalUser? = null
    override var isLocalUserApplyingAssistant: Boolean = false

    @Synchronized
    override fun resetLocalUser() {
        Logger.i(TAG, "reset local user")
        localUser?.isHost = false
        localUser?.isInterrupt = false
        localUser?.isCloudRecording = false
        localUser?.inviteBy = null

        localUser?.audioState = false
        localUser?.isAudioPending = false
        localUser?.videoState = false
        localUser?.isVideoPending = false
        localUser?.isSpeaking = false
        localUser?.isAttendee = false
        localUser?.quality = 0
        localUser?.shareId = 0
        localUser?.parentStreamId = 0

        isLocalUserApplyingAssistant = false
    }

    @Synchronized
    override fun updateLocalUserBy(user: ARoomUser) {
        localUser?.update(user)
        if (application is MyApplication) {
            application.appContainer.localUser = localUser?.copy()
        }
    }

    override fun getLocalUser(platform: String, deviceName: String): Observable<LocalUser> {
        return Observable
            .concat(queryLocalUserFromLocal(), queryLocalUserFromRemote())
            .firstElement()
            .flatMapObservable { _userInfo ->
                val localUser = this.localUser ?: LocalUser()
                localUser.updateUserInfo(_userInfo)
                saveLocalUser(localUser)
                Observable.just(localUser)
            }
    }

    private fun queryLocalUserFromRemote(
        platform: String = DeviceUtils.getPlatform(),
        deviceName: String = DeviceUtils.getModel()
    ): Observable<UserInfo> {
        return getRetrofitService(ApiService::class.java)
            .getToken(TokenReqBody(0, 1, platform, deviceName))
            .doOnNext {
                Logger.i(TAG, "query local user from remote")
                if (localUser?.name != null && localUser?.name != it.name) {
                    it.name = localUser?.name ?: ""
                    it.portraitId = localUser?.portraitId ?: DEFAULT_AVATAR_ID
                    updateUserAccountInfo(
                        localUser?.name,
                        localUser?.portraitId
                    ).subscribe()
                }

                saveUserInfo(it)
            }
    }

    private fun queryLocalUserFromLocal(): Observable<UserInfo?> {
        return Observable
            .create { _emitter ->
                val userInfo = getUserInfo()

                if (userInfo == null) {
                    _emitter.onComplete()
                } else {
                    _emitter.onNext(userInfo)
                }
            }
    }

    @Synchronized
    override fun saveUserInfo(userInfo: UserInfo) {
        Logger.i(TAG, "save user info, user info:$userInfo")
        localUser?.updateUserInfo(userInfo)

        if (application is MyApplication) {
            Logger.i(TAG, "refresh app container local user")
            application.appContainer.localUser = localUser?.copy()
        }

        spLite.setUserInfo(GsonUtils.toJson(userInfo))
    }

    @Synchronized
    override fun updateUserInfo(userInfo: UserInfo) {
        Logger.i(TAG, "update user info, user info:$userInfo")
        localUser?.updateUserInfo(userInfo)
        if (application is MyApplication) {
            Logger.i(TAG, "refresh app container local user")
            application.appContainer.localUser = localUser?.copy()
        }
    }

    @Synchronized
    override fun saveInnerInfo(value: InnerInfo?) {
        getUserInfo()?.let {
            Logger.i(TAG, "save inner info, inner info:$value")
            it.innerInfo = value
            saveUserInfo(it)
        }
    }

    @Synchronized
    override fun updateInnerInfo(value: InnerInfo) {
        Logger.i(TAG, "update inner info, inner info:$value")
        getUserInfo()?.let {
            it.innerInfo = value
            localUser?.updateUserInfo(it)
            if (application is MyApplication) {
                Logger.i(TAG, "refresh app container local user")
                application.appContainer.localUser = localUser?.copy()
            }
        }
    }

    @Synchronized
    override fun removeInnerInfo() {
        localUser?.removeInnerInfo()
        saveInnerInfo(null)
    }

    @Synchronized
    override fun saveUserName(value: String) {
        Logger.i(TAG, "save user name, username:$value")
        localUser?.name = value

        if (application is MyApplication) {
            Logger.i(TAG, "refresh app container local user")
            application.appContainer.localUser = localUser?.copy()
        }

        spLite.setUserName(value)
    }

    override fun updateUserName(userName: String): Observable<BaseNo> {
        Logger.i(TAG, "update user name, userName:$userName")
        return updateUserAccountInfo(userName = userName, null)
    }

    override fun login(
        token: String,
        source: Int,
        platform: String,
        deviceName: String
    ): Observable<InnerSession> {
        val data = JsonObject()
        data.addProperty("code", token)
        data.addProperty("source", source)
        data.addProperty("platform", platform)
        data.addProperty("deviceName", deviceName)
        return getRetrofitService(ApiService::class.java)
            .queryInnerSession(data)
    }

    private fun updateUserAccountInfo(userName: String?, portraitId: String?): Observable<BaseNo> {
        return getRetrofitService(ApiService::class.java)
            .updateUsrInfo(UserUpdateBody(name = userName, portraitId = portraitId))
            .doOnNext {
                if (it?.success == true) {
                    userName?.let {
                        saveUserName(userName)
                    }
                }
            }
    }

    @Synchronized
    private fun saveLocalUser(localUser: LocalUser) {
        Logger.i(TAG, "save local user:$localUser")
        this.localUser = localUser
        if (application is MyApplication) {
            Logger.i(TAG, "refresh app container local user")
            application.appContainer.localUser = localUser.copy()
        }
    }

    @Synchronized
    override fun getUserInfo(): UserInfo? {
        val tokenJson = spLite.getUserInfo()
        return try {
            GsonUtils.fromJson(tokenJson, UserInfo::class.java)
        } catch (e: JsonSyntaxException) {
            Logger.e(TAG, "get user info error, error:", e)
            null
        }
    }

    companion object {
        private const val TAG = "[Repository][User]"
    }
}
