package io.agora.avc.screenshare

import android.content.Intent
import android.content.res.Configuration
import android.os.Build
import androidx.annotation.RequiresApi
import io.agora.avc.MyApplication
import io.agora.avc.video.ExternalVideoInputManager
import io.agora.avc.video.IEngine
import io.agora.logger.LogToFile.getRtcScreenShareLogPath
import io.agora.rtc.mediaio.IVideoSource
import io.agora.rtc.video.VideoEncoderConfiguration

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
class ScreenShareEngine(
    private val application: MyApplication,
    private val intent: Intent,
    private val screenShareClient: ScreenShareClient
) : IEngine {
    private val channelId: String? = intent.getStringExtra(ScreenConstants.INTENT_RID)
    private val streamId: Int = intent.getIntExtra(ScreenConstants.INTENT_MEDIA_ID, -1)
    private val token: String? = intent.getStringExtra(ScreenConstants.INTENT_MEDIA_TOKEN)
    private val parentStreamId: Int = intent.getIntExtra(ScreenConstants.INTENT_PARENT_MEDIA_ID, -1)
    private val parentName: String = intent.getStringExtra(ScreenConstants.INTENT_NAME) ?: ""
    private val encryptionMode: Int =
        intent.getIntExtra(ScreenConstants.INTENT_ENCRYPTION_MODE, -1)
    private val encryptionKey: String =
        intent.getStringExtra(ScreenConstants.INTENT_ENCRYPTION_KEY) ?: ""
    private val encryptionSalt: String =
        intent.getStringExtra(ScreenConstants.INTENT_ENCRYPTION_SALT) ?: ""
    private val hasWatermark: Boolean =
        intent.getBooleanExtra(ScreenConstants.INTENT_HAS_WATERMARK, false)
    private val cname: String =
        intent.getStringExtra(ScreenConstants.INTENT_CNAME) ?: ""
    private val ips: ArrayList<String>? = intent.getStringArrayListExtra(ScreenConstants.INTENT_IPS)
    private val domain: String = intent.getStringExtra(ScreenConstants.INTENT_DOMAIN) ?: ""

    private val maxWidth = 1280
    private val maxHeight = 720
    private var width =
        intent.getIntExtra(ExternalVideoInputManager.FLAG_SCREEN_WIDTH, maxWidth)
    private var height =
        intent.getIntExtra(ExternalVideoInputManager.FLAG_SCREEN_HEIGHT, maxHeight)

    override fun prepare(videoSource: IVideoSource) {
        screenShareClient.prepare(application, videoSource, getRtcScreenShareLogPath())
        setVideoConfiguration(width, height, getCurrentOrientation())
        if (channelId == null) {
            throw KotlinNullPointerException("please check your channelId")
        }
        if (token != null) {
            screenShareClient.joinChannel(
                channelId,
                streamId,
                parentName,
                parentStreamId,
                token,
                encryptionMode,
                encryptionKey,
                encryptionSalt,
                cname,
                ips,
                domain,
                hasWatermark
            )
        } else {
            throw NullPointerException("rtc token is empty")
        }
    }

    private fun getCurrentOrientation(): Int {
        return application.resources.configuration.orientation
    }

    override fun onConfigurationChanged(orientationMode: Int) {
        setVideoConfiguration(width, height, orientationMode)
    }

    private fun setVideoConfiguration(inWidth: Int, inHeight: Int, orientationMode: Int) {
        var width: Int
        var height: Int
        if (inWidth > inHeight) {
            width = inWidth
            height = inHeight
        } else {
            width = inHeight
            height = inWidth
        }

        if (width > maxWidth || height > maxHeight) {
            if ((width.toFloat() / height) > (maxWidth.toFloat() / maxHeight)) {
                // width is too big, reduce by width
                height = (height.toFloat() / width * maxWidth).toInt()
                width = maxWidth
            } else {
                // height is too big, reduce by height
                width = (width.toFloat() / height * maxHeight).toInt()
                height = maxHeight
            }
        }

        val orientationMode = if (orientationMode == Configuration.ORIENTATION_LANDSCAPE) {
            VideoEncoderConfiguration.ORIENTATION_MODE.ORIENTATION_MODE_FIXED_LANDSCAPE
        } else {
            VideoEncoderConfiguration.ORIENTATION_MODE.ORIENTATION_MODE_FIXED_PORTRAIT
        }

        val videoEncoderConfiguration = VideoEncoderConfiguration(
            VideoEncoderConfiguration.VideoDimensions(width, height),
            VideoEncoderConfiguration.FRAME_RATE.FRAME_RATE_FPS_7,
            VideoEncoderConfiguration.STANDARD_BITRATE,
            orientationMode
        )
        screenShareClient.setVideoEncoderConfiguration(videoEncoderConfiguration)
    }

    override fun release() {
        screenShareClient.leaveRoom()
    }

    companion object {
        private const val TAG = "[Comm][ScreenShareEngine]"
    }

}