package io.agora.avc.screenshare

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.hardware.display.DisplayManager
import android.hardware.display.VirtualDisplay
import android.media.projection.MediaProjection
import android.media.projection.MediaProjectionManager
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.util.Size
import android.view.Surface
import androidx.annotation.RequiresApi
import io.agora.avc.video.GLThreadContext
import io.agora.avc.video.IExternalVideoInput
import io.agora.logger.Logger

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
class ScreenShareInput(
    private val context: Context,
    private val surfaceWidth: Int,
    private val surfaceHeight: Int,
    private val screenDpi: Int,
    frameRate: Int,
    data: Intent
) : IExternalVideoInput {
    private val frameInterval: Int = 1000 / frameRate
    private val intent: Intent = data
    private var mediaProjection: MediaProjection? = null
    private var virtualDisplay: VirtualDisplay? = null

    @Volatile
    private var stopped = false
    private var surface: Surface? = null

    private val mCallback: MediaProjection.Callback by lazy {
        object : MediaProjection.Callback() {
            override fun onStop() {
                Logger.e(TAG, "media projection onStop callback")
                if (isRunning) {
                    onResetMediaProject()
                }
            }
        }
    }

    @Synchronized
    override fun onVideoInitialized(target: Surface) {
        if (surface != target) {
            surface = target
        }
        val pm =
            context.getSystemService(Context.MEDIA_PROJECTION_SERVICE) as MediaProjectionManager
        try {
            mediaProjection = pm.getMediaProjection(Activity.RESULT_OK, intent)
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to create media projection", e)
            return
        }
        if (mediaProjection == null) {
            Logger.e(TAG, "Failed to create media projection")
            return
        }
        mediaProjection?.registerCallback(mCallback, Handler(Looper.getMainLooper()))
        virtualDisplay = mediaProjection?.createVirtualDisplay(
            VIRTUAL_DISPLAY_NAME,
            surfaceWidth,
            surfaceHeight,
            screenDpi,
            DisplayManager.VIRTUAL_DISPLAY_FLAG_AUTO_MIRROR,
            target,
            null,
            null
        )
    }

    @Synchronized
    override fun onVideoStopped(context: GLThreadContext) {
        Logger.w(TAG, "Stop screen sharing")
        stopped = true
        onRelease()
    }

    @Synchronized
    private fun onRelease() {
        virtualDisplay?.release()
        mediaProjection?.unregisterCallback(mCallback)
        mediaProjection?.stop()
    }

    @Synchronized
    override fun isRunning(): Boolean {
        return !stopped
    }

    override fun onFrameAvailable(
        context: GLThreadContext,
        textureId: Int,
        transform: FloatArray
    ) {
        // Screen sharing do not process or show local preview
    }

    override fun onGetFrameSize(): Size {
        return Size(surfaceWidth, surfaceHeight)
    }

    override fun timeToWait(): Int {
        return frameInterval
    }

    private fun onResetMediaProject() {
        try {
            onRelease()
            surface?.let {
                onVideoInitialized(it)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "on reset media project exception,due to dead thread!", e)
        }
    }

    companion object {
        private const val TAG = "[Comm][ScreenShare]"
        private const val VIRTUAL_DISPLAY_NAME = "agora screen share"
    }
}