package io.agora.avc.utils

import android.net.Uri
import android.util.Base64
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.logger.Logger

/***
 * Created by liqilin on 2019-10-17
 *
 */
object SchemeUtils {
    private const val TAG = "[Comm][Scheme]"

    const val SCHEME_ROOM = "room"
    const val DEBUG_SCHEME_ROOM = "avc_private_test/room"
    const val SECURE_ROOM = "secureroom"
    const val DEBUG_SECURE_ROOM = "avc_private_test/secureroom"
    private const val SCHEME_PARAM_RID = "rid"
    private const val SCHEME_PARAM_PWD = "pwd"
    private const val SCHEME_PARAM_CODE = "code"
    private const val SCHEME_PARAM_TYPE = "type"

    const val URI_SCHEME = "https"
    const val URI_HOST = "videocall.agora.io"
    private const val DEBUG_URI_SCHEME_HOST = "https://avc-source.sh3.agoralab.co"
    private const val RELEASE_URI_SCHEME_HOST = "https://avc-source.sh3.agoralab.co"

    private fun getUriScheme(): String {
        return if (SPLiteProxy.isTestServer()) DEBUG_URI_SCHEME_HOST else RELEASE_URI_SCHEME_HOST
    }

    fun getPathFromURI(uri: Uri): String? {
        val path = uri.path ?: return null
        if (path.isEmpty() || path.length <= 1) return path
        return path.substring(1)// remove backslash
    }

    fun checkNewMeeting(uri: Uri?): Array<String?>? {
        if (uri == null) return null
        val path = getPathFromURI(uri) ?: return null
        if (path == SCHEME_ROOM) {
            return arrayOf(getRidFromURI(uri), getPwdFromURI(uri))
        }
        return null
    }

    fun getRidFromURI(uri: Uri): String? {
        var rid = uri.getQueryParameter(SCHEME_PARAM_RID)
        try {
            rid?.let {
                rid = String(Base64.decode(it, Base64.URL_SAFE))
            }
        } catch (e: IllegalArgumentException) {
            Logger.e(TAG, "decode room name:$rid from uri failed")

            return null
        }
        return rid
    }

    fun getPwdFromURI(uri: Uri): String? {
        var pwd = uri.getQueryParameter(SCHEME_PARAM_PWD)
        try {
            pwd?.let {
                pwd = String(Base64.decode(it, Base64.URL_SAFE))
            }
        } catch (e: IllegalArgumentException) {
            Logger.e(TAG, "decode room pwd:$pwd from uri failed")

            return null
        }
        return pwd
    }

    fun getTypeFromURI(uri: Uri): Int? {
        val type = uri.getQueryParameter(SCHEME_PARAM_TYPE)
        try {
            return type?.toInt()
        } catch (e: Exception) {
            Logger.e(TAG, "decode room type:$type from uri failed")
        }
        return null
    }

    fun getCodeFromURI(uri: Uri): String? {
        return uri.getQueryParameter(SCHEME_PARAM_CODE)
    }

    fun generateRoomLink(rid: String, pwd: String?): String {
        val buffer = StringBuffer()
        buffer.append(getUriScheme())
            .append("/")
            .append(SCHEME_ROOM)
            .append("?")
            .append(SCHEME_PARAM_RID)
            .append("=")
            .append(String(Base64.encode(rid.toByteArray(), Base64.NO_PADDING)).removeSuffix("\n"))
        if (pwd != null && pwd.isNotEmpty()) {
            buffer.append("&")
                .append(SCHEME_PARAM_PWD)
                .append("=")
                .append(
                    String(
                        Base64.encode(pwd.toByteArray(), Base64.NO_PADDING)
                    ).removeSuffix("\n")
                )
        }
        Logger.i(TAG, "Generate room link:$buffer")
        return buffer.toString()
    }

    fun generateAgoraLink(rid: String, pwd: String?, type: Int, code: String? = null): Any? {
        val buffer = StringBuffer()
        buffer.append(getUriScheme())
            .append("/")
            .append(SECURE_ROOM)
            .append("?")
            .append(SCHEME_PARAM_RID)
            .append("=")
            .append(String(Base64.encode(rid.toByteArray(), Base64.NO_PADDING)).removeSuffix("\n"))

        if (pwd != null && pwd.isNotEmpty()) {
            buffer.append("&")
                .append(SCHEME_PARAM_PWD)
                .append("=")
                .append(
                    String(
                        Base64.encode(pwd.toByteArray(), Base64.NO_PADDING)
                    ).removeSuffix("\n")
                )
        }

        buffer.append("&")
            .append(SCHEME_PARAM_TYPE)
            .append("=")
            .append(type)

        if (code != null && code.isNotEmpty()) {
            buffer.append("&")
                .append(SCHEME_PARAM_CODE)
                .append("=")
                .append(code)
        }

        Logger.i(TAG, "Generate agora link:$buffer")
        return buffer.toString()

    }
}