package io.agora.avc.widget

import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.AppCompatTextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.databinding.DataBindingUtil
import io.agora.avc.R
import io.agora.avc.bo.NetworkQuality
import io.agora.avc.databinding.AvcLayoutActionBarBinding
import io.agora.widget.BackImageView

/***
 * Created by liqilin on 2019-09-04
 *
 */
class CustomActionBar @JvmOverloads constructor(
    context: Context,
    attributeSet: AttributeSet? = null,
    defStyleAttr: Int = 0
) : ConstraintLayout(context, attributeSet, defStyleAttr) {

    private var binding: AvcLayoutActionBarBinding = DataBindingUtil.inflate(
        LayoutInflater.from(context),
        R.layout.avc_layout_action_bar,
        this,
        true
    )
    private var rtmTipWindow: TipPopupWindow? = null
    private var signalTipWindow: TextPopupWindow? = null
    private var isRTMTips: Boolean = true
    private var actionTxt: String? = null
    private var actionTxtClickListener: OnActionTxtClickListener? = null

    init {
        val typedArray = context.obtainStyledAttributes(attributeSet, R.styleable.AvcCustomActionBar)

        val barTitle = typedArray.getString(R.styleable.AvcCustomActionBar_avc_bar_title)
        val actionTxt = typedArray.getString(R.styleable.AvcCustomActionBar_avc_action_txt)
        typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_action_txt_enable, true).let {
            binding.tvAction.isEnabled = it
        }
        val hasBackIcon = typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_hasBackIcon, true)
        val backIconDrawable = typedArray.getDrawable(R.styleable.AvcCustomActionBar_avc_backIcon)
        val hasDurationText =
            typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_hasDurationText, true)
        val hasSignalIcon = typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_hasSignalIcon, true)
        val hasSettingsIcon =
            typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_hasSettingsIcon, true)
        val ellipsisTitle = typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_ellipsisTitle, false)
        val constrainedWidth =
            typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_constrainedWidth, false)
        isRTMTips = typedArray.getBoolean(R.styleable.AvcCustomActionBar_avc_isRTMTip, true)

        val menuId = typedArray.getResourceId(R.styleable.AvcCustomActionBar_avc_action_menu, -1)

        typedArray.recycle()

        barTitle?.let {
            setTitle(it)
        }

        setActionTxt(actionTxt)

        binding.btBack.visibility = if (hasBackIcon) View.VISIBLE else View.GONE

        if (backIconDrawable != null) {
            binding.btBack.setImageDrawable(backIconDrawable)
        }

        binding.btBack.setOnClickListener {
            onBackClicked()
        }

//        binding.duration.visibility = if (hasDurationText) View.INVISIBLE else View.GONE

        binding.tvAction.setOnClickListener {
            actionTxtClickListener?.onActionTxtClicked()
        }

        binding.disconnect.setOnClickListener {
            showDisconnectTip()
        }
        binding.disconnect.isFocusable = false

        binding.icSignal.visibility = if (hasSignalIcon) View.VISIBLE else View.GONE
        binding.icSignal.setOnClickListener {
            showSignalTip()
        }

        binding.btSettings.visibility = if (hasSettingsIcon) View.VISIBLE else View.GONE
        binding.btSettings.isMenu = true

        // Some small dpi screen, title is to narrow.
        if (ellipsisTitle) {
            (binding.title.layoutParams as ViewGroup.LayoutParams).width = 0
        }
        if (constrainedWidth) {
            (binding.title.layoutParams as LayoutParams).constrainedWidth = true
        }

        if (menuId != -1) {
            binding.stub.viewStub?.layoutResource = menuId
            binding.stub.viewStub?.inflate()
            binding.stub.root?.visibility = View.VISIBLE
        } else {
            binding.stub.root?.visibility = View.GONE
        }
    }

    fun setTitle(title: CharSequence) {
        binding.title.text = title
    }

//    fun setDisplayTime(time: String) {
//        if (binding.duration.visibility != View.VISIBLE) {
//            binding.duration.visibility = View.VISIBLE
//        }
//
//        var timeStr = time
//        if (timeStr.startsWith("00:")) {
//            timeStr = timeStr.substring(3) // reduce 00: hour
//        }
//        binding.duration.text = timeStr
//    }

    fun setRoomRtmState(visible: Boolean) {
        binding.disconnect.visibility = if (visible) View.VISIBLE else View.INVISIBLE
    }

    private fun showDisconnectTip() {
        val pos = IntArray(2)
        binding.disconnect.getLocationOnScreen(pos)

        pos[0] += binding.disconnect.width / 2
        pos[1] += binding.disconnect.height * 4 / 5

        rtmTipWindow = TextPopupWindow(
            context as Activity,
            if (isRTMTips) R.string.avc_pop_rtm_disconnected else R.string.avc_business_connecting
        )
        rtmTipWindow?.showAtLocation(pos, TipPopupWindow.DIRECTION_DOWN)
        rtmTipWindow?.setOnDismissListener {
            rtmTipWindow = null
        }
    }

    private fun showSignalTip() {
        val pos = IntArray(2)
        binding.icSignal.getLocationOnScreen(pos)

        pos[0] += binding.icSignal.width / 2
        pos[1] += binding.icSignal.height

        signalTipWindow =
            TextPopupWindow(context as Activity, getSignalText(binding.icSignal.quality))
                .apply {
                    showAtLocation(pos, TipPopupWindow.DIRECTION_DOWN)

                    setOnDismissListener {
                        signalTipWindow = null
                    }
                }
    }

    fun setSignalQuality(quality: Int) {
        binding.icSignal.setQuality(quality)

        signalTipWindow?.updateText(context.getText(getSignalText(binding.icSignal.quality)))
    }

    private fun onBackClicked() {
        (context as AppCompatActivity).onBackPressed()
    }

    fun getTitleView(): AppCompatTextView = binding.title

    fun getBackIconView(): BackImageView = binding.btBack

    fun getSettingsView(): SettingsIconView = binding.btSettings

    fun hideTipWindow() {
        if (rtmTipWindow?.isShowing == true) {
            rtmTipWindow?.dismiss()
            rtmTipWindow = null
        }

        if (signalTipWindow?.isShowing == true) {
            signalTipWindow?.dismiss()
            signalTipWindow = null
        }
    }

    fun isTipWindowShowing() = rtmTipWindow?.isShowing == true || signalTipWindow?.isShowing == true

    fun consumeRightInset(rightInset: Int) {
        (binding.btSettings.layoutParams as MarginLayoutParams).setMargins(
            0,
            0,
            rightInset + context.resources.getDimensionPixelSize(R.dimen.avc_inset_end_small),
            0
        )
    }

    /**
     * set state mark rtc or rtm
     */
    fun setRTMTips(bool: Boolean) {
        this.isRTMTips = bool
    }

    fun setLogoVisible(visible: Boolean) {
        if (visible) {
            binding.logo.visibility = View.VISIBLE
        } else {
            binding.logo.visibility = View.GONE
        }
    }

    fun getActionMenu(): View? {
        return binding.stub.root
    }

    fun setOnActionTxtClickListener(listener: OnActionTxtClickListener) {
        this.actionTxtClickListener = listener
    }

    fun setActionTxt(txt: String?) {
        this.actionTxt = txt
        binding.tvAction.text = txt ?: ""
        binding.tvAction.visibility = if (txt.isNullOrEmpty()) View.GONE else View.VISIBLE
    }

    fun setActionEnable(activated: Boolean) {
        binding.tvAction.isEnabled = activated
    }
}

interface OnActionTxtClickListener {
    fun onActionTxtClicked()
}

fun getSignalText(quality: Int): Int {
    return when (quality) {
        NetworkQuality.EXCELLENT,
        NetworkQuality.GOOD -> R.string.avc_signal_good
        NetworkQuality.POOR -> R.string.avc_signal_soft
        NetworkQuality.BAD,
        NetworkQuality.VBAD -> R.string.avc_signal_weak
        else -> R.string.avc_signal_check
    }
}