package io.agora.avc.widget

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Rect
import android.util.AttributeSet
import android.util.SparseArray
import android.view.MotionEvent
import android.view.View
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.util.forEach
import androidx.core.view.ViewCompat
import androidx.customview.widget.ViewDragHelper
import io.agora.avc.R
import io.agora.avc.utils.ScreenUtils

class DragDownLayout @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr) {

    private var isFirsLayout = true

    private var dragViewId = View.NO_ID
    private lateinit var dragView: View

    private var dragDownAreaPercent = DEFAULT_DRAG_DOWN_AREA_PERCENT
    private var targetViewId = View.NO_ID
    private lateinit var targetView: View

    /**
     * Since the dragged position is not the real position,
     * it will return to the original position after re-layout,
     * so here we need to save the dragged position
     */
    private val dragViews by lazy {
        SparseArray<View>()
    }
    private val dragViewPositions by lazy {
        mutableMapOf<View, Rect>()
    }

    private var _isTargetViewShowing = false

    var onTargetViewVisibilityChanged: ((Boolean) -> Unit)? = null

    var priorityView: View? = null

    private val viewDragHelper: ViewDragHelper =
        ViewDragHelper.create(this, object : ViewDragHelper.Callback() {
            override fun tryCaptureView(child: View, pointerId: Int): Boolean {
                return true
            }

            override fun clampViewPositionVertical(child: View, top: Int, dy: Int): Int {
                return when {
                    child === dragView -> {
                        top.coerceAtLeast(0)
                    }
                    child === targetView -> {
                        top.coerceAtMost(0)
                    }
                    else -> {
                        top
                    }
                }
            }

            override fun onViewPositionChanged(
                changedView: View,
                left: Int,
                top: Int,
                dx: Int,
                dy: Int
            ) {
                when (changedView) {
                    dragView -> {
                        ViewCompat.offsetTopAndBottom(targetView, dy)
                    }
                    else -> {
                        ViewCompat.offsetTopAndBottom(dragView, dy)
                    }
                }
                savePositionOfDragViews()
            }

            override fun onViewReleased(releasedChild: View, xvel: Float, yvel: Float) {
                when (releasedChild) {
                    dragView -> {
                        if (dragView.top.coerceAtLeast(dragView.top + (yvel * HIDE_FRICTION).toInt()) >= this@DragDownLayout.height * 0.5f) {
                            settleCapturedViewAt(this@DragDownLayout.height)
                            setTargetViewShowing(true)
                        } else {
                            settleCapturedViewAt(0)
                        }
                    }
                    targetView -> {
                        if (targetView.bottom.coerceAtMost(dragView.bottom + (yvel * HIDE_FRICTION * 2).toInt()) >= this@DragDownLayout.height * 0.5f) {
                            settleCapturedViewAt(0)
                        } else {
                            settleCapturedViewAt(-targetView.height)
                            setTargetViewShowing(false)
                        }
                    }
                }
                savePositionOfDragViews()
                invalidate()
            }

            override fun onViewDragStateChanged(state: Int) {
                viewDragHelperCallback?.onViewDragStateChanged(state)
            }
        })

    private fun settleCapturedViewAt(finalTop: Int) {
        viewDragHelper.settleCapturedViewAt(0, finalTop)
    }

    private fun setTargetViewShowing(showing: Boolean) {
        _isTargetViewShowing = showing
        onTargetViewVisibilityChanged?.invoke(showing)
    }

    var viewDragHelperCallback: ViewDragHelper.Callback? = null

    init {
        context.obtainStyledAttributes(attrs, R.styleable.AvcDragDownLayout).apply {
            dragViewId = getResourceId(R.styleable.AvcDragDownLayout_dragView, View.NO_ID)
            targetViewId = getResourceId(R.styleable.AvcDragDownLayout_targetView, View.NO_ID)
            dragDownAreaPercent = getFloat(
                R.styleable.AvcDragDownLayout_dragDownAreaPercent,
                DEFAULT_DRAG_DOWN_AREA_PERCENT
            )
            recycle()
        }
    }

    override fun onFinishInflate() {
        super.onFinishInflate()
        if (dragViewId == View.NO_ID) {
            throw IllegalArgumentException("please set dragView!")
        }
        dragView = findViewById(dragViewId)
        dragViews.put(dragView.id, dragView)
        dragViewPositions[dragView] = Rect()

        if (targetViewId == View.NO_ID) {
            throw IllegalArgumentException("please set targetView!")
        }
        targetView = findViewById(targetViewId)
        dragViews.put(targetView.id, targetView)
        dragViewPositions[targetView] = Rect()
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        if (_isTargetViewShowing) {
            dragView.top = getTopOffset()
        } else {
            targetView.top = -getTopOffset()
        }
        savePositionOfDragViews()
    }

    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        super.onLayout(changed, left, top, right, bottom)
        if (dragViewPositions.isNotEmpty() && isFirsLayout) {
            isFirsLayout = false
            savePositionOfDragViews()
        }
        dragViews.forEach { _, value ->
            value.layout(
                dragViewPositions[value]?.left ?: value.left,
                dragViewPositions[value]?.top ?: value.top,
                (dragViewPositions[value]?.left ?: value.left) + value.measuredWidth,
                (dragViewPositions[value]?.top ?: value.top) + value.measuredHeight,
            )
        }
    }

    override fun computeScroll() {
        super.computeScroll()
        if (viewDragHelper.continueSettling(true)) {
            invalidate()
        }
    }

    override fun onInterceptTouchEvent(ev: MotionEvent): Boolean {
        return viewDragHelper.shouldInterceptTouchEvent(ev)
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        viewDragHelper.processTouchEvent(event)
        if (viewDragHelper.capturedView === targetView
            || (viewDragHelper.capturedView === dragView
                    && priorityView?.dispatchTouchEvent(event) != true)
        ) {
            return true
        }
        return super.onTouchEvent(event)
    }

    /**
     * Since the dragged position is not the real position,
     * it will return to the original position after re-layout,
     * so here we need to save the dragged position
     */
    fun savePositionOfDragViews() {
        for (dragViewPosition in dragViewPositions) {
            dragViewPosition.value.left = dragViewPosition.key.left
            dragViewPosition.value.top = dragViewPosition.key.top
        }
    }

    fun reset() {
        viewDragHelper.smoothSlideViewTo(
            dragView,
            0,
            0
        )
        savePositionOfDragViews()
        _isTargetViewShowing = false
    }

    private fun getTopOffset() = ScreenUtils.getScreenHeight()

    companion object {
        const val TAG = "[UI][DragDownLayout]"
        private const val HIDE_FRICTION = 0.1f
        private const val DEFAULT_DRAG_DOWN_AREA_PERCENT = 0.25f
    }
}