package io.agora.avc.widget

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.drawable.ClipDrawable
import android.graphics.drawable.Drawable
import android.graphics.drawable.LayerDrawable
import android.graphics.drawable.StateListDrawable
import android.os.Parcel
import android.os.Parcelable
import android.util.AttributeSet
import android.util.SparseArray
import android.view.LayoutInflater
import android.view.ViewGroup
import android.view.animation.AlphaAnimation
import android.view.animation.Animation
import android.widget.ImageView
import androidx.annotation.IntDef
import androidx.appcompat.content.res.AppCompatResources
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.isVisible
import androidx.customview.view.AbsSavedState
import androidx.databinding.DataBindingUtil
import io.agora.avc.R
import io.agora.avc.databinding.AvcLayoutLoadingImageButtonBinding


class LoadingImageButton @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr) {

    private var tintColor: ColorStateList? = null
    var drawableChecked: Drawable? = null
    var drawableUnChecked: Drawable? = null

    val binding: AvcLayoutLoadingImageButtonBinding by lazy {
        DataBindingUtil.inflate(
            LayoutInflater.from(context),
            R.layout.avc_layout_loading_image_button,
            this,
            true
        )
    }

    @State
    private var state = State.LOADING
        set(value) {
            field = value
            binding.ivSpeaking.isVisible = value == State.SPEAKING
            binding.loadingView.isVisible = value == State.LOADING
            binding.ivState.isVisible = (value != State.LOADING && value != State.SPEAKING)
            binding.ivState.isChecked = value == State.CHECKED
        }

    var opened = false
        set(value) {
            field = value
            state = if (value) State.CHECKED else State.UNCHECKED
        }

    var loading = false
        set(value) {
            field = value
            state = if (value) State.LOADING else (if (opened) State.CHECKED else State.UNCHECKED)
        }

    var speaking = false
        set(value) {
            field = value
            state = if (value) State.SPEAKING else (if (opened) State.CHECKED else State.UNCHECKED)
        }

    var volume = 0
        set(value) {
            field = value
            ((drawableChecked as? LayerDrawable)?.getDrawable(1) as? ClipDrawable)?.level =
                value * 10000 / 255
        }

    init {

        var tempOpened: Boolean

        var tintEnable: Boolean

        context.obtainStyledAttributes(attrs, R.styleable.AvcLoadingImageButton).apply {

            getDimension(
                R.styleable.AvcLoadingImageButton_avc_imageWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT.toFloat()
            ).apply {
                binding.ivState.layoutParams.width = this.toInt()
                binding.ivSpeaking.layoutParams.width = this.toInt()
                binding.loadingView.layoutParams.width = this.toInt()
            }

            getDimension(
                R.styleable.AvcLoadingImageButton_avc_imageHeight,
                ViewGroup.LayoutParams.WRAP_CONTENT.toFloat()
            ).apply {
                binding.ivState.layoutParams.height = this.toInt()
                binding.ivSpeaking.layoutParams.height = this.toInt()
                binding.loadingView.layoutParams.height = this.toInt()
            }

            getInt(R.styleable.AvcLoadingImageButton_android_scaleType, 3).let {
                binding.ivState.scaleType = sScaleTypeArray[it]
                binding.ivSpeaking.scaleType = sScaleTypeArray[it]
            }

            binding.ivState.layoutParams = binding.ivState.layoutParams
            binding.ivSpeaking.layoutParams = binding.ivSpeaking.layoutParams

            drawableChecked = getDrawable(R.styleable.AvcLoadingImageButton_avc_openedDrawable)
            drawableUnChecked = getDrawable(R.styleable.AvcLoadingImageButton_avc_closeDrawable)

            tintColor = getColorStateList(R.styleable.AvcLoadingImageButton_avc_tint)
            tintEnable = getBoolean(R.styleable.AvcLoadingImageButton_avc_tintEnable, true)

            tempOpened = getBoolean(R.styleable.AvcLoadingImageButton_avc_opened, false)

            isActivated = getBoolean(R.styleable.AvcLoadingImageButton_avc_activated, true)

            recycle()
        }

        StateListDrawable().apply {
            addState(intArrayOf(android.R.attr.state_checked), drawableChecked)
            addState(intArrayOf(), drawableUnChecked)

            binding.ivState.setImageDrawable(this)
        }

        if (tintColor == null && tintEnable) {
            tintColor =
                AppCompatResources.getColorStateList(
                    context,
                    R.color.avc_selector_action_image_tint_color
                )
        }

        if (tintEnable) {
            binding.ivState.imageTintList = tintColor
        }

        opened = tempOpened
    }

    fun startAnimator() {
        stopAnimator()
        AlphaAnimation(0f, 1f).apply {
            duration = 500
            repeatCount = -1
            repeatMode = Animation.REVERSE
            binding.ivState.startAnimation(this)
        }
    }

    fun stopAnimator() {
        binding.ivState.clearAnimation()
    }

    override fun onDetachedFromWindow() {
        stopAnimator()
        super.onDetachedFromWindow()
    }

    @IntDef(
        State.CHECKED,
        State.UNCHECKED,
        State.LOADING,
        State.SPEAKING
    )
    annotation class State {
        companion object {
            const val CHECKED = 0
            const val UNCHECKED = 1
            const val LOADING = 2
            const val SPEAKING = 3
        }
    }

    companion object {
        private val sScaleTypeArray = arrayOf(
            ImageView.ScaleType.MATRIX,
            ImageView.ScaleType.FIT_XY,
            ImageView.ScaleType.FIT_START,
            ImageView.ScaleType.FIT_CENTER,
            ImageView.ScaleType.FIT_END,
            ImageView.ScaleType.CENTER,
            ImageView.ScaleType.CENTER_CROP,
            ImageView.ScaleType.CENTER_INSIDE
        )
    }

    override fun onSaveInstanceState(): Parcelable {
        val superState = super.onSaveInstanceState()
        val savedState = SavedState(superState)
        savedState.childrenStates = SparseArray(3)
        for (i in 0 until childCount) {
            getChildAt(i).saveHierarchyState(savedState.childrenStates)
        }
        return savedState
    }

    override fun onRestoreInstanceState(state: Parcelable) {
        if (state is SavedState) {
            val savedState = state
            super.onRestoreInstanceState(savedState.superState)
            for (i in 0 until childCount) {
                getChildAt(i).restoreHierarchyState(savedState.childrenStates)
            }
        } else {
            super.onRestoreInstanceState(state)
        }
    }

    override fun dispatchSaveInstanceState(container: SparseArray<Parcelable>?) {
        dispatchFreezeSelfOnly(container)
    }

    override fun dispatchRestoreInstanceState(container: SparseArray<Parcelable>?) {
        dispatchThawSelfOnly(container)
    }

    class SavedState : AbsSavedState {
        var childrenStates: SparseArray<Parcelable>? = null

        constructor(superState: Parcelable?) : super(superState!!) {}

        constructor(source: Parcel, loader: ClassLoader?) : super(source, loader) {
            readFromParcel(source, loader)
        }

        override fun writeToParcel(out: Parcel, flags: Int) {
            super.writeToParcel(out, flags)
            out.writeSparseArray(childrenStates)
        }

        private fun readFromParcel(`in`: Parcel, loader: ClassLoader?) {
            childrenStates = `in`.readSparseArray(loader)
        }

        companion object {
            @JvmField
            val CREATOR: Parcelable.Creator<SavedState> =
                object : Parcelable.ClassLoaderCreator<SavedState> {
                    override fun createFromParcel(`in`: Parcel, loader: ClassLoader): SavedState {
                        return SavedState(`in`, loader)
                    }

                    override fun createFromParcel(`in`: Parcel): SavedState {
                        return SavedState(`in`, null)
                    }

                    override fun newArray(size: Int): Array<SavedState?> {
                        return arrayOfNulls(size)
                    }
                }
        }
    }
}

