package io.agora.avc.widget

import android.content.Context
import android.content.DialogInterface
import android.view.LayoutInflater
import android.view.View
import androidx.appcompat.app.AlertDialog
import androidx.databinding.DataBindingUtil
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.bo.ConversationQuality
import io.agora.avc.databinding.AvcDialogQualityReportBinding
import io.agora.avc.utils.KeyboardUtils

class QualityReportDialog(val context: Context) {

    private var submitClickListener: OnSubmitClickListener? = null
    private var dismissListener: DialogInterface.OnDismissListener? = null
    private var onShowListener: DialogInterface.OnShowListener? = null
    private lateinit var conversationQuality: ConversationQuality
    private var dialog: AlertDialog? = null
    private var startTime: Long = 0

    private val dataBinding: AvcDialogQualityReportBinding by lazy {
        DataBindingUtil.inflate<AvcDialogQualityReportBinding>(
            LayoutInflater.from(context),
            R.layout.avc_dialog_quality_report,
            null,
            false
        )
    }

    private fun create(): MaterialAlertDialogBuilder {
        val margin = context.resources.getDimension(R.dimen.avc_inset_start).toInt()
        return MaterialAlertDialogBuilder(context, R.style.avc_CustomMaterialAlertDialog)
            .setView(dataBinding.root)
            .setBackgroundInsetStart(margin)
            .setBackgroundInsetEnd(margin)
            .setBackgroundInsetTop(margin)
            .setBackgroundInsetBottom(margin)
            .setCancelable(false)
            .setOnDismissListener {
                dismissListener?.onDismiss(it)
            }
    }

    private fun initListener() {
        dataBinding.cbAudio.setOnCheckedChangeListener { _, isChecked ->
            if (isChecked) {
                dataBinding.inputAudioComment.visibility = View.VISIBLE
            } else {
                dataBinding.inputAudioComment.visibility = View.GONE
            }
        }
        dataBinding.cbVideo.setOnCheckedChangeListener { _, isChecked ->
            if (isChecked) {
                dataBinding.inputVideoComment.visibility = View.VISIBLE
            } else {
                dataBinding.inputVideoComment.visibility = View.GONE
            }
        }
        dataBinding.audioRating.setOnRatingBarChangeListener { ratingBar, rating, fromUser ->
            if (rating < 1f) {
                ratingBar.rating = 1f
                return@setOnRatingBarChangeListener
            }
            when (rating.toInt()) {
                1 -> {
                    dataBinding.tvAudioQuality.setText(R.string.avc_quality_bad)
                }
                2 -> {
                    dataBinding.tvAudioQuality.setText(R.string.avc_quality_below_average)
                }
                3 -> {
                    dataBinding.tvAudioQuality.setText(R.string.avc_quality_average)
                }
                4 -> {
                    dataBinding.tvAudioQuality.setText(R.string.avc_quality_good)
                }
                5 -> {
                    dataBinding.tvAudioQuality.setText(R.string.avc_quality_excellent)
                }
                else -> {
                    dataBinding.tvAudioQuality.text = ""
                }
            }
        }
        dataBinding.videoRating.setOnRatingBarChangeListener { ratingBar, rating, fromUser ->
            if (rating < 1f) {
                ratingBar.rating = 1f
                return@setOnRatingBarChangeListener
            }
            when (rating.toInt()) {
                1 -> {
                    dataBinding.tvVideoQuality.setText(R.string.avc_quality_bad)
                }
                2 -> {
                    dataBinding.tvVideoQuality.setText(R.string.avc_quality_below_average)
                }
                3 -> {
                    dataBinding.tvVideoQuality.setText(R.string.avc_quality_average)
                }
                4 -> {
                    dataBinding.tvVideoQuality.setText(R.string.avc_quality_good)
                }
                5 -> {
                    dataBinding.tvVideoQuality.setText(R.string.avc_quality_excellent)
                }
                else -> {
                    dataBinding.tvVideoQuality.text = ""
                }
            }
        }
        dataBinding.ivClose.setOnClickListener {
            KeyboardUtils.hideSoftInput(dataBinding.root)
            dismiss()
        }
        dataBinding.btnSubmit.setOnClickListener {
            collectConversationData()
        }
    }

    private fun collectConversationData() {
        val audioRemark = if (dataBinding.cbAudio.isChecked) {
            dataBinding.etAudioComment.text?.toString()
        } else {
            ""
        }
        val videoRemark = if (dataBinding.cbVideo.isChecked) {
            dataBinding.etVideoComment.text?.toString()
        } else {
            ""
        }

        val audioLabel = IntArray(dataBinding.audioChips.checkedChipIds.size)
        var audioChipIndex = 0
        for (chipId in dataBinding.audioChips.checkedChipIds) {
            when (chipId) {
                R.id.chip1_1 -> {
                    audioLabel[audioChipIndex++] = 1
                }
                R.id.chip1_2 -> {
                    audioLabel[audioChipIndex++] = 2
                }
                R.id.chip1_3 -> {
                    audioLabel[audioChipIndex++] = 3
                }
                R.id.chip1_4 -> {
                    audioLabel[audioChipIndex++] = 4
                }
                R.id.chip1_5 -> {
                    audioLabel[audioChipIndex++] = 5
                }
            }
        }

        val videoLabel = IntArray(dataBinding.videoChips.checkedChipIds.size)
        var videoChipIndex = 0
        for (chipId in dataBinding.videoChips.checkedChipIds) {
            when (chipId) {
                R.id.chip2_1 -> {
                    videoLabel[videoChipIndex++] = 1
                }
                R.id.chip2_2 -> {
                    videoLabel[videoChipIndex++] = 2
                }
                R.id.chip2_3 -> {
                    videoLabel[videoChipIndex++] = 3
                }
                R.id.chip2_4 -> {
                    videoLabel[videoChipIndex++] = 4
                }
            }
        }

        conversationQuality.audioRating = dataBinding.audioRating.rating
        conversationQuality.audioRemark = audioRemark ?: ""
        conversationQuality.videoRating = dataBinding.videoRating.rating
        conversationQuality.videoRemark = videoRemark ?: ""
        conversationQuality.audioLabel = audioLabel
        conversationQuality.videoLabel = videoLabel
        conversationQuality.delta = System.currentTimeMillis() - startTime

        submitClickListener?.onSubmitClick(conversationQuality)
        dialog?.dismiss()
    }

    fun show() {
        initialize()
        initListener()
        val builder = create()
        this.dialog = builder.create()
        this.dialog?.setOnShowListener { dialog ->
            startTime = System.currentTimeMillis()
            onShowListener?.onShow(dialog)
        }
        this.dialog?.show()
    }

    private fun initialize() {
        if (conversationQuality.manual) {
            dataBinding.title.setText(R.string.avc_quality_feedback_title)
        } else {
            dataBinding.title.setText(R.string.avc_quality_report_title)
        }
    }

    fun setOnSubmitClickListener(onSubmitClickListener: OnSubmitClickListener) {
        this.submitClickListener = onSubmitClickListener
    }

    fun setOnDismissListener(listener: DialogInterface.OnDismissListener?) {
        this.dismissListener = listener
    }

    fun setOnShowListener(listener: DialogInterface.OnShowListener) {
        this.onShowListener = listener
    }

    fun setConversationQuality(quality: ConversationQuality) {
        this.conversationQuality = quality
    }

    fun dismiss() {
        dialog?.let { alert ->
            if (alert.isShowing) {
                alert.dismiss()
            }
        }
    }

    interface OnSubmitClickListener {
        fun onSubmitClick(conversationQuality: ConversationQuality)
    }

}
