package io.agora.avc.widget

import android.content.Context
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.PopupWindow
import androidx.core.view.isVisible
import io.agora.avc.R
import io.agora.avc.app.developer.DeveloperOptions
import io.agora.avc.databinding.AvcLayoutRoomMenuBinding
import io.agora.avc.utils.ConvertUtils
import io.agora.avc.utils.ScreenUtils
import io.agora.avc.utils.ToastUtils
import io.agora.logger.Logger

/***
 * Created by liqilin on 2019-11-27
 * Did not refresh View,when user share state changed,
 * because the scene recurs with a small probability
 * If necessary,add it latter
 */
class RoomMenuPopupWindow constructor(
    private val ctx: Context,
    private var isSharingScreen: Boolean,
    private var isThirdPartyLoggedIn: Boolean,
    private var isCloudRecording: Boolean,
    private val isAgoraRoom: Boolean,
    private val developerOptions: DeveloperOptions?,
    private var isVideoRating: Boolean,
    private var hasUnreadMessage: Boolean,
    private var videoRateTime: Int,
    val callback: MenuCallback
) : PopupWindow(), PopupWindow.OnDismissListener {

    private val binding: AvcLayoutRoomMenuBinding by lazy {
        AvcLayoutRoomMenuBinding.inflate(
            LayoutInflater.from(ctx),
            null,
            false
        )
    }

    private var dismissListener: OnDismissListener? = null

    init {
        initView()
    }

    fun updateVideoRateTime(videoRateTime: Int) {
        if (videoRateTime > 0) {
            binding.tvVideoRating.text =
                "${ctx.getString(R.string.avc_test_video_rate)}(${videoRateTime})"
        } else {
            videoRateReset()
        }
    }

    fun videoRateReset() {
        binding.tvVideoRating.text = ctx.getString(R.string.avc_test_video_rate)
    }

    fun updateCloudRecord(
        isThirdPartyLoggedIn: Boolean = this.isThirdPartyLoggedIn,
        isCloudRecording: Boolean = this.isCloudRecording,
        isAgoraRoom: Boolean = this.isAgoraRoom
    ) {
//        binding.llCloudRecord.isVisible = isAgoraRoom && isThirdPartyLoggedIn
//        binding.ivCloudRecord.opened = isCloudRecording
    }

    fun updateShareItem(isSharingScreen: Boolean = this.isSharingScreen) {
        if (binding.ivScreenShare.opened != isSharingScreen) {
            Logger.i(TAG, "setup menu popupWindow, isScreenSharing:$isSharingScreen")
            binding.ivScreenShare.opened = isSharingScreen
            if (isSharingScreen) {
                binding.ivScreenShare.startAnimator()
            } else {
                binding.ivScreenShare.stopAnimator()
            }
        }
    }

    fun updateHasUnreadMessage(hasUnreadMessage: Boolean) {
        binding.ivChatDot.isVisible = hasUnreadMessage
    }

    private fun initView() {
        contentView = getRootView()
        updateShareItem()
        updateCloudRecord()
        initDeveloperItem()
        initChatItem()
        initVideoRateItem()
        initClickListener()
        initPopupStyle()
    }

    private fun initVideoRateItem() {
        updateVideoRateTime(videoRateTime)
    }

    private fun initChatItem() {
        updateHasUnreadMessage(hasUnreadMessage)
    }

    private fun initDeveloperItem() {
        binding.llQualityReport.isVisible = developerOptions?.qualityReport == true
        binding.llVideoRating.isVisible = developerOptions?.videoRate == true
        binding.ivVideoRating.opened = isVideoRating
    }

    private fun initClickListener() {
        binding.llShareScreen.setOnClickListener {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                isSharingScreen = !isSharingScreen
                callback.onShareScreenStateChange(binding.llShareScreen, isSharingScreen)
            } else {
                ToastUtils.showShort(R.string.avc_screen_share_not_support)
            }
            dismiss()
        }
        binding.llCloudRecord.setOnClickListener {
            isCloudRecording = !isCloudRecording
            callback.onCloudRecordingChange(it, isCloudRecording)
            dismiss()
        }
        binding.llChat.setOnClickListener {
            callback.onClickChat()
            dismiss()
        }
        binding.llVideoRating.setOnClickListener {
            callback.onVideoRatingClicked()
            dismiss()
        }
        binding.llQualityReport.setOnClickListener {
            callback.onQualityReportClicked()
            dismiss()
        }
    }

    private fun getRootView(): View {
        return binding.root
    }

    private fun initPopupStyle() {
        // Check it on emulator api17 or QA's oppo device
        // Under api 24, not set background can not cancel popup window by touch outside and back key.
        setBackgroundDrawable(ColorDrawable())
        width = (ScreenUtils.getScreenWidth() * 0.61f).toInt()
        height = ViewGroup.LayoutParams.WRAP_CONTENT
        isFocusable = true
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            elevation = ctx.resources.getDimension(R.dimen.avc_shadow_height)
        }
    }

    fun showUpAnchorView(anchorView: View) {
        val isRightToLeft = ctx.resources.getBoolean(R.bool.avc_is_right_to_left)
        animationStyle = if (!isRightToLeft) {
            R.style.avc_menu_pop_animation_bottom_right
        } else {
            R.style.avc_menu_pop_animation_bottom_left
        }
        showAtLocation(
            anchorView,
            Gravity.START or Gravity.BOTTOM,
            (ScreenUtils.getAppScreenWidth() * 0.36f).toInt(),
            anchorView.height - ConvertUtils.dp2px(16f)
        )
        setOnDismissListener(this)
    }

    override fun onDismiss() {
        dismissListener?.onDismiss()
    }

    override fun setOnDismissListener(onDismissListener: OnDismissListener?) {
        super.setOnDismissListener(this)
        if (onDismissListener != this) {
            dismissListener = onDismissListener
        }
    }

    interface MenuCallback {

        /**
         * @param open start cloud record
         */
        fun onCloudRecordingChange(view: View, open: Boolean)

        /**
         * @param open start share screen
         */
        fun onShareScreenStateChange(view: View, open: Boolean)
        fun onClickChat()
        fun onVideoRatingClicked()
        fun onQualityReportClicked()
    }

    companion object {
        const val TAG = "[UI][RoomMenuPop]"
    }
}