package io.agora.avc.widget

import android.content.Context
import android.graphics.*
import android.util.AttributeSet
import android.util.TypedValue
import android.view.View
import androidx.annotation.IntDef
import io.agora.avc.R

class RoundProgressBar @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : View(context, attrs, defStyleAttr) {

    private var boundRectF = RectF()
    private var arcRectF = RectF()

    private val progressPaint by lazy {
        Paint(Paint.ANTI_ALIAS_FLAG)
    }

    private val bgPaint by lazy {
        Paint(Paint.ANTI_ALIAS_FLAG)
    }

    private var startAngle = 270f
    private var endAngle = 630f
    private var cornerRadius = 0f

    private val clipPath by lazy {
        Path()
    }
    var progress: Int = 0
        set(value) {
            field = value
            invalidate()
        }

    var max: Int = 0

    var indicatorColor: Int = 0
        set(value) {
            field = value
            progressPaint.color = value
        }

    @IndicatorDirection
    var indicatorDirection: Int = IndicatorDirection.clockwise

    init {
        context.obtainStyledAttributes(attrs, R.styleable.AvcRoundProgressBar).apply {

            cornerRadius = getDimension(R.styleable.AvcRoundProgressBar_radius, 0f)

            progress = getInt(R.styleable.AvcRoundProgressBar_progress, 0)
            max = getInt(R.styleable.AvcRoundProgressBar_max, 0)

            indicatorDirection = getInt(
                R.styleable.AvcRoundProgressBar_indicatorDirection,
                IndicatorDirection.clockwise
            )

            indicatorColor =
                getColor(R.styleable.AvcRoundProgressBar_indicatorColor, getColorPrimary())

            getColor(R.styleable.AvcRoundProgressBar_trackColor, getColorPrimary()).let {
                bgPaint.color = it
            }


            recycle()
        }
        progressPaint.color = indicatorColor
    }

    private fun clockwise() = indicatorDirection == IndicatorDirection.counterclockwise

    private fun getColorPrimary(): Int {
        val typedValue = TypedValue()
        context.theme.resolveAttribute(R.attr.colorPrimary, typedValue, true)
        return typedValue.data
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        boundRectF.right = w.toFloat()
        boundRectF.bottom = h.toFloat()

        arcRectF.left = -w / 2f
        arcRectF.top = -h / 2f
        arcRectF.right = w + w / 2f
        arcRectF.bottom = h + h / 2f

    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)

        val df = 360f / max * (if (clockwise()) progress else max - progress)
        val sweepAngle = (df).coerceAtMost(endAngle - startAngle)

        canvas.save()
        clipPath.apply {
            reset()
            addRoundRect(boundRectF, cornerRadius, cornerRadius, Path.Direction.CW)
        }
        canvas.clipPath(clipPath)
        canvas.drawArc(arcRectF, startAngle, sweepAngle, true, progressPaint)
        canvas.restore()
    }

    @IntDef(
        IndicatorDirection.clockwise,
        IndicatorDirection.counterclockwise
    )
    annotation class IndicatorDirection {
        companion object {
            const val clockwise = 0
            const val counterclockwise = 1
        }
    }
}