package io.agora.avc.widget

import android.annotation.SuppressLint
import android.app.Activity
import android.graphics.Rect
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout
import android.widget.PopupWindow
import androidx.annotation.LayoutRes
import io.agora.avc.R
import io.agora.avc.utils.ConvertUtils
import io.agora.avc.utils.ScreenUtils
import io.agora.widget.TriangleOutline


/***
 * Created by liqilin on 2019/5/28
 *
 */
abstract class TipPopupWindow(val activity: Activity) : PopupWindow() {

    lateinit var triangle: View
    lateinit var container: FrameLayout
    private lateinit var linearLayout: LinearLayout

    private fun initView(@LayoutRes parentLayoutId: Int) {
        linearLayout = LayoutInflater.from(activity).inflate(
            parentLayoutId,
            null
        ) as LinearLayout

        contentView = linearLayout

        container = linearLayout.findViewById(R.id.container)
        triangle = linearLayout.findViewById(R.id.triangle)

        container.addView(getCustomView())

        initPopupStyle()
    }

    private fun initPopupStyle() {
        setBackgroundDrawable(ColorDrawable())//Under api 24, not set background can not cancel popup window by touch outside and back key.

        width = ViewGroup.LayoutParams.WRAP_CONTENT
        height = ViewGroup.LayoutParams.WRAP_CONTENT

        isFocusable = true

        setOutsideTouchable()
    }

    abstract fun getCustomView(): View

    /**
     * Controls whether the pop-up will be informed of touch events outside
     * of its window.  This only makes sense for pop-ups that are touchable
     * but not focusable, which means touches outside of the window will
     * be delivered to the window behind.  The default is false.
     */
    open fun setOutsideTouchable() {
        isOutsideTouchable = true
    }

    open fun getParentLayoutId(): Int {
        return R.layout.avc_layout_pop_tip_item
    }

    @SuppressLint("RtlHardcoded")
    fun showAtLocation(location: IntArray, direction: Int) {
        showAtLocation(location, direction, null)
    }

    @SuppressLint("RtlHardcoded")
    fun showAtLocation(location: IntArray, direction: Int, windowInsets: Rect?) {
        if (!::container.isInitialized) {
            initView(getParentLayoutId())
        }

        val screenWidth = ScreenUtils.getAppScreenWidth()
        val screenHeight = ScreenUtils.getAppScreenHeight()

        val triangleLength =
            activity.resources.getDimensionPixelSize(R.dimen.avc_pop_win_triangle_width)
        val shadowMargin = ConvertUtils.dp2px(10f) // set margin to leave some space for shadow
        val cornerMargin =
            activity.resources.getDimensionPixelOffset(R.dimen.avc_shape_corner_radius_medium)

        var viewWidth = container.layoutParams.width
        if (container.layoutParams.width == ViewGroup.LayoutParams.WRAP_CONTENT
            || container.layoutParams.width == ViewGroup.LayoutParams.MATCH_PARENT
        ) {
            container.measure(
                View.MeasureSpec.makeMeasureSpec(screenWidth, View.MeasureSpec.AT_MOST),
                View.MeasureSpec.UNSPECIFIED
            )
            viewWidth = container.measuredWidth
        }
        var viewHeight = container.measuredHeight

        var x = 0
        var y = 0
        linearLayout.removeView(triangle)
        val triLayoutParams = triangle.layoutParams as LinearLayout.LayoutParams
        val containerLayoutParams = container.layoutParams as ViewGroup.MarginLayoutParams

        val isRightToLeft = activity.resources.getBoolean(R.bool.avc_is_right_to_left)

        when (direction) {
            DIRECTION_DOWN, DIRECTION_UP -> {
                // 1. Calculate PopupWindow position
                viewWidth += shadowMargin * 2
                viewHeight += shadowMargin

                x = location[0] - viewWidth / 2

                if (x + viewWidth > screenWidth) {
                    x = screenWidth - viewWidth
                }

                x = x.coerceAtLeast(windowInsets?.left ?: 0)

                y = if (direction == DIRECTION_UP) {
                    location[1] - viewHeight - triangleLength / 2
                } else {
                    location[1] - triangleLength / 2
                }

                // 2. Adjust LinearLayout LayoutParams
                linearLayout.orientation = LinearLayout.VERTICAL
                if (direction == DIRECTION_DOWN) {
                    linearLayout.addView(triangle, 0)

                    containerLayoutParams.setMargins(shadowMargin, 0, shadowMargin, shadowMargin)

                    animationStyle = R.style.avc_pop_animation_top
                } else {
                    linearLayout.addView(triangle)

                    containerLayoutParams.setMargins(shadowMargin, shadowMargin, shadowMargin, 0)

                    animationStyle = R.style.avc_pop_animation_bottom
                }

                // 3. Adjust triangle shape and margin
                if (direction == DIRECTION_DOWN) {
                    triangle.rotation = 0f
                } else {
                    triangle.rotation = 180f
                }

                val minLeftMargin = cornerMargin + shadowMargin
                val maxLeftMargin = viewWidth - (cornerMargin + shadowMargin) - triangleLength
                var leftMargin = location[0] - x - triangleLength / 2
                if (leftMargin < minLeftMargin) {
                    leftMargin = minLeftMargin
                } else if (leftMargin > maxLeftMargin) {
                    leftMargin = maxLeftMargin
                }

                triLayoutParams.setMargins(leftMargin, 0, 0, 0)
                triLayoutParams.gravity =
                    Gravity.LEFT // user Gravity.LEFT not Gravity.START to adjust RTL layout
            }
            DIRECTION_END, DIRECTION_START -> {
                // 1. Calculate PopupWindow position
                viewHeight += shadowMargin * 2
                viewWidth += shadowMargin

                x =
                    if ((direction == DIRECTION_END && !isRightToLeft) || (direction == DIRECTION_START && isRightToLeft)) {
                        location[0] - triangleLength / 2
                    } else {
                        location[0] - viewWidth - triangleLength / 2
                    }

                y = location[1] - viewHeight / 2
                if (y < 0) {
                    y = 0
                }

                if (y + viewHeight > screenHeight) {
                    y = screenHeight - viewHeight
                }

                // 2. Adjust LinearLayout LayoutParams
                linearLayout.orientation = LinearLayout.HORIZONTAL
                if (direction == DIRECTION_END) {
                    linearLayout.addView(triangle, 0)

                    containerLayoutParams.marginStart = 0
                    containerLayoutParams.marginEnd = shadowMargin

                    // No way to programmatically change animation: https://stackoverflow.com/questions/13873412/android-how-to-set-windowanimation-programmatically
                    animationStyle =
                        if (isRightToLeft) R.style.avc_pop_animation_right else R.style.avc_pop_animation_left
                } else {
                    linearLayout.addView(triangle)

                    containerLayoutParams.marginStart = shadowMargin
                    containerLayoutParams.marginEnd = 0

                    animationStyle =
                        if (isRightToLeft) R.style.avc_pop_animation_left else R.style.avc_pop_animation_right
                }
                containerLayoutParams.topMargin = shadowMargin
                containerLayoutParams.bottomMargin = shadowMargin

                // 3. Adjust triangle shape and margin
                if ((direction == DIRECTION_END && !isRightToLeft) || (direction == DIRECTION_START && isRightToLeft)) {
                    triangle.rotation = 270f
                } else {
                    triangle.rotation = 90f
                }

                val minTopMargin = cornerMargin + shadowMargin
                val maxTopMargin = viewHeight - (cornerMargin + shadowMargin) - triangleLength
                var topMargin = location[1] - y - triangleLength / 2
                if (topMargin < minTopMargin) {
                    topMargin = minTopMargin
                } else if (topMargin > maxTopMargin) {
                    topMargin = maxTopMargin
                }

                triLayoutParams.setMargins(0, topMargin, 0, 0)
                triLayoutParams.gravity = Gravity.TOP
            }
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            triangle.outlineProvider = TriangleOutline(triangleLength.toFloat())
        }

        triangle.layoutParams = triLayoutParams
        container.layoutParams = containerLayoutParams

        showAtLocation(activity.window.decorView, Gravity.START or Gravity.TOP, x, y)

        contentView.isFocusable = true
        contentView.post {
            contentView.requestFocus()

            contentView.setOnClickListener { dismiss() }
        }
    }

    companion object {
        const val TIP_TYPE_MICROPHONE = 0x01
        const val TIP_TYPE_CAMERA = 0x02
        const val TIP_TYPE_MEDIA = 0x03

        const val DIRECTION_UP = 0
        const val DIRECTION_DOWN = 1
        const val DIRECTION_END = 2
        const val DIRECTION_START = 3
    }
}
