package io.agora.avc.widget.helper

import android.app.Activity
import android.app.PictureInPictureParams
import android.os.Build
import android.util.Rational
import android.view.Surface
import androidx.annotation.RequiresApi
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleObserver
import io.agora.avc.bo.VideoStats
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.config.RESOLUTION_LOW
import io.agora.avc.config.RESOLUTION_MEDIUM
import io.agora.avc.extensions.getVideoStats
import io.agora.avc.extensions.isMySelf
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.ScreenUtils
import io.agora.logger.LogConverter
import io.agora.logger.Logger

class PictureInPictureHelper constructor(
    private var activity: FragmentActivity? = null,
    private var callback: Callback? = null
) : LifecycleObserver {


    var isInPictureInPictureMode: Boolean = false
        set(value) {
            field = value
            Logger.i(TAG, "in picture in picture mode:$value")
        }

    /**
     * 'true' I can enter the picture-in-picture mode
     */
    var enable = true
        set(value) {
            field = value
            Logger.i(TAG, "picture in picture enable:${LogConverter.enable(value)}")
        }

    fun enterPictureInPicture(): Boolean {
        var success = false
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && DeviceUtils.isSupportPicInPic()) {
            try {
                activity?.let { activity ->
                    PictureInPictureParams.Builder().apply {
                        setAspectRatio(getCurrentVideoRational())
                        success = activity.enterPictureInPictureMode(build())
                        isInPictureInPictureMode = true
                    }
                }
            } catch (e: IllegalStateException) {
                Logger.e(TAG, "start picture in picture model error!", e)
            }
        }
        return success
    }

    fun changeAspectRatioInPicMode(user: ARoomUser) {
        if (!isInPictureInPictureMode || !user.videoState) {
            return
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && DeviceUtils.isSupportPicInPic()) {
            activity?.let {
                val params = PictureInPictureParams.Builder().apply {
                    setAspectRatio(getCurrentVideoRational())
                }
                setPictureInPictureParams(params.build())
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.O)
    private fun setPictureInPictureParams(params: PictureInPictureParams) {
        activity?.setPictureInPictureParams(params)
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    private fun getCurrentVideoRational(): Rational {
        val maxMultiple = 2
        var ret = Rational(360, 640)
        callback?.getMainScreenUser()?.let { user ->
            if (!user.videoState) {
                return@let
            }
            ret = if (user.isMySelf() && activity != null) {
                getLocalVideoRational(activity!!)
            } else {
                getRemoteVideoRational(user.getVideoStats())
            }
            if (ret.toDouble() > maxMultiple) {
                ret = Rational(maxMultiple, 1)
            } else if (ret.toDouble() < 1f / maxMultiple) {
                ret = Rational(1, maxMultiple)
            }
        }
        Logger.i(
            TAG,
            "Pic in pic mode, screen width:${ret.numerator} screen height:${ret.denominator}"
        )

        return ret
    }


    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    private fun getLocalVideoRational(activity: Activity): Rational {
        val rotation = ScreenUtils.getScreenRotation(activity)
        val reverse = rotation == Surface.ROTATION_0 || rotation == Surface.ROTATION_180
        return when (callback?.getResolutionOption()) {
            RESOLUTION_LOW -> if (reverse) Rational(360, 640) else Rational(640, 360)
            RESOLUTION_MEDIUM -> if (reverse) Rational(480, 640) else Rational(640, 480)
            else -> if (reverse) Rational(720, 960) else Rational(960, 720)
        }
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun getRemoteVideoRational(videoStats: VideoStats): Rational {
        val rotation = videoStats.rotation
        val reverse = rotation == Surface.ROTATION_0 || rotation == Surface.ROTATION_180

        if (videoStats.width == 0 || videoStats.height == 0) {
            return Rational(360, 640)
        }

        return if (reverse) {
            Rational(videoStats.width, videoStats.height)
        } else {
            Rational(videoStats.height, videoStats.width)
        }
    }

    fun onPictureInPictureModeChanged(isInPictureInPictureMode: Boolean) {
        Logger.i(
            TAG,
            "onPictureInPictureModeChanged: open = $isInPictureInPictureMode"
        )
        this.isInPictureInPictureMode = isInPictureInPictureMode
    }

    fun onDestroy() {
        activity = null
        callback = null
    }

    interface Callback {
        fun getResolutionOption(): Int
        fun getMainScreenUser(): ARoomUser?
    }

    companion object {
        private const val TAG = "[UI][PictureInPicture]"
    }
}
