package io.agora.avc.widget.pop

import android.os.SystemClock
import android.text.TextUtils
import io.agora.avc.base.AppContainer
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.EventObserver
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.MeetingNotification
import io.agora.avc.bo.MeetingNotificationType
import io.agora.avc.bo.NoticeCode
import io.agora.avc.config.KEY_REQUEST_ID
import io.agora.avc.manager.notice.NoticeType
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.Observer
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.util.*
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.TimeUnit

class NotificationPopViewModel(
    val appContainer: AppContainer,
) {

    val notificationLiveData = EventLiveData<CopyOnWriteArrayList<MeetingNotification>>()
    private val notificationList = CopyOnWriteArrayList<MeetingNotification>()

    private var disappearDisposable: Disposable? = null

    private var roomInfo = appContainer.room?.copy()

    private val eventObserver: EventObserver by lazy {
        object : EventObserver {
            override fun update(o: java.util.Observable?, arg: MessageEvent) {
                onEventReceived(arg)
            }
        }
    }

    init {
        appContainer.appController?.registerObserver(eventObserver, getUIEvents())
    }

    fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.NOTIFY_EVENT.ordinal -> {
                val notification = arg.obj
                if (notification is MeetingNotification) {
                    Logger.i(TAG, "notification changed event, notification:$notification")
                    if (notification.show) {
                        put(notification)
                    } else {
                        val requestId = notification.data?.getString(KEY_REQUEST_ID)
                        when {
                            !TextUtils.isEmpty(requestId) -> {
                                remove(requestId!!)
                            }
                            notification.code != NoticeCode.NOT_CODE -> {
                                remove(notification.code)
                            }
                            notification.group != null -> {
                                notification.group.forEach {
                                    remove(it)
                                }
                            }
                            else -> {
                                remove(notification.notificationType)
                            }
                        }
                    }
                }
            }
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal -> {
                (arg.obj as? Boolean)?.let {
                    if (it) {
                        remove(MeetingNotificationType.NO_8)
                    }
                }
            }
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                //I'm give up host, need hide request apply assistant notification
                if (roomInfo?.hostUid == appContainer.localUser?.uid
                    && appContainer.room?.hasHost() != true
                ) {
                    remove(NoticeCode.CODE_AS_ROOM_ASK_HOST_CN)
                    remove(NoticeCode.CODE_AS_ROOM_ASK_HOST_EN)
                }
                roomInfo = appContainer.room?.copy()
            }
            else -> {
            }
        }
    }

    fun getItem(position: Int): MeetingNotification? {
        if (position in 0 until position) {
            return notificationList[position]
        }
        return null
    }

    fun put(meetingNotification: MeetingNotification) {
        meetingNotification.showTime = SystemClock.elapsedRealtime()

        var hasTheSameType = false
        notificationList.forEachIndexed { index, item ->
            if (item.notificationType == meetingNotification.notificationType) {
                notificationList[index] = meetingNotification
                hasTheSameType = true
                return@forEachIndexed
            }
        }
        if (!hasTheSameType) {
            notificationList.add(meetingNotification)
        }

        if (!disappearTaskIsRunning()) {
            runDisappearTask()
        }

        postNotification()
    }

    fun remove(meetingNotificationType: Int, notify: Boolean = true) {
        notificationList.forEach {
            if (it.notificationType == meetingNotificationType) {
                notificationList.remove(it)
                return@forEach
            }
        }
        if (notify) {
            postNotification()
        }
    }

    fun removeAt(position: Int, notify: Boolean = true) {
        if (position in 0 until notificationList.size) {
            notificationList.removeAt(position)
            if (notify) {
                postNotification()
            }
        }
    }

    fun remove(code: NoticeCode, notify: Boolean = true) {
        notificationList.forEach {
            if (it.code == code) {
                notificationList.remove(it)
                return@forEach
            }
        }
        if (notify) {
            postNotification()
        }
    }

    fun remove(requestId: String, notify: Boolean = true) {
        notificationList.forEach {
            if (TextUtils.equals(it.data?.getString(KEY_REQUEST_ID), requestId)) {
                notificationList.remove(it)
                return@forEach
            }
        }
        if (notify) {
            postNotification()
        }
    }

    private fun runDisappearTask() {
        Observable
            .interval(0, 1, TimeUnit.SECONDS)
            .subscribeOn(Schedulers.io())
            .subscribe(
                object : Observer<Long> {
                    override fun onComplete() {

                    }

                    override fun onSubscribe(d: Disposable) {
                        disappearDisposable = d
                    }

                    override fun onNext(t: Long) {
                        notificationList.forEach {
                            val timeOut = if (it.noticeType == NoticeType.NOTIFICATION_C) {
                                MeetingNotification.NOTICE_TYPE_C_TIME_OUT
                            } else {
                                MeetingNotification.NOTICE_TYPE_AB_TIME_OUT
                            }
                            if (SystemClock.elapsedRealtime() - it.showTime > timeOut) {
                                notificationList.remove(it)
                            }
                        }

                        if (notificationList.isEmpty()) {
                            disappearDisposable?.dispose()
                            disappearDisposable = null
                        }

                        postNotification()
                    }

                    override fun onError(e: Throwable) {
                    }
                }
            )
    }

    private fun disappearTaskIsRunning() = disappearDisposable != null

    private fun postNotification() {
        notificationLiveData.postValue(CopyOnWriteArrayList(notificationList))
    }

    fun getUIEvents(): Array<AppEvent> {
        return arrayOf(
            AppEvent.NOTIFY_EVENT,
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT,
            AppEvent.ROOM_INFO_CHANGED,
        )
    }

    fun onDestroy() {
        appContainer.appController?.unregisterObserver(eventObserver, getUIEvents())
    }

    companion object {
        const val TAG = "[VM][NotificationPop]"
    }
}