/*
 * This source file is part of the Amy open source project.
 * For more information see github.com/AmyAssist
 * 
 * Copyright (c) 2018 the Amy project authors.
 *
 * SPDX-License-Identifier: Apache-2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For more information see notice.md
 */

package io.github.amyassist.amy.core.taskscheduler.api;

import java.time.Instant;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

/**
 * Service to schedule and execute concurrent tasks.
 * 
 * @author Leon Kiefer
 */
public interface TaskScheduler {
	/**
	 * Executes the given command at some time in the future.
	 * 
	 * @param runnable
	 *            The runnable to run in the future
	 */
	void execute(Runnable runnable);

	/**
	 * Schedules the given task to execute at the given time
	 * 
	 * @param task
	 *            the task to execute
	 * @param instant
	 *            The instant at which to execute that task
	 * @return the reference to the scheduled task
	 */
	@Nonnull
	ScheduledFuture<?> schedule(Runnable task, Instant instant);

	/**
	 * Schedules the given task to execute at the given delay
	 * 
	 * @param task
	 *            the task to execute
	 * @param delay
	 *            the delay
	 * @param timeUnit
	 *            the time unit of the delay
	 * @return the reference to the scheduled task
	 */
	@Nonnull
	ScheduledFuture<?> schedule(Runnable task, long delay, TimeUnit timeUnit);
}
