package io.goeasy.manager;

import com.google.gson.Gson;
import io.goeasy.presence.PresencesResponse;
import io.goeasy.publish.GoEasyError;
import io.goeasy.publish.GoEasyErrorCode;
import io.goeasy.ssl.PubSubX509TrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.*;
import java.util.Collection;
import java.util.Map;

/**
 * @author Chuanbao
 */
public abstract class RestAccessManager {
    private Logger log = LoggerFactory.getLogger(RestAccessManager.class);
    protected String appkey;
    protected String restUrl;
    private int maxTries = 5;
    private static final int CONNECT_TIMEOUT=3*1000;
    private static final int READ_TIMEOUT=3*1000;
    private Gson gson = new Gson();

    public RestAccessManager(String appkey, String url) {
        this.appkey = appkey;
        this.restUrl = url;
    }

    public void execute(Map<String,Object> paramsMap, RestAccessListener restAccessListener) {
        String params = null;
        try {
            params = buildParameters(paramsMap);
        } catch (UnsupportedEncodingException e) {
            log.error("Bad request parameters",e);
            restAccessListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), GoEasyErrorCode.PARAMETER_ERROR.content()));
            return;
        }
        String accessUrl = buildAccessUrl(params);
        log.debug("restUrl:{}", accessUrl);
        for (int i = 1; i <= maxTries; i++) {
            BufferedReader in = null;
            try {
                URL url = new URL(accessUrl);
                HttpURLConnection conn = (HttpURLConnection) url.openConnection();
                conn.setRequestMethod(method());
                setDoOutput(conn);
                conn.setDoOutput(false);
                conn.setDoInput(true);
                conn.setUseCaches(false);
                conn.setRequestProperty("Accept-Charset", "utf-8");
                conn.setRequestProperty("contentType", "utf-8");
                conn.setConnectTimeout(CONNECT_TIMEOUT);
                conn.setReadTimeout(READ_TIMEOUT);
                if (restUrl.startsWith("https://")) {
                    System.setProperty("https.protocols", "TLSv1");
                    TrustManager[] tm = {new PubSubX509TrustManager()};
                    SSLContext sslContext = SSLContext.getInstance("SSL", "SunJSSE");
                    sslContext.init(null, tm, new java.security.SecureRandom());
                    SSLSocketFactory ssf = sslContext.getSocketFactory();
                    HttpsURLConnection httpsConn = (HttpsURLConnection) conn;
                    httpsConn.setSSLSocketFactory(ssf);
                }
                conn.connect();
                int responseCode = conn.getResponseCode();
                if(responseCode == 200){
                    StringBuilder result = new StringBuilder();
                    in = new BufferedReader(new InputStreamReader(conn.getInputStream(), "UTF-8"));
                    String line;
                    while ((line = in.readLine()) != null) {
                        result.append(line);
                    }
                    String resultStr = result.toString();
                    PresencesResponse presencesResponse = gson.fromJson(resultStr, PresencesResponse.class);
                    restAccessListener.onSuccess(presencesResponse.getContent());
                }else{
                    in = new BufferedReader(new InputStreamReader(conn.getErrorStream()));
                    String result = null;
                    String temp = in.readLine();
                    while (temp != null) {
                        if (result != null)
                            result += temp;
                        else
                            result = temp;
                        temp = in.readLine();
                    }
                    GoEasyError goEasyError = gson.fromJson(result, GoEasyError.class);
                    restAccessListener.onFailed(goEasyError);
                }
                return;
            } catch (ConnectException e){
                onFailed(i, restAccessListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT, e);
            } catch (SocketException e) {
                onFailed(i, restAccessListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT ,e);
            } catch (SocketTimeoutException e) {
                onFailed(i, restAccessListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT ,e);
            } catch (UnknownHostException e) {
                onFailed(i, restAccessListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT ,e);
            } catch (Exception e) {
                onFailed(i, restAccessListener, GoEasyErrorCode.UNKNOWN_ERROR, e);
            }finally {
                try{
                    if(in!=null){
                        in.close();
                    }
                }
                catch(IOException ex){
                    ex.printStackTrace();
                }
            }
        }
    }

    private void onFailed(int triedTimes, RestAccessListener restAccessListener, GoEasyErrorCode error, Exception e) {
        if(triedTimes == maxTries){
            log.error("Finally call presence still failed[code:{},message:{}] after tried max times:{}", error.code(), error.content(),maxTries, e);
            restAccessListener.onFailed(new GoEasyError(error.code(), error.content()));
        }else{
            log.debug("Failed[code:{},message:{}] to fetch presence after tried {} times", error.code(), error.content(), triedTimes);
        }
    }

    protected String buildParameters(Map<String,Object> params) throws UnsupportedEncodingException {
        StringBuilder builder = new StringBuilder();
        builder.append("appkey="+appkey);
        if(params != null){
            for (String key : params.keySet()) {
                Object value = params.get(key);
                if(value instanceof Collection){
                    Collection valueConllection = (Collection)value;
                    for (Object collectionValue : valueConllection) {
                        appendParam(builder, key, collectionValue);
                    }
                }else{
                    appendParam(builder, key, value);
                }
            }
        }
        return builder.toString();
    }

    private void appendParam(StringBuilder builder, String key, Object value) throws UnsupportedEncodingException {
        if(value instanceof String){
            String strValue = (String)value;
            String encodedStrValue = URLEncoder.encode(strValue, "utf-8");
            builder.append("&"+key+"="+encodedStrValue);
        }else{
            builder.append("&"+key+"="+value);
        }
    }

    protected String buildAccessUrl(String params) {
        return restUrl;
    }

    protected abstract String method();

    protected void setDoOutput(HttpURLConnection conn){
        conn.setDoOutput(false);
    }

    public void setHttps() {
        this.restUrl = this.restUrl.replace("http://","https://");
    }
}
