package io.goeasy.publish;

import com.google.gson.Gson;
import io.goeasy.GoEasy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.*;
import java.io.*;
import java.net.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.UUID;

/**
 * @author Chuanbao
 */
public class Publisher {
    private static Logger log = LoggerFactory.getLogger(Publisher.class);
    private String appkey;
    private String PUBLISH_URL;
    private static final int CONNECT_TIMEOUT=3*1000;
    private static final int READ_TIMEOUT=3*1000;
    private int maxRetries = 5;
    private Gson gson = new Gson();

    public Publisher(String regionHost, String appkey) {
        this.appkey = appkey;
        boolean hasProtocol = regionHost.startsWith("http");
        if(!hasProtocol){
            regionHost = "http://"+regionHost;
        }
        this.PUBLISH_URL = regionHost+"/v2/pubsub/publish";
    }
    public void publish(String messageId, String channel, String content, String notificationTitle, String notificationBody, PublishListener publishListener) {
        if(isEmpty(channel)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Channel is required"));
            return;
        }
        if (isEmpty(content)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Content is required"));
            return;
        }else if(content.length() > 2500){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Content over max length 2500"));
            return;
        }
        if(isEmpty(notificationTitle)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Notification title is required"));
            return;
        }else if(notificationTitle.length() > 32){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Notification title is over max length 32"));
            return;
        }
        if(isEmpty(notificationBody)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Notification body is required"));
            return;
        }else if(notificationBody.length() > 50){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Notification body is over max length 100"));
            return;
        }
        doPublish(messageId, channel, content, false, notificationTitle, notificationBody, publishListener);
    }
    /**
     * This method is different from publish :
     * basic authentication not used any more, the publish key will be send as a post property as normal form data
     *  @param channel
     * @param content
     * @param retained that means the message will be retained,
     *                 for this kind of message, no matter when
     *                 i.e: Message A is a retained message and published on 15-8-2015 09:30:20
     * @param publishListener
     */
    public void publish(String messageId, String channel, String content, boolean retained, PublishListener publishListener) {
        if(isEmpty(channel)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Channel is required"));
            return;
        }
        if (isEmpty(content)){
            publishListener.onFailed(new GoEasyError(GoEasyErrorCode.PARAMETER_ERROR.code(), "Content is required"));
            return;
        }
        doPublish(messageId, channel, content, retained, null, null, publishListener);
    }

    private void doPublish(String messageId, String channel, String content, boolean retained, String notificationTitle, String notificationBody, PublishListener publishListener) {
        int retried = 0;
        int responseCode = GoEasyErrorCode.UNKNOWN_ERROR.code();
        boolean success = false;
        String guid = messageId;
        if(guid == null || guid.trim().length()==0){
            guid = UUID.randomUUID().toString();
        }
        while(!success && retried <= maxRetries){
            synchronized (guid){
                try {
                    if (retried > 0) {
                        log.debug("retring {} times", retried);
                        guid.wait(70);
                    }
                } catch (InterruptedException e) {
                    log.error("Delay publish[{}] error.",guid);
                }
            }
            try {
                URL url = new URL(PUBLISH_URL);

                HttpURLConnection conn = (HttpURLConnection) url.openConnection();
                conn.setDoInput(true);
                conn.setDoOutput(true);
                conn.setUseCaches(false);
                conn.setRequestMethod("POST");
                conn.setRequestProperty("Accept-Charset", "utf-8");
                conn.setRequestProperty("contentType", "utf-8");
                conn.setConnectTimeout(CONNECT_TIMEOUT);
                conn.setReadTimeout(READ_TIMEOUT);
                if (PUBLISH_URL.startsWith("https://")) {
                    System.setProperty("https.protocols", "TLSv1");
                    TrustManager[] tm = {new PubSubX509TrustManager()};
                    SSLContext sslContext = SSLContext.getInstance("SSL", "SunJSSE");
                    sslContext.init(null, tm, new java.security.SecureRandom());
                    SSLSocketFactory ssf = sslContext.getSocketFactory();
                    HttpsURLConnection httpsConn = (HttpsURLConnection) conn;
                    httpsConn.setSSLSocketFactory(ssf);
                }
                String encodedChannel = URLEncoder.encode(channel,"utf-8");
                String encodedContent = URLEncoder.encode(content,"utf-8");
                String input = "channel=" + encodedChannel + "&content=" + encodedContent + "&retained=" + retained + "&appkey=" + appkey+"&guid="+guid+"&retried="+retried +"&artifactVersion="+ GoEasy.artifactVersion;
                if(!isEmpty(notificationTitle)){
                    String encodedTitle = URLEncoder.encode(notificationTitle, "utf-8");
                    String encodedBody = URLEncoder.encode(notificationBody, "utf-8");
                    input = input+"&notification_title="+encodedTitle+"&notification_body="+encodedBody;
                }
                OutputStream os = conn.getOutputStream();
                os.write(input.getBytes("utf-8"));
                os.flush();

                responseCode = conn.getResponseCode();
                showResult(conn, responseCode, publishListener);

                conn.disconnect();
                success = true;
            } catch (ConnectException e) {
                onFailed(retried, publishListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT ,e);
            } catch (SocketException e) {
                onFailed(retried, publishListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT, e);
            } catch (SocketTimeoutException e){
                onFailed(retried, publishListener, GoEasyErrorCode.UNREACHABLE_TIMEOUT, e);
            } catch (UnknownHostException e){
                onFailed(retried, publishListener, 408, "java.net.UnknownHostException: "+PUBLISH_URL, e);
            } catch (IOException e){
                onFailed(retried, publishListener, responseCode, "java.io.IOException", e);
            } catch (Exception e) {
                onFailed(retried, publishListener, responseCode, e.getMessage(), e);
            }finally {
                retried++;
            }
        }
    }

    private void onFailed(int retried, PublishListener publishListener, GoEasyErrorCode errorCode, Exception e) {
        if(retried == maxRetries){
            log.error("Publish finally still failed code[{}] with message[{}] after tried {} times", errorCode.code(), errorCode.content(), maxRetries+1, e);
            publishListener.onFailed(new GoEasyError(errorCode.code(),errorCode.content()));
        }else{
            log.debug("Publish failed code[{}] with message[{}] after tried {} times", errorCode.code(), errorCode.content(), retried+1, e);
        }
    }

    private void onFailed(int retried, PublishListener publishListener, int customCode, String customMessage, Exception e) {
        if(retried == maxRetries){
            log.error("Publish finally still failed code[{}] with message[{}] after tried {} times", customCode, customMessage, maxRetries+1, e);
            publishListener.onFailed(new GoEasyError(customCode,customMessage));
        }else{
            log.debug("Publish failed code[{}] with message[{}] after tried {} times", customCode, customMessage, retried+1, e);
        }
    }



    private void showResult(HttpURLConnection conn, int responseCode, PublishListener publishListener) throws IOException {
        if(responseCode == 200){
            publishListener.onSuccess();
        }else {
            String result = null;
            InputStream errorStream = conn.getErrorStream();
            if(errorStream != null){
                BufferedReader in = new BufferedReader(new InputStreamReader(errorStream));
                String temp = in.readLine();
                while (temp != null) {
                    if (result != null)
                        result += temp;
                    else
                        result = temp;
                    temp = in.readLine();
                }
            }else{
                InputStream inputStream = conn.getInputStream();
                if(inputStream != null){
                    BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
                    String temp = in.readLine();
                    while (temp != null) {
                        if (result != null)
                            result += temp;
                        else
                            result = temp;
                        temp = in.readLine();
                    }
                }else{
                    publishListener.onFailed(new GoEasyError(responseCode, "Input stream and error stream are both null"));
                    return;
                }
            }
            GoEasyError goEasyError = gson.fromJson(result, GoEasyError.class);
            publishListener.onFailed(goEasyError);
        }
    }

    private boolean isEmpty(String text){
        return text == null || text.trim().length()==0;
    }

    public void setHttps(boolean https) {
        if(https){
            this.PUBLISH_URL = this.PUBLISH_URL.replace("http://","https://");
        }
    }

    class PubSubX509TrustManager implements X509TrustManager {

        public void checkClientTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {

        }

        public void checkServerTrusted(X509Certificate[] x509Certificates, String s) throws CertificateException {

        }

        public X509Certificate[] getAcceptedIssuers() {
            return null;
        }
    }

    public void setPublisUrl(String url){
        this.PUBLISH_URL = url;
    }
}
