package io.kotest.property.arbitrary

import io.kotest.property.Gen
import io.kotest.property.RandomSource
import io.kotest.property.Sample
import io.kotest.property.Shrinker
import io.kotest.property.ShrinkingMode
import io.kotest.property.map
import io.kotest.property.sampleOf

/**
 * An [Arb] (for arbitrary) is a provider of property test data in two categories: edgecases and samples.
 *
 * Edge cases are values that are a common source of bugs. For example, a function using ints is
 * more likely to fail for common edge cases like zero, minus 1, positive 1, [Int.MAX_VALUE] and [Int.MIN_VALUE]
 * rather than random values like 965489. Therefore, edge cases can be included in sequences
 * generated by an arbitrary.
 *
 * Not all arbitraries will utilize edge cases. For example, if you define an integer generator
 * using a subset of the number space - say from 100 to 250,000 - then no edge cases are provided.
 *
 * Samples are chosen randomly from the sample space and are used to give a greater breadth to
 * the test cases. For example, in the case of a function using integers, these random values
 * could be from across the entire integer number line, or could be limited to a subset of ints
 * such as natural numbers or even numbers.
 *
 * In order to use an [Arb] inside a property test, one must invoke the [take] method, passing in the
 * number of iterations required and optionally a [ShrinkingMode].
 */
interface Arb<A> : Gen<A> {

   /**
    * Edgecase values for this type A.
    */
   fun edgecases(): List<A>

   fun samples(rs: RandomSource): Sequence<Sample<A>>

   override fun minIterations(): Int = 0

   override fun generate(rs: RandomSource): Sequence<Sample<A>> {
      return edgecases().asSequence().map { Sample(it) } + samples(rs)
   }

   companion object
}

interface BasicArb<A> : Arb<A> {
   fun sample(rs: RandomSource): Sample<A>
   override fun samples(rs: RandomSource): Sequence<Sample<A>> = generateSequence { sample(rs) }
}

/**
 * Returns a sequence of size [count] from values generated from this arb.
 */
fun <A> Arb<A>.take(count: Int, rs: RandomSource = RandomSource.Default): Sequence<A> =
   generate(rs).take(count).map { it.value }

/**
 * Returns a single value generated from this arb ignoring edgecases.
 */
fun <A> Arb<A>.single(rs: RandomSource = RandomSource.Default): A = samples(rs).first().value
fun <A> Arb<A>.next(rs: RandomSource = RandomSource.Default): A = single(rs)

/**
 * Creates a new [Arb] that performs no shrinking, and generates each value
 * from successive invocations of the given function f.
 */
fun <A> arb(edgecases: List<A> = emptyList(), f: (RandomSource) -> A) = object : Arb<A> {
   override fun edgecases(): List<A> = edgecases
   override fun samples(rs: RandomSource): Sequence<Sample<A>> = generateSequence { Sample(f(rs)) }
}

/**
 * Creates a new [Arb] that performs shrinking using the supplied shrinker and generates each value
 * from successive invocations of the given function f.
 */
fun <A> arb(
   shrinker: Shrinker<A>,
   f: (RandomSource) -> A
) = object : Arb<A> {
   override fun edgecases(): List<A> = emptyList()
   override fun samples(rs: RandomSource): Sequence<Sample<A>> = generateSequence { sampleOf(f(rs), shrinker) }
}

/**
 * Creates a new [Arb] that performs shrinking using the supplied shrinker and generates each value
 * from successive invocations of the given function f.
 */
fun <A> arb(
   shrinker: Shrinker<A>,
   edgecases: List<A> = emptyList(),
   f: (RandomSource) -> A
) = object : Arb<A> {
   override fun edgecases(): List<A> = edgecases
   override fun samples(rs: RandomSource): Sequence<Sample<A>> = generateSequence { sampleOf(f(rs), shrinker) }
}

/**
 * Returns a new [Arb] which takes its elements from the receiver and filters them using the supplied
 * predicate. This gen will continue to request elements from the underlying gen until one satisfies
 * the predicate.
 */
fun <A> Arb<A>.filter(predicate: (A) -> Boolean) = object : Arb<A> {
   override fun edgecases(): List<A> = this@filter.edgecases().filter(predicate)
   override fun samples(rs: RandomSource): Sequence<Sample<A>> =
      this@filter.samples(rs).filter { predicate(it.value) }
}

/**
 * @return a new [Arb] by filtering this arbs output by the negated function [f]
 */
fun <A> Arb<A>.filterNot(f: (A) -> Boolean): Arb<A> = filter { !f(it) }

/**
 * Returns a new [Arb] which takes its elements from the receiver and maps them using the supplied function.
 */
fun <A, B> Arb<A>.map(f: (A) -> B): Arb<B> = object : Arb<B> {
   override fun edgecases(): List<B> = this@map.edgecases().map(f)
   override fun samples(rs: RandomSource): Sequence<Sample<B>> {
      return this@map.samples(rs).map { Sample(f(it.value), it.shrinks.map(f)) }
   }
}

/**
 * Returns a new [Arb] which takes its elements from the receiver and flat maps them using the supplied function.
 */
fun <A, B> Arb<A>.flatMap(f: (A) -> Arb<B>): Arb<B> = object : Arb<B> {
   override fun edgecases(): List<B> = this@flatMap.edgecases().flatMap { f(it).edgecases() }
   override fun samples(rs: RandomSource): Sequence<Sample<B>> {
      return this@flatMap.samples(rs).flatMap { f(it.value).generate(rs) }
   }
}

/**
 * Returns a new [Arb] which ensures only unique values are generated by keeping track of previously
 * generated values.
 *
 * Note: This arb can result in an infinite loop if more elements are requested than can be generated uniquely.
 */
fun <A> Arb<A>.distinct() = object : Arb<A> {
   override fun edgecases(): List<A> = this@distinct.edgecases()
   override fun samples(rs: RandomSource): Sequence<Sample<A>> {
      val seen = mutableSetOf<A>()
      return this@distinct.samples(rs).filter { seen.add(it.value) }
   }
}

fun <A> Arb.Companion.constant(a: A) = arb { a }
