package io.kotest.property.arbitrary

import io.kotest.property.Gen
import io.kotest.property.Shrinker
import kotlin.jvm.JvmOverloads
import kotlin.random.nextInt

/**
 * Returns an [Arb] whose values are chosen randomly from those in the supplied collection.
 * May not cover all items. If you want an exhaustive selection from the list, see [Exhaustive.collection]
 */
fun <T> Arb.Companion.element(collection: Collection<T>): Arb<T> = arb { collection.random(it.random) }

/**
 * Returns an [Arb] whose of values are a set of values generated by the given element generator.
 * The size of each set is determined randomly by the specified [range].
 *
 * Note: This may fail to terminate if the element generator cannot produce a large enough number of
 * unique items to satify the required set size
 */
@JvmOverloads
fun <A> Arb.Companion.set(gen: Gen<A>, range: IntRange = 0..100): Arb<Set<A>> = arb {
   val size = it.random.nextInt(range)
   val set = mutableSetOf<A>()
   gen.generate(it).takeWhile { set.size < size }.forEach { set.add(it.value) }
   set.toSet()
}

/**
 * Returns an [Arb] whose of values are a list of values generated by the given generator.
 * The size of each list is determined randomly by the specified [range].
 *
 * Shrinking is performed by removing elements from the list until the empty list.
 */
@JvmOverloads
fun <A> Arb.Companion.list(gen: Gen<A>, range: IntRange = 0..100): Arb<List<A>> = arb(ListShrinker()) { rs ->
   val size = rs.random.nextInt(range)
   gen.generate(rs).take(size).map { it.value }.toList()
}

class ListShrinker<A> : Shrinker<List<A>> {
   override fun shrink(value: List<A>): List<List<A>> {
      return if (value.isEmpty()) emptyList() else {
         listOf(
            emptyList(),
            value.takeLast(1),
            value.take(value.size / 3),
            value.take(value.size / 2),
            value.take(value.size * 2 / 3),
            value.dropLast(1)
         )
      }
   }
}
