package io.proxsee.sdk;

import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Handler;
import android.os.RemoteException;
import android.util.Log;

import com.android.volley.Response;
import com.android.volley.VolleyError;

import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.logging.LogManager;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import io.proxsee.sdk.misc.Utils;
import io.proxsee.sdk.model.Beacon;
import io.proxsee.sdk.model.CachedBeacon;
import io.proxsee.sdk.model.TaggedBeacon;
import io.proxsee.sdk.model.TagsChangedSet;
import io.proxsee.sdk.network.ServerAPI;
import io.proxsee.sdk.network.library.ResponseListener;

/**
 * Created by Ahmad Shami on 4/9/15.
 */
class BootstrapBeaconsMonitor implements RangeNotifier, BeaconConsumer {
    private final static String TAG = BootstrapBeaconsMonitor.class.getSimpleName();

    private Context mContext;
    private BeaconManager mBeaconManager;
    private BeaconDiscoveryListener mRegionStateListener;
    private Database mDatabase;
    private Handler mHandler;
    private Region masterRegion;
    private Set<Beacon> mLastSeenBeacons;
    private HashMap<String, TaggedBeacon> cachedTagsDB;
    private long mLastTimeFlushed;
    private ServerAPI mServerApi;
    private TagsChangedSet previousTags;
    private List<Region> regions;

    public BootstrapBeaconsMonitor(Context context, Database database, ServerAPI serverApi, BeaconDiscoveryListener listener){
        this.mContext = context;
        this.mDatabase = database;
        this.mServerApi = serverApi;
        this.mRegionStateListener = listener;
        this.mHandler = new Handler();
        this.cachedTagsDB = new HashMap<String, TaggedBeacon>();
        this.previousTags = new TagsChangedSet(-1000);
        initBeaconManager();
    }

    private void initBeaconManager(){
        this.mBeaconManager = BeaconManager.getInstanceForApplication(mContext);

        //Foreground scan settings
        mBeaconManager.setForegroundScanPeriod(Constants.BLUETOOTH_SCAN_PERIOD);
        mBeaconManager.setForegroundBetweenScanPeriod(Constants.BLUETOOTH_SCAN_BETWEEN_PERIOD);
        mBeaconManager.setBackgroundScanPeriod(Constants.BLUETOOTH_SCAN_PERIOD);
        mBeaconManager.setBackgroundBetweenScanPeriod(Constants.BLUETOOTH_SCAN_BETWEEN_PERIOD);

        mBeaconManager.setRangeNotifier(this);
        mBeaconManager.setBackgroundMode(false);
        mBeaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(Constants.IBEACONS_LAYOUT));
    }

    private synchronized Set<Beacon> getLastSeenBeacons(){
        if(mLastSeenBeacons == null) {

            Set<Beacon> beacons = new HashSet<Beacon>();
            for (CachedBeacon cachedBeacon : mDatabase.getAll(CachedBeacon.class)) {
                beacons.add(new Beacon(cachedBeacon.getMajor(), cachedBeacon.getMinor(), cachedBeacon.getLastSeenTime()));
            }

            mLastSeenBeacons = beacons;
        }

        return mLastSeenBeacons;
    }

    private void setLastSeenBeacons(Set<Beacon> beacons) {
        mLastSeenBeacons = beacons;

        Set<CachedBeacon> cachedBeacons = new HashSet<CachedBeacon>();

        for(Beacon beacon : beacons){
            CachedBeacon cb = new CachedBeacon();
            cb.setMajor(beacon.getMajor());
            cb.setMinor(beacon.getMinor());
            cb.setLastSeenTime(beacon.getLastSeen());
            cachedBeacons.add(cb);
        }

        mDatabase.set(CachedBeacon.class, cachedBeacons);
    }

    public void startMonitoring(List<Region> regions){
        Log.d(TAG, "Monitoring: " + Arrays.toString(regions.toArray()));

        this.regions = regions;
        mBeaconManager.bind(this);

        ProxSeeSDKManager.Log("Started monitoring: " + regions.size());

        Intent intent = new Intent(mContext, BackgroundService.class);
        mContext.startService(intent);
    }

    @Override
    public void onBeaconServiceConnect() {
        try {
            for (Region region : regions) {
                LogManager.d(TAG, "Region monitoring activated for region %s", region);
                mBeaconManager.startRangingBeaconsInRegion(region);
                if (mBeaconManager.isBackgroundModeUninitialized()) {
                    mBeaconManager.setBackgroundMode(true);
                }
            }
        } catch (RemoteException e) {
            LogManager.e(e, TAG, "Can't set up region monitoring");
        }
    }

    @Override
    public Context getApplicationContext() {
        return mContext;
    }

    @Override
    public void unbindService(ServiceConnection serviceConnection) {
        mContext.unbindService(serviceConnection);
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection serviceConnection, int i) {
        return mContext.bindService(intent, serviceConnection, i);
    }

    @Override
    public void didRangeBeaconsInRegion(final Collection<org.altbeacon.beacon.Beacon> seenBeacons, final Region region) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {

                Set<Beacon> beaconsAroundMe = convert(seenBeacons);
                Set<Beacon> lastSeenBeacons = getLastSeenBeacons();

                Log.v(TAG, "Beacons around me: " + beaconsAroundMe);

                Set<Beacon> discoveredBeacons = new HashSet<Beacon>();
                discoveredBeacons.addAll(beaconsAroundMe);
                discoveredBeacons.removeAll(lastSeenBeacons);

                if (!discoveredBeacons.isEmpty()) {

                    Log.v(TAG, "New beacons detected: " + discoveredBeacons);

                    for (Beacon cur : discoveredBeacons) {
                        Log.i(TAG, "BROADCAST ENTER REGION");
                        mRegionStateListener.onBeaconDiscovered(cur);
                    }

                    lastSeenBeacons.addAll(discoveredBeacons);
                }

                Set<Beacon> lostBeacons = new HashSet<Beacon>();
                for (Beacon beacon : lastSeenBeacons) {
                    if (beaconsAroundMe.contains(beacon)) {
                        beacon.setLastSeen(System.currentTimeMillis());
                        Log.i(TAG, "Beacon still visible, updating last seen time. lastSeen: " + beacon.getLastSeen());
                    } else if (System.currentTimeMillis() - beacon.getLastSeen() > Constants.BEACON_EXPIRY) {
                        Log.i(TAG, "Beacon lost for over " + Constants.BEACON_EXPIRY + "ms, dropping. lastSeen: " + beacon.getLastSeen());
                        lostBeacons.add(beacon);
                    } else {
                        Log.i(TAG, "Waiting " + Constants.BEACON_EXPIRY + "ms before dropping lost beacon. lastSeen: " + beacon.getLastSeen());
                    }
                }
                lostBeacons.removeAll(beaconsAroundMe);

                if (!lostBeacons.isEmpty()) {
                    Log.v(TAG, "Beacons missing: " + lostBeacons);

                    for (Beacon cur : lostBeacons) {
                        Log.i(TAG, "BROADCAST EXIT REGION");
                        mRegionStateListener.onBeaconLost(cur);
                    }

                    lastSeenBeacons.removeAll(lostBeacons);
                }

                setLastSeenBeacons(lastSeenBeacons);

                processTagChanges(lastSeenBeacons);

            }
        });
    }

    private void processTagChanges(final Set<Beacon> seenBeacons){
        TagsChangedSet previousTags = new TagsChangedSet(BootstrapBeaconsMonitor.this.previousTags);
        TagsChangedSet currentTags = new TagsChangedSet(Utils.getUtcInMillis());
        long time = System.currentTimeMillis() - mLastTimeFlushed;

        if(time > Constants.CACHE_EXPIRY) {
            cachedTagsDB.clear();
            mLastTimeFlushed = System.currentTimeMillis();
        }

        for(Beacon cur:seenBeacons){
            int major = cur.getMajor();
            int minor = cur.getMinor();

            String key = major + "," + minor;

            if(cachedTagsDB.containsKey(key)){
                TaggedBeacon taggedBeacon = cachedTagsDB.get(key);
                Collections.addAll(currentTags.getTags(), taggedBeacon.getTags());

            }else{
                mServerApi.getBeaconsAround(major, minor, Constants.BEACONS_AROUND_RADIUS, new ResponseListener<TaggedBeacon[]>() {
                    @Override
                    public void onResponse(int responseCode, TaggedBeacon[] taggedBeacons) {
                        for(TaggedBeacon cur:taggedBeacons){
                            int major = cur.getMajor();
                            int minor = cur.getMinor();
                            String key = major + "," + minor;

                            cachedTagsDB.put(key, cur);
                            Log.d(TAG, "TaggedBeacon: " + cur);
                        }

                        processTagChanges(seenBeacons);
                    }
                }, new Response.ErrorListener() {
                    @Override
                    public void onErrorResponse(VolleyError volleyError) {
                        Log.e(TAG, "Could not fetch nearby beacons", volleyError);
                    }
                });
                return;
            }
        }

        if(!previousTags.getTags().equals(currentTags.getTags())) {
            mRegionStateListener.onTagsChanged(previousTags, currentTags);
            this.previousTags = currentTags;
        }
    }

    private Set<Beacon> convert(Iterable<org.altbeacon.beacon.Beacon> beacons){
        Set<Beacon> list = new HashSet<Beacon>();
        for (org.altbeacon.beacon.Beacon cur : beacons) {
            list.add(new Beacon(cur.getId2().toInt(), cur.getId3().toInt(), System.currentTimeMillis()));
        }

        return list;
    }

    public interface BeaconDiscoveryListener {
        void onBeaconDiscovered(Beacon beacon);
        void onBeaconLost(Beacon beacon);
        void onTagsChanged(TagsChangedSet previousTags, TagsChangedSet currentTags);
    }
}
