package io.proxsee.sdk;

import android.app.Application;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.util.Base64;
import android.util.Log;
import android.widget.Toast;

import com.android.volley.Response;
import com.android.volley.VolleyError;

import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.Region;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import io.proxsee.sdk.misc.Utils;
import io.proxsee.sdk.model.Beacon;
import io.proxsee.sdk.model.BeaconNotificationObject;
import io.proxsee.sdk.model.MonitoringRegion;
import io.proxsee.sdk.model.TagsChangedSet;
import io.proxsee.sdk.network.ServerAPI;
import io.proxsee.sdk.network.library.ResponseListener;

/**
 * Created by Ahmad Shami on 3/24/15.
 */
public class ProxSeeSDKManager implements BootstrapBeaconsMonitor.BeaconDiscoveryListener {
    private static final String TAG = ProxSeeSDKManager.class.getSimpleName(); 
    public final static String VERSION = BuildConfig.CONCATINATED_VERSION_NAME;

    private static ProxSeeSDKManager instance;
    private static HashSet<ProxseeListener> mListeners = new HashSet<ProxseeListener>();
    private static boolean mLogging;


    public static void initialize(Application application, String apiKey){
        initialize(application, apiKey, null);
    }

    public static void initialize(Application application, String apiKey, String baseUrl){
        if(instance != null)
            throw new RuntimeException("ProxSee SDK is already launched");

        if(isAndroidOsCompatible()) {
            InternalCache.init(application);
            instance = new ProxSeeSDKManager(application, apiKey, baseUrl==null? Constants.DEFAULT_BASE_URL : baseUrl);
            instance.launch();
        }
    }

    public static void updateMetadata(HashMap<String, Object> metadata, final CompletionHandler completionHandler){
        if(isAndroidOsCompatible()) {
            throwExceptionIfNotLaunched();

            instance.updateMetadata(metadata, new ResponseListener<Void>() {
                @Override
                public void onResponse(int responseCode, Void model) {
                    completionHandler.onUpdateCompleted(true, null);
                }
            }, new Response.ErrorListener() {
                @Override
                public void onErrorResponse(VolleyError error) {
                    completionHandler.onUpdateCompleted(false, error);
                }
            });
        }
    }

    public static void setLoggingEnabled(Context context, boolean enabled){
        mLogging = enabled;
    }

    private static boolean isLoggingEnabled(){
        return mLogging;
    }

    public static void addListener(ProxseeListener listener){
        mListeners.add(listener);
    }

    public static void removeListener(ProxseeListener listener){
        mListeners.remove(listener);
    }

    private static void throwExceptionIfNotLaunched(){
        if(instance == null)
            throw new RuntimeException("ProxSee SDK is not launched");
    }

    private static boolean isAndroidOsCompatible(){
        if(Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2) {
            Log.e(TAG, "ProxSee SDK requires Android 4.3 and up");
            return false;
        }

        return true;
    }

    private Application mContext;
    private ServerAPI mServerAPI;
    private Database mDatabase;
    private List<MonitoringRegion> mRegions;
    private BootstrapBeaconsMonitor mRegionMonitor;
    private Handler mHandler;
    private final String UUID;

    private ProxSeeSDKManager(Application application, String apiKey, String baseUrl) {
        this.mContext = application;
        this.mDatabase = Database.open(mContext);
        this.mServerAPI = new ServerAPI(mContext, apiKey, baseUrl);
        this.mRegionMonitor = new BootstrapBeaconsMonitor(mContext, mDatabase, mServerAPI, this);
        this.UUID = Utils.getUniqueID(application);
        this.mHandler = new Handler();

        saveFingerprint(apiKey, baseUrl);
        this.mRegions = mDatabase.getAll(MonitoringRegion.class);

    }

    private void launch(){
        if(mRegions.isEmpty())
            fetchRegionsThenStartMonitoring();
        else
            startMonitoring();
    }

    private void fetchRegionsThenStartMonitoring(){
        Log("FETCHING REGIONS-STARTED");
        mServerAPI.fetchRegions(new ResponseListener<MonitoringRegion[]>() {
            @Override
            public void onResponse(int responseCode, MonitoringRegion[] model) {
                setRegions(model);
                startMonitoring();

                Log("Regions were fetched: " + model.length);
            }
        }, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError error) {
                Log("ERR Fetching regions");
            }
        });
    }

    private void setRegions(MonitoringRegion[] regions){
        this.mRegions = Arrays.asList(regions);
        mDatabase.set(MonitoringRegion.class, mRegions);

        for(MonitoringRegion cur: mRegions)
            Log(false, String.format("uuid=%s, major=%d", cur.getUuid(), cur.getMajor()));
    }

    public void updateMetadata(HashMap<String, Object> metadata, ResponseListener<Void> listener, Response.ErrorListener errorListener){
        mServerAPI.setMetadata(metadata, UUID, listener, errorListener);
    }

    public void startMonitoring() {
        Set<Region> set = new HashSet<Region>(mRegions.size());

        for (MonitoringRegion cur: mRegions) {
            Identifier uuid = Identifier.parse(cur.getUuid());
            Identifier major = Identifier.fromInt(cur.getMajor());

            set.add(new Region(uuid + ":" + major, uuid, major, null));
        }

        mRegionMonitor.startMonitoring(new ArrayList<Region>(set));
    }

    @Override
    public void onBeaconDiscovered(final Beacon region) {
        Log("EnterRegion: " + region);

        mServerAPI.checkIn(region, UUID, new ResponseListener<Void>() {

            @Override
            public void onResponse(int responseCode, Void model) {
                Log("Checkin request Succeeded");
            }
        }, new Response.ErrorListener() {

            @Override
            public void onErrorResponse(VolleyError error) {
                Log("Checkin request Failed: " + error.getClass());
            }
        });
    }

    @Override
    public void onBeaconLost(final Beacon beacon) {
        Log("ExitRegion: " + beacon);

        mServerAPI.checkOut(beacon, UUID, new ResponseListener<Void>() {

            @Override
            public void onResponse(int responseCode, Void model) {
                Log("Checkout request Succeeded");
            }
        }, new Response.ErrorListener() {

            @Override
            public void onErrorResponse(VolleyError error) {
                Log("Checkout request Failed: " + error.getClass());
            }
        });
    }

    @Override
    public void onTagsChanged(TagsChangedSet previousTags, TagsChangedSet currentTags) {
        final BeaconNotificationObject beaconNotificationObject = new BeaconNotificationObject(previousTags, currentTags);

        for(final ProxseeListener cur:mListeners){
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    cur.didChangeTagsSet(beaconNotificationObject);
                }
            });
        }
    }

    private void saveFingerprint(String apiKey, String baseUrl){
        String oldHash = InternalCache.getConfigFingerprint();
        String newHash = generateFingerprint(apiKey, baseUrl);

        if(!newHash.equals(oldHash)){
            Log.d(TAG, "CLEARING CACHE");
            mDatabase.clear(MonitoringRegion.class);
            InternalCache.clearAll();
        }

        InternalCache.setConfigHash(newHash);
    }

    private String generateFingerprint(String apiKey, String baseUrl) {
        try{
            byte[] raw = MessageDigest.getInstance("SHA-512").digest((apiKey + "-" + baseUrl).getBytes("UTF-8"));
            byte[] base64 = Base64.encode(raw, Base64.NO_WRAP);
            return new String(base64, "UTF-8");

        } catch (NoSuchAlgorithmException e) {
            Log.wtf(TAG, "MD5 Algorithm is not supported", e);

        } catch (UnsupportedEncodingException e) {
            Log.wtf(TAG, "UTF-8 encoding is not supported", e);
        }

        //this should never ever occur, SHA-512 Algorithm and UTF8 encoding are core essential parts of the Android system
        throw new RuntimeException("System is corrupted, cannot proceed");
    }

    public static interface ProxseeListener {
        public void didChangeTagsSet(BeaconNotificationObject beaconNotificationObject);
    }

    static void Log(String message){
        Log(true, message);
    }

    static void Log(boolean toast, String message){
        if(isLoggingEnabled()) {
            Log.d(TAG, message);

            if (toast)
                Toast.makeText(instance.mContext, message, Toast.LENGTH_SHORT).show();
        }
    }

    public static interface CompletionHandler{
        public void onUpdateCompleted(boolean success, Exception error);
    }
}