package io.proxsee.sdk;

import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Handler;
import android.os.RemoteException;
import android.util.Log;

import com.android.volley.Response;
import com.android.volley.ServerError;
import com.android.volley.VolleyError;

import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.logging.LogManager;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import io.proxsee.sdk.misc.Utils;
import io.proxsee.sdk.model.Beacon;
import io.proxsee.sdk.model.TaggedBeacon;
import io.proxsee.sdk.model.TagsChangedSet;
import io.proxsee.sdk.model.persist.PersistentBeacon;
import io.proxsee.sdk.network.ServerAPI;
import io.proxsee.sdk.network.library.ResponseListener;

/**
 * Created by Ahmad Shami on 4/9/15.
 */
class BootstrapBeaconsMonitor implements RangeNotifier, BeaconConsumer {
    private final static String TAG = BootstrapBeaconsMonitor.class.getSimpleName();

    private Context mContext;
    private BeaconManager mBeaconManager;
    private BeaconDiscoveryListener mRegionStateListener;
    private Database mDatabase;
    private Handler mHandler;
    private Set<Beacon> mLastSeenBeacons;
    private HashMap<String, Beacon> cachedTagsDB;
    private long mLastTimeFlushed;
    private ServerAPI mServerApi;
    private TagsChangedSet previousTags;
    private List<Region> regions;
    private GeofenceManager mGeofenceManager;

    public BootstrapBeaconsMonitor(Context context, Database database, ServerAPI serverApi, BeaconDiscoveryListener listener){
        this.mContext = context;
        this.mDatabase = database;
        this.mServerApi = serverApi;
        this.mRegionStateListener = listener;
        this.mHandler = new Handler();
        this.cachedTagsDB = new HashMap<String, Beacon>();
        this.previousTags = new TagsChangedSet(-1000);
        initBeaconManager();
        initGeofenceManager();
    }

    private void initBeaconManager(){
        this.mBeaconManager = BeaconManager.getInstanceForApplication(mContext);

        //Foreground scan settings
        mBeaconManager.setForegroundScanPeriod(Constants.BLUETOOTH_SCAN_PERIOD);
        mBeaconManager.setForegroundBetweenScanPeriod(Constants.BLUETOOTH_SCAN_BETWEEN_PERIOD);
        mBeaconManager.setBackgroundScanPeriod(Constants.BLUETOOTH_SCAN_PERIOD);
        mBeaconManager.setBackgroundBetweenScanPeriod(Constants.BLUETOOTH_SCAN_BETWEEN_PERIOD);

        mBeaconManager.setRangeNotifier(this);
        mBeaconManager.setBackgroundMode(false);
        mBeaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(Constants.IBEACONS_LAYOUT));
    }

    private void initGeofenceManager(){
        mGeofenceManager = new GeofenceManager(mContext, new GeofenceManager.GeofenceListener() {
            @Override
            public void onGeofenceEntered(final Beacon virtualBeacon) {
                mHandler.post(new Runnable() {

                    @Override
                    public void run() {
                        virtualBeacon.setLastSeen(Utils.getUtcInMillis());

                        Set<Beacon> lastSeenBeacons = getLastSeenBeacons();

                        //send server checkin handshake
                        mRegionStateListener.onBeaconDiscovered(virtualBeacon);
                        refreshBeaconsAround(virtualBeacon, null);

                        // update last seen / add virtual beacon if it doesn't exist
                        lastSeenBeacons.add(virtualBeacon);

                        checkOutVirtualBeaconsExcept(virtualBeacon, lastSeenBeacons);

                        setLastSeenBeacons(lastSeenBeacons);
                        processTagChanges();
                    }
                });
            }

            @Override
            public void onGeofenceExited(final Beacon virtualBeacon) {
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        Set<Beacon> lastSeenBeacons = getLastSeenBeacons();

                        //send server checkout handshake
                        mRegionStateListener.onBeaconLost(virtualBeacon, false);

                        lastSeenBeacons.remove(virtualBeacon);
                        setLastSeenBeacons(lastSeenBeacons);
                        processTagChanges();

                    }
                });
            }
        });
    }

    private synchronized Set<Beacon> getLastSeenBeacons(){
        if(mLastSeenBeacons == null) {

            Set<Beacon> beacons = new HashSet<Beacon>();
            for (PersistentBeacon cur : mDatabase.getAll(PersistentBeacon.class)){
                beacons.add(new Beacon(cur));
            }

            mLastSeenBeacons = beacons;
        }

        return mLastSeenBeacons;
    }

    private synchronized void setLastSeenBeacons(Set<Beacon> beacons) {
        mLastSeenBeacons = beacons;

        Set<PersistentBeacon> persistentBeacons = new HashSet<PersistentBeacon>();

        for(Beacon beacon : beacons){
            persistentBeacons.add(new PersistentBeacon(beacon));
        }

        mDatabase.set(PersistentBeacon.class, persistentBeacons);
    }

    public void startMonitoring(List<Region> regions){
        Log.d(TAG, "Monitoring: " + Arrays.toString(regions.toArray()));

        this.regions = regions;
        mBeaconManager.bind(this);

        ProxSeeSDKManager.Log("Started monitoring: " + regions.size());

        List<PersistentBeacon> beacons = mDatabase.CustomQueries.findAllBeacons(true);
        List<Beacon> list = convertPersistentBeacons(beacons);
        mGeofenceManager.start(list);

        Intent intent = new Intent(mContext, BackgroundService.class);
        mContext.startService(intent);
    }

    @Override
    public void onBeaconServiceConnect() {
        try {
            for (Region region : regions) {
                LogManager.d(TAG, "Region monitoring activated for region %s", region);
                mBeaconManager.startRangingBeaconsInRegion(region);
                if (mBeaconManager.isBackgroundModeUninitialized()) {
                    mBeaconManager.setBackgroundMode(true);
                }
            }
        } catch (RemoteException e) {
            LogManager.e(e, TAG, "Can't set up region monitoring");
        }
    }

    @Override
    public Context getApplicationContext() {
        return mContext;
    }

    @Override
    public void unbindService(ServiceConnection serviceConnection) {
        mContext.unbindService(serviceConnection);
    }

    @Override
    public boolean bindService(Intent intent, ServiceConnection serviceConnection, int i) {
        return mContext.bindService(intent, serviceConnection, i);
    }

    @Override
    public void didRangeBeaconsInRegion(final Collection<org.altbeacon.beacon.Beacon> seenBeacons, Region region) {
        mHandler.post(new Runnable() {
            @Override
            public void run() {

                Set<Beacon> currentlySeenRealBeacons = convertAltBeacons(seenBeacons);
                Set<Beacon> lastSeenBeacons = getLastSeenBeacons();

                Log.v(TAG, "Currently seen beacons: " + currentlySeenRealBeacons);

                Set<Beacon> discoveredBeacons = new HashSet<Beacon>();
                discoveredBeacons.addAll(currentlySeenRealBeacons);
                discoveredBeacons.removeAll(lastSeenBeacons);

                if (!discoveredBeacons.isEmpty()) {

                    Log.v(TAG, "New beacons detected: " + discoveredBeacons);

                    for (Beacon cur : discoveredBeacons) {
                        Log.i(TAG, "BROADCAST ENTER REGION");
                        mRegionStateListener.onBeaconDiscovered(cur);
                    }

                    lastSeenBeacons.addAll(discoveredBeacons);
                }

                Set<Beacon> lostRealBeacons = new HashSet<Beacon>();
                for (Beacon beacon : lastSeenBeacons) {
                    // Geofences shouldn't be removed by expiry here
                    if (beacon.isGeofence()) {
                        continue;
                    }

                    if (currentlySeenRealBeacons.contains(beacon)) {
                        beacon.setLastSeen(System.currentTimeMillis());
                        Log.i(TAG, "Beacon still visible, updating last seen time. lastSeen: " + beacon.getLastSeen());
                    } else if (System.currentTimeMillis() - beacon.getLastSeen() > Constants.BEACON_EXPIRY) {
                        Log.i(TAG, "Beacon lost for over " + Constants.BEACON_EXPIRY + "ms, dropping. lastSeen: " + beacon.getLastSeen());
                        lostRealBeacons.add(beacon);
                    } else {
                        Log.i(TAG, "Waiting " + Constants.BEACON_EXPIRY + "ms before dropping lost beacon. lastSeen: " + beacon.getLastSeen());
                    }
                }

                Log.v(TAG, "Beacons missing: " + lostRealBeacons);
                checkOutBeacons(lostRealBeacons, lastSeenBeacons, false);

                Beacon activeVirtualBeacon = getVirtualBeaconToBeKept(lastSeenBeacons);
                checkOutVirtualBeaconsExcept(activeVirtualBeacon, lastSeenBeacons);

                setLastSeenBeacons(lastSeenBeacons);
                processTagChanges();
            }
        });
    }

    private Beacon getVirtualBeaconToBeKept(Set<Beacon> activeBeacons) {
        if (containsRealBeacons(activeBeacons)) {
            // Don't keep any virtual beacons if there are any real active beacons
            return null;
        }

        // get the latest virtual beacon
        Beacon latestVirtualBeacon = null;
        for (Beacon beacon : activeBeacons) {
            if (beacon.isGeofence()) {
                if (latestVirtualBeacon == null || latestVirtualBeacon.getLastSeen() < beacon.getLastSeen()) {
                    latestVirtualBeacon = beacon;
                }
            }
        }

        return latestVirtualBeacon;
    }

    private boolean containsRealBeacons(Set<Beacon> beacons) {
        for (Beacon beacon : beacons) {
            if (!beacon.isGeofence()) {
                return true;
            }
        }

        return false;
    }

    private synchronized void processTagChanges(){
        TagsChangedSet previousTags = new TagsChangedSet(BootstrapBeaconsMonitor.this.previousTags);
        TagsChangedSet currentTags = new TagsChangedSet(Utils.getUtcInMillis());
        long time = System.currentTimeMillis() - mLastTimeFlushed;

        if(time > Constants.CACHE_EXPIRY) {
            cachedTagsDB.clear();
            mLastTimeFlushed = System.currentTimeMillis();
        }

        LinkedList<Beacon> list = new LinkedList<Beacon>(getLastSeenBeacons());

        while(!list.isEmpty()){
            Beacon cur = list.removeFirst();

            String key = cur.getMajorMinorKey();

            if(cachedTagsDB.containsKey(key)){
                Beacon beacon = cachedTagsDB.get(key);
                Collections.addAll(currentTags.getTags(), beacon.getTags());

            } else {
                refreshBeaconsAround(cur, list);
                return;
            }
        }

        if(!previousTags.getTags().equals(currentTags.getTags())) {
            mRegionStateListener.onTagsChanged(previousTags, currentTags);
            this.previousTags = currentTags;
        }
    }

    private void refreshBeaconsAround(final Beacon beacon, final LinkedList<Beacon> list){
        mServerApi.getBeaconsAround(beacon.getMajor(), beacon.getMinor(), Constants.BEACONS_AROUND_RADIUS, new ResponseListener<TaggedBeacon[]>() {
            @Override
            public void onResponse(int responseCode, TaggedBeacon[] taggedBeacons) {
                for(TaggedBeacon cur:taggedBeacons){
                    Beacon beacon = new Beacon(cur, 0);
                    String key = beacon.getMajorMinorKey();

                    cachedTagsDB.put(key, beacon);
                    Log.d(TAG, "Beacon: " + cur);
                }

                processVirtualBeacons(taggedBeacons);
                processTagChanges();
            }

            private void processVirtualBeacons(TaggedBeacon[] taggedBeacons){
                List<PersistentBeacon> vBeacons = new LinkedList<PersistentBeacon>();

                for(TaggedBeacon cur:taggedBeacons){
                    if(cur.isGeofence()){
                        vBeacons.add(new PersistentBeacon(cur));
                    }
                }

                List<Beacon> virtualBeacons = convertPersistentBeacons(vBeacons);
                mGeofenceManager.setVirtualBeacons(virtualBeacons);
            }

        }, new Response.ErrorListener() {
            @Override
            public void onErrorResponse(VolleyError volleyError) {
                if(volleyError instanceof ServerError && volleyError.networkResponse.statusCode == 404){
                    if(list != null && !list.isEmpty()) {
                        refreshBeaconsAround(list.removeFirst(), list);
                    }
                    Log.w(TAG, "Unrecognized beacon: " + beacon);
                }else{
                    Log.e(TAG, "Could not fetch nearby beacons", volleyError);
                }
            }
        });
    }

    private Set<Beacon> convertAltBeacons(Iterable<org.altbeacon.beacon.Beacon> beacons){
        Set<Beacon> list = new HashSet<Beacon>();
        for (org.altbeacon.beacon.Beacon cur : beacons) {
            list.add(new Beacon(cur, System.currentTimeMillis()));
        }

        return list;
    }

    public List<Beacon> convertPersistentBeacons(Iterable<PersistentBeacon> PersistentBeacons){
        List<Beacon> beacons = new LinkedList<Beacon>();
        for(PersistentBeacon cur:PersistentBeacons){
            beacons.add(new Beacon(cur));
        }
        return beacons;
    }

    /**
     * Not thread-safe - only use in single-threaded manner.
     *
     * Performs an implied check-out on all other virtual beacons
     *
     * @param activeVirtualBeacon The beacon that should not be checked out (a new active beacon that is
     *                            invalidating previous virtual beacons). If null, checkouts all virtual
     *                            beacons
     * @param lastSeenBeacons checks for existing virtual beacons in this set and removes then after checkout
     */
    public void checkOutVirtualBeaconsExcept(Beacon activeVirtualBeacon, Set<Beacon> lastSeenBeacons) {
        List<Beacon> virtualBeaconsToCheckout = new ArrayList<Beacon>();
        for (Beacon currentBeacon : lastSeenBeacons) {
            if (currentBeacon.isGeofence() && (activeVirtualBeacon == null || !currentBeacon.equals(activeVirtualBeacon))) {
                virtualBeaconsToCheckout.add(currentBeacon);
            }
        }

        checkOutBeacons(virtualBeaconsToCheckout, lastSeenBeacons, true);
    }

    /**
     * Not thread-safe - only use in single-threaded manner.
     *
     * Performs a check-out on all given beacons
     *
     * @param beaconsToCheckOut The beacon that should be checked out. Gracefully accepts null and empty (does nothing
     * @param lastSeenBeacons removes any checked out beacons from this list
     * @param impliedCheckout if true, it is an implied checkout, otherwise a regular one
     */
    private void checkOutBeacons(Collection<Beacon> beaconsToCheckOut, Set<Beacon> lastSeenBeacons, boolean impliedCheckout) {
        if (beaconsToCheckOut == null || beaconsToCheckOut.isEmpty()) {
            return;
        }

        for (Beacon beacon : beaconsToCheckOut) {
            Log.i(TAG, "BROADCAST EXIT REGION - implied: " + impliedCheckout + " name: " + beacon.getName() );
            mRegionStateListener.onBeaconLost(beacon, impliedCheckout);
        }

        lastSeenBeacons.removeAll(beaconsToCheckOut);
    }

    public interface BeaconDiscoveryListener {
        void onBeaconDiscovered(Beacon beacon);
        void onBeaconLost(Beacon beacon, boolean implied);
        void onTagsChanged(TagsChangedSet previousTags, TagsChangedSet currentTags);
    }
}