package io.proxsee.sdk;

import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.location.LocationManager;
import android.os.Bundle;
import android.util.Log;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingEvent;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import io.proxsee.sdk.misc.BeaconMajorMinorComparator;
import io.proxsee.sdk.misc.Utils;
import io.proxsee.sdk.model.Beacon;

/**
 * Created by Ahmad Shami on 7/21/15.
 */
public class GeofenceManager implements ResultCallback<Status> {

    protected static final String TAG = GeofenceManager.class.getSimpleName();

    /**
     * Provides the entry point to Google Play services.
     */
    protected GoogleApiClient mGoogleApiClient;

    /**
     * Used when requesting to add or remove geofences.
     */
    private PendingIntent mGeofencePendingIntent;

    private Context mContext;
    private GeofenceListener mGeofenceListener;

    private Object beaconMapLock = new Object();
    private LinkedHashMap<String, Beacon> mVirtualBeaconMap;
    private GeofenceEventReceiver mBroadcastReceiver;
    private boolean mStarted;

    public GeofenceManager(Context context, GeofenceListener geofenceListener){
        this.mContext = context;
        this.mGeofenceListener = geofenceListener;
        this.mVirtualBeaconMap = new LinkedHashMap<String, Beacon>();
        this.mBroadcastReceiver = new GeofenceEventReceiver();
    }

    @Override
    public void onResult(Status status) {
        if (status.isSuccess()) {

        } else {
            // Get the status code for the error and log it using a user-friendly message.
            switch (status.getStatusCode()) {
                case GeofenceStatusCodes.GEOFENCE_NOT_AVAILABLE:
                    ;
                case GeofenceStatusCodes.GEOFENCE_TOO_MANY_GEOFENCES:
                    ;
                case GeofenceStatusCodes.GEOFENCE_TOO_MANY_PENDING_INTENTS:
                    ;
                default:
            }

            Log.e(TAG, "onResult status: " + status);
        }
    }

    public void setVirtualBeacons(List<Beacon> virtualBeacons){
        // Synchronization as this can be encountered from the thread start or from refreshBeaconsAround
        // Can result in double check-in on start up
        synchronized (beaconMapLock) {
            // pre-sort the list to return consistent entries between calls in the following sublist
            Collections.sort(virtualBeacons, new BeaconMajorMinorComparator());

            virtualBeacons = virtualBeacons.subList(0, Math.min(virtualBeacons.size(), Constants.MAX_GEOFENCE));

            if (areVirtualBeaconsGeoLocationsChanged(virtualBeacons)) {
                removeAllVirtualBeacons();
                addVirtualBeacons(virtualBeacons);
            } else {
                // no need to change the geolocations (and cause additional hits) - just update the map in case info like the name have changed
                for (Beacon currentBeacon : virtualBeacons) {
                    mVirtualBeaconMap.put(currentBeacon.getMajorMinorKey(), currentBeacon);
                }
            }
        }
    }

    private boolean areVirtualBeaconsGeoLocationsChanged(List<Beacon> virtualBeacons) {
        if (virtualBeacons.size() != mVirtualBeaconMap.size()) {
            return true;
        }

        for (Beacon currentBeacon : virtualBeacons){
            String beaconKey = currentBeacon.getMajorMinorKey();

            Beacon potentiallyMatchingBeacon = mVirtualBeaconMap.get(beaconKey);

            if (potentiallyMatchingBeacon == null) {
                return true;
            }

            if (!Utils.areApproximatelyEqual(potentiallyMatchingBeacon.getLat(), currentBeacon.getLat()) ||
                    !Utils.areApproximatelyEqual(potentiallyMatchingBeacon.getLng(), currentBeacon.getLng()) ||
                    !Utils.areApproximatelyEqual(potentiallyMatchingBeacon.getRadius(), currentBeacon.getRadius())) {
                return true;
            }
        }

        return false;
    }

    private void addVirtualBeacons(List<Beacon> virtualBeacons){
        if(virtualBeacons.isEmpty()){
            return;
        }

        List<Geofence> geofences = new LinkedList<Geofence>();
        LinkedHashMap<String, Beacon> map = new LinkedHashMap<String, Beacon>();

        for(Beacon cur:virtualBeacons){
            String key = cur.getMajorMinorKey();
            Log.d(TAG, "key: " + key);

            map.put(key, cur);
            geofences.add(new Geofence.Builder()
                    .setRequestId(key)
                    .setCircularRegion(cur.getLat(), cur.getLng(), cur.getRadius())
                    .setExpirationDuration(Geofence.NEVER_EXPIRE)
                    .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER | Geofence.GEOFENCE_TRANSITION_EXIT)
//                    .setLoiteringDelay(Constants.GEOFENCE_LOITERING_DELAY)
                    .setNotificationResponsiveness(Constants.GEOFENCE_NOTIFICATION_RESPONSIVENESS)
                    .build());
        }

        LocationServices.GeofencingApi.addGeofences(
                mGoogleApiClient,
                // The GeofenceRequest object.
                buildGeofencingRequest(geofences),
                // A pending intent that that is reused when calling removeGeofences(). This
                // pending intent is used to generate an intent when a matched geofence
                // transition is observed.
                getGeofencePendingIntent()
        ).setResultCallback(this); // Result processed in onResult().

        mVirtualBeaconMap.putAll(map);
    }

    private void removeAllVirtualBeacons(){
        // Remove geofences.
        LocationServices.GeofencingApi.removeGeofences(
                mGoogleApiClient,
                // This is the same pending intent that was used in addGeofences().
                getGeofencePendingIntent()
        ).setResultCallback(this); // Result processed in onResult().

        mVirtualBeaconMap.clear();
    }

    private PendingIntent getGeofencePendingIntent() {
        // Reuse the PendingIntent if we already have it.
        if (mGeofencePendingIntent != null) {
            return mGeofencePendingIntent;
        }
        Intent intent = new Intent(GeofenceEventReceiver.ACTION_GEOFENCE_TRANSITION);
        // We use FLAG_UPDATE_CURRENT so that we get the same pending intent back when calling
        // addGeofences() and removeGeofences().
        return mGeofencePendingIntent = PendingIntent.getBroadcast(mContext, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
    }

    public void start(final List<Beacon> beacons){
        mGoogleApiClient = new GoogleApiClient.Builder(mContext)
                .addConnectionCallbacks(new GoogleApiClient.ConnectionCallbacks() {
                    @Override
                    public void onConnected(Bundle bundle) {
                        setVirtualBeacons(beacons);

                        Log.i(TAG, "Connected to GoogleApiClient");
                    }

                    @Override
                    public void onConnectionSuspended(int i) {
                        // The connection to Google Play services was lost for some reason.
                        // onConnected() will be called again automatically when the service reconnects
                        Log.i(TAG, "Connection suspended");
                    }
                })
                .addOnConnectionFailedListener(new GoogleApiClient.OnConnectionFailedListener() {
                    @Override
                    public void onConnectionFailed(ConnectionResult connectionResult) {
                        // Refer to the javadoc for ConnectionResult to see what error codes might be returned in
                        // onConnectionFailed.
                        Log.e(TAG, "Connection failed: ConnectionResult.getErrorCode() = " + connectionResult.getErrorCode());
                    }
                })
                .addApi(LocationServices.API)
                .build();

        IntentFilter filter = new IntentFilter();
        filter.addAction(LocationManager.PROVIDERS_CHANGED_ACTION);
        filter.addAction(GeofenceEventReceiver.ACTION_GEOFENCE_TRANSITION);
        mContext.registerReceiver(mBroadcastReceiver, filter);

        mStarted = true;
        mGoogleApiClient.connect();
    }

    public void stop(){
        mStarted = false;
        mContext.unregisterReceiver(mBroadcastReceiver);
        mGoogleApiClient.disconnect();
    }

    public boolean isConnected(){
        return mGoogleApiClient.isConnected();
    }

    public boolean isStarted(){
        return mStarted;
    }

    private GeofencingRequest buildGeofencingRequest(List<Geofence> geofenceList) {
        GeofencingRequest.Builder builder = new GeofencingRequest.Builder();

        // Note: no initial trigger events - the geofence is only expected to trigger after
        // hitting a beacon - not if suddenly initialized (location services turned on) or in
        // proximity to a beacon

        // Add the geofences to be monitored by geofencing service.
        builder.addGeofences(geofenceList);

        // Return a GeofencingRequest.
        return builder.build();
    }

    public interface GeofenceListener{
        public void onGeofenceEntered(Beacon virtualBeacon);
        public void onGeofenceExited(Beacon virtualBeacon);
    }

    private class GeofenceEventReceiver extends BroadcastReceiver{
        private static final String ACTION_GEOFENCE_TRANSITION = "io.proxsee.sdk.ACTION_GEOFENCE_TRANSITION";

        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();

            if(LocationManager.PROVIDERS_CHANGED_ACTION.equals(action)){
                if(mStarted && !mGoogleApiClient.isConnected()){
                    mGoogleApiClient.connect();
                }
            }else if(ACTION_GEOFENCE_TRANSITION.equals(action)){
                GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(intent);

                if(geofencingEvent.hasError()){
                    Log.e(TAG, "Error while retrieving event, code: " + geofencingEvent.getErrorCode());
                }

                List<Geofence> geofences = geofencingEvent.getTriggeringGeofences();
                if(geofences != null) {
                    for (Geofence cur : geofences) {
                        int transition = geofencingEvent.getGeofenceTransition();
                        String key = cur.getRequestId();
                        Beacon virtualBeacon = mVirtualBeaconMap.get(key);

                        switch (transition) {
                            case Geofence.GEOFENCE_TRANSITION_ENTER:
//                                Log.d(TAG, "ENTERED GEOFENCE: " + virtualBeacon);
                                mGeofenceListener.onGeofenceEntered(virtualBeacon);
                                break;

                            case Geofence.GEOFENCE_TRANSITION_EXIT:
//                                Log.d(TAG, "EXITED GEOFENCE: " + virtualBeacon);
                                mGeofenceListener.onGeofenceExited(virtualBeacon);
                                break;

                            default:
                                Log.e(TAG, "Invalid transition type: " + transition);
                        }
                    }
                }else{
                }
            }
        }
    }
}
