/*
 * Decompiled with CFR 0.152.
 */
package org.altbeacon.beacon;

import java.nio.ByteBuffer;
import java.nio.LongBuffer;
import java.util.Arrays;
import java.util.UUID;
import java.util.regex.Pattern;

public class Identifier
implements Comparable<Identifier> {
    private static final Pattern HEX_PATTERN = Pattern.compile("^0x[0-9A-Fa-f]*$");
    private static final Pattern DECIMAL_PATTERN = Pattern.compile("^[0-9]{1,5}$");
    private static final Pattern UUID_PATTERN = Pattern.compile("^[0-9A-Fa-f]{8}-?[0-9A-Fa-f]{4}-?[0-9A-Fa-f]{4}-?[0-9A-Fa-f]{4}-?[0-9A-Fa-f]{12}$");
    private static final int MAX_INTEGER = 65535;
    private final byte[] mValue;

    public static Identifier parse(String stringValue) {
        if (stringValue == null) {
            throw new NullPointerException("Identifiers cannot be constructed from null pointers but \"stringValue\" is null.");
        }
        if (HEX_PATTERN.matcher(stringValue).matches()) {
            return Identifier.parseHex(stringValue.substring(2));
        }
        if (UUID_PATTERN.matcher(stringValue).matches()) {
            return Identifier.parseHex(stringValue.replace("-", ""));
        }
        if (DECIMAL_PATTERN.matcher(stringValue).matches()) {
            int value = -1;
            try {
                value = Integer.valueOf(stringValue);
            }
            catch (Throwable t) {
                throw new IllegalArgumentException("Unable to parse Identifier in decimal format.", t);
            }
            return Identifier.fromInt(value);
        }
        throw new IllegalArgumentException("Unable to parse Identifier.");
    }

    private static Identifier parseHex(String identifierString) {
        String str = identifierString.length() % 2 == 0 ? "" : "0";
        str = str + identifierString.toUpperCase();
        byte[] result = new byte[str.length() / 2];
        for (int i = 0; i < result.length; ++i) {
            result[i] = (byte)(Integer.parseInt(str.substring(i * 2, i * 2 + 2), 16) & 0xFF);
        }
        return new Identifier(result);
    }

    public static Identifier fromInt(int intValue) {
        if (intValue < 0 || intValue > 65535) {
            throw new IllegalArgumentException("Identifers can only be constructed from integers between 0 and 65535 (inclusive).");
        }
        byte[] newValue = new byte[]{(byte)(intValue >> 8), (byte)intValue};
        return new Identifier(newValue);
    }

    public static Identifier fromBytes(byte[] bytes, int start, int end, boolean littleEndian) {
        if (bytes == null) {
            throw new NullPointerException("Identifiers cannot be constructed from null pointers but \"bytes\" is null.");
        }
        if (start < 0 || start > bytes.length) {
            throw new ArrayIndexOutOfBoundsException("start < 0 || start > bytes.length");
        }
        if (end > bytes.length) {
            throw new ArrayIndexOutOfBoundsException("end > bytes.length");
        }
        if (start > end) {
            throw new IllegalArgumentException("start > end");
        }
        byte[] byteRange = Arrays.copyOfRange(bytes, start, end);
        if (littleEndian) {
            Identifier.reverseArray(byteRange);
        }
        return new Identifier(byteRange);
    }

    public static Identifier fromUuid(UUID uuid) {
        ByteBuffer buf = ByteBuffer.allocate(16);
        buf.putLong(uuid.getMostSignificantBits());
        buf.putLong(uuid.getLeastSignificantBits());
        return new Identifier(buf.array());
    }

    @Deprecated
    public Identifier(Identifier identifier) {
        if (identifier == null) {
            throw new NullPointerException("Identifiers cannot be constructed from null pointers but \"identifier\" is null.");
        }
        this.mValue = identifier.mValue;
    }

    protected Identifier(byte[] value) {
        if (value == null) {
            throw new NullPointerException("Identifiers cannot be constructed from null pointers but \"value\" is null.");
        }
        this.mValue = value;
    }

    public String toString() {
        if (this.mValue.length == 2) {
            return Integer.toString(this.toInt());
        }
        if (this.mValue.length == 16) {
            return this.toUuid().toString();
        }
        return this.toHexString();
    }

    public int toInt() {
        if (this.mValue.length > 2) {
            throw new UnsupportedOperationException("Only supported for Identifiers with max byte length of 2");
        }
        int result = 0;
        for (int i = 0; i < this.mValue.length; ++i) {
            result |= (this.mValue[i] & 0xFF) << (this.mValue.length - i - 1) * 8;
        }
        return result;
    }

    public byte[] toByteArrayOfSpecifiedEndianness(boolean bigEndian) {
        byte[] copy = Arrays.copyOf(this.mValue, this.mValue.length);
        if (!bigEndian) {
            Identifier.reverseArray(copy);
        }
        return copy;
    }

    private static void reverseArray(byte[] bytes) {
        for (int i = 0; i < bytes.length / 2; ++i) {
            int mirroredIndex = bytes.length - i - 1;
            byte tmp = bytes[i];
            bytes[i] = bytes[mirroredIndex];
            bytes[mirroredIndex] = tmp;
        }
    }

    public int getByteCount() {
        return this.mValue.length;
    }

    public int hashCode() {
        return Arrays.hashCode(this.mValue);
    }

    public boolean equals(Object that) {
        if (!(that instanceof Identifier)) {
            return false;
        }
        Identifier thatIdentifier = (Identifier)that;
        return Arrays.equals(this.mValue, thatIdentifier.mValue);
    }

    public String toHexString() {
        StringBuilder sb = new StringBuilder(2 + 2 * this.mValue.length);
        sb.append("0x");
        for (byte item : this.mValue) {
            sb.append(String.format("%02x", item));
        }
        return sb.toString();
    }

    @Deprecated
    public String toUuidString() {
        return this.toUuid().toString();
    }

    public UUID toUuid() {
        if (this.mValue.length != 16) {
            throw new UnsupportedOperationException("Only Identifiers backed by a byte array with length of exactly 16 can be UUIDs.");
        }
        LongBuffer buf = ByteBuffer.wrap(this.mValue).asLongBuffer();
        return new UUID(buf.get(), buf.get());
    }

    public byte[] toByteArray() {
        return (byte[])this.mValue.clone();
    }

    @Override
    public int compareTo(Identifier that) {
        if (this.mValue.length != that.mValue.length) {
            return this.mValue.length < that.mValue.length ? -1 : 1;
        }
        for (int i = 0; i < this.mValue.length; ++i) {
            if (this.mValue[i] == that.mValue[i]) continue;
            return this.mValue[i] < that.mValue[i] ? -1 : 1;
        }
        return 0;
    }
}

