/*
 * Decompiled with CFR 0.152.
 */
package overflowdb;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import overflowdb.Config;
import overflowdb.Edge;
import overflowdb.EdgeFactory;
import overflowdb.HeapUsageMonitor;
import overflowdb.IndexManager;
import overflowdb.Node;
import overflowdb.NodeDb;
import overflowdb.NodeFactory;
import overflowdb.NodeRef;
import overflowdb.ReferenceManager;
import overflowdb.storage.NodeDeserializer;
import overflowdb.storage.NodeSerializer;
import overflowdb.storage.NodesWriter;
import overflowdb.storage.OdbStorage;
import overflowdb.util.IteratorUtils;
import overflowdb.util.MultiIterator;
import overflowdb.util.NodesList;
import overflowdb.util.PropertyHelper;

public final class Graph
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(Graph.class);
    protected final AtomicLong currentId = new AtomicLong(-1L);
    protected final NodesList nodes = new NodesList();
    public final IndexManager indexManager = new IndexManager(this);
    private final Config config;
    private boolean closed = false;
    protected final Map<String, NodeFactory> nodeFactoryByLabel;
    protected final Map<String, EdgeFactory> edgeFactoryByLabel;
    protected final OdbStorage storage;
    public final NodeSerializer nodeSerializer;
    protected final NodeDeserializer nodeDeserializer;
    protected final Optional<HeapUsageMonitor> heapUsageMonitor;
    protected final boolean overflowEnabled;
    protected final ReferenceManager referenceManager;
    protected final NodesWriter nodesWriter;

    public static Graph open(Config configuration, List<NodeFactory<?>> nodeFactories, List<EdgeFactory<?>> edgeFactories, Function<Object, Object> convertPropertyForPersistence) {
        HashMap<String, NodeFactory> nodeFactoryByLabel = new HashMap<String, NodeFactory>(nodeFactories.size());
        HashMap<String, EdgeFactory> edgeFactoryByLabel = new HashMap<String, EdgeFactory>(edgeFactories.size());
        nodeFactories.forEach(factory -> nodeFactoryByLabel.put(factory.forLabel(), (NodeFactory)factory));
        edgeFactories.forEach(factory -> edgeFactoryByLabel.put(factory.forLabel(), (EdgeFactory)factory));
        return new Graph(configuration, nodeFactoryByLabel, edgeFactoryByLabel, convertPropertyForPersistence);
    }

    public static Graph open(Config configuration, List<NodeFactory<?>> nodeFactories, List<EdgeFactory<?>> edgeFactories) {
        return Graph.open(configuration, nodeFactories, edgeFactories, Function.identity());
    }

    private Graph(Config config, Map<String, NodeFactory> nodeFactoryByLabel, Map<String, EdgeFactory> edgeFactoryByLabel, Function<Object, Object> convertPropertyForPersistence) {
        this.config = config;
        this.nodeFactoryByLabel = nodeFactoryByLabel;
        this.edgeFactoryByLabel = edgeFactoryByLabel;
        this.storage = config.getStorageLocation().isPresent() ? OdbStorage.createWithSpecificLocation(config.getStorageLocation().get().toFile()) : OdbStorage.createWithTempFile();
        this.nodeDeserializer = new NodeDeserializer(this, nodeFactoryByLabel, config.isSerializationStatsEnabled(), this.storage);
        this.nodeSerializer = new NodeSerializer(config.isSerializationStatsEnabled(), this.storage, convertPropertyForPersistence);
        this.nodesWriter = new NodesWriter(this.nodeSerializer, this.storage);
        config.getStorageLocation().ifPresent(l -> this.initElementCollections(this.storage));
        this.overflowEnabled = config.isOverflowEnabled();
        if (this.overflowEnabled) {
            this.referenceManager = config.getExecutorService().isPresent() ? new ReferenceManager(this.storage, this.nodesWriter, config.getExecutorService().get()) : new ReferenceManager(this.storage, this.nodesWriter);
            this.heapUsageMonitor = Optional.of(new HeapUsageMonitor(config.getHeapPercentageThreshold(), this.referenceManager));
        } else {
            this.referenceManager = null;
            this.heapUsageMonitor = Optional.empty();
        }
    }

    private void initElementCollections(OdbStorage storage) {
        long start = System.currentTimeMillis();
        Set<Map.Entry<Long, byte[]>> serializedNodes = storage.allNodes();
        logger.info(String.format("initializing %d nodes from existing storage", serializedNodes.size()));
        int importCount = 0;
        long maxId = this.currentId.get();
        for (Map.Entry<Long, byte[]> entry : serializedNodes) {
            try {
                NodeRef nodeRef = this.nodeDeserializer.deserializeRef(entry.getValue());
                this.nodes.add(nodeRef);
                if (++importCount % 131072 == 0) {
                    logger.debug("imported " + importCount + " elements - still running...");
                }
                if (nodeRef.id <= maxId) continue;
                maxId = nodeRef.id;
            }
            catch (IOException e) {
                throw new RuntimeException("error while initializing vertex from storage: id=" + entry.getKey(), e);
            }
        }
        this.currentId.set(maxId + 1L);
        this.indexManager.initializeStoredIndices(storage);
        long elapsedMillis = System.currentTimeMillis() - start;
        logger.debug(String.format("initialized %s from existing storage in %sms", this, elapsedMillis));
    }

    public Node addNode(String label, Object ... keyValues) {
        return this.addNodeInternal(this.currentId.incrementAndGet(), label, keyValues);
    }

    public Node addNode(long id, String label, Object ... keyValues) {
        long currentIdAfter;
        if (this.nodes.contains(id)) {
            throw new IllegalArgumentException(String.format("Node with id already exists: %s", id));
        }
        long currentIdBefore = this.currentId.get();
        if (!this.currentId.compareAndSet(currentIdBefore, currentIdAfter = Long.max(id, this.currentId.get()))) {
            return this.addNode(id, label, keyValues);
        }
        return this.addNodeInternal(id, label, keyValues);
    }

    private Node addNodeInternal(long id, String label, Object ... keyValues) {
        if (this.isClosed()) {
            throw new IllegalStateException("cannot add more elements, graph is closed");
        }
        NodeRef node = this.createNode(id, label, keyValues);
        this.nodes.add(node);
        return node;
    }

    private NodeRef createNode(long idValue, String label, Object ... keyValues) {
        if (!this.nodeFactoryByLabel.containsKey(label)) {
            throw new IllegalArgumentException("No NodeFactory for label=" + label + " available.");
        }
        NodeFactory factory = this.nodeFactoryByLabel.get(label);
        Object node = factory.createNode(this, idValue);
        PropertyHelper.attachProperties(node, keyValues);
        if (this.referenceManager != null) {
            this.referenceManager.registerRef(((NodeDb)node).ref);
        }
        return ((NodeDb)node).ref;
    }

    public void applyBackpressureMaybe() {
        if (this.referenceManager != null) {
            this.referenceManager.applyBackpressureMaybe();
        }
    }

    public void registerNodeRef(NodeRef ref) {
        if (this.referenceManager != null) {
            this.referenceManager.registerRef(ref);
        }
    }

    public String toString() {
        return String.format("%s [%d nodes]", this.getClass().getSimpleName(), this.nodeCount());
    }

    @Override
    public synchronized void close() {
        if (this.isClosed()) {
            logger.info("graph is already closed");
        } else {
            this.closed = true;
            this.shutdownNow();
        }
    }

    private void shutdownNow() {
        logger.info("shutdown: start");
        try {
            this.heapUsageMonitor.ifPresent(monitor -> monitor.close());
            if (this.config.getStorageLocation().isPresent()) {
                this.indexManager.storeIndexes(this.storage);
                if (this.referenceManager != null) {
                    this.referenceManager.clearAllReferences();
                } else {
                    this.nodes.persistAll(this.nodesWriter);
                }
            }
        }
        finally {
            if (this.referenceManager != null) {
                this.referenceManager.close();
            }
            this.storage.close();
        }
        logger.info("shutdown finished");
    }

    public int nodeCount() {
        return this.nodes.size();
    }

    public int nodeCount(String label) {
        return this.nodes.cardinality(label);
    }

    public int edgeCount() {
        int edgeCount = 0;
        Iterator<Node> nodes = this.nodes();
        while (nodes.hasNext()) {
            NodeDb node = this.getNodeDb(nodes.next());
            edgeCount += node.outEdgeCount();
        }
        return edgeCount;
    }

    public Iterator<Edge> E() {
        return this.edges();
    }

    public Iterator<Edge> edges() {
        return IteratorUtils.flatMap(this.nodes(), node -> node.outE());
    }

    public Iterator<Edge> edges(String label) {
        return IteratorUtils.flatMap(this.nodes(), node -> node.outE(label));
    }

    public Iterator<Node> V() {
        return this.nodes();
    }

    public final Iterator<Node> nodes() {
        return this.nodes.iterator();
    }

    public Iterator<Node> V(long ... ids) {
        return this.nodes(ids);
    }

    public Node node(long id) {
        return this.nodes.nodeById(id);
    }

    public Iterator<Node> nodes(long ... ids) {
        if (ids.length == 0) {
            return Collections.emptyIterator();
        }
        if (ids.length == 1) {
            return IteratorUtils.from(this.node(ids[0]));
        }
        return IteratorUtils.map(Arrays.stream(ids).iterator(), this::node);
    }

    public Iterator<Node> nodes(String label) {
        return this.nodes.nodesByLabel(label).iterator();
    }

    public Iterator<Node> nodes(String ... labels) {
        MultiIterator<Node> multiIterator = new MultiIterator<Node>();
        for (String label : labels) {
            this.addNodesToMultiIterator(multiIterator, label);
        }
        return multiIterator;
    }

    public Iterator<Node> nodes(Set<String> labels) {
        MultiIterator<Node> multiIterator = new MultiIterator<Node>();
        for (String label : labels) {
            this.addNodesToMultiIterator(multiIterator, label);
        }
        return multiIterator;
    }

    public Iterator<Node> nodes(Predicate<String> labelPredicate) {
        MultiIterator<Node> multiIterator = new MultiIterator<Node>();
        for (String label : this.nodes.nodeLabels()) {
            if (!labelPredicate.test(label)) continue;
            this.addNodesToMultiIterator(multiIterator, label);
        }
        return multiIterator;
    }

    private final void addNodesToMultiIterator(MultiIterator<Node> multiIterator, String label) {
        ArrayList<Node> ret = this.nodes.nodesByLabel(label);
        if (ret != null) {
            multiIterator.addIterator(ret.iterator());
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public OdbStorage getStorage() {
        return this.storage;
    }

    public void copyTo(Graph destination) {
        if (destination.nodeCount() > 0) {
            throw new AssertionError((Object)"destination graph must be empty, but isn't");
        }
        this.nodes().forEachRemaining(node -> destination.addNode(node.id(), node.label(), PropertyHelper.toKeyValueArray(node.propertiesMap())));
        this.edges().forEachRemaining(edge -> {
            Node inNode = destination.node(edge.inNode().id());
            Node outNode = destination.node(edge.outNode().id());
            outNode.addEdge(edge.label(), inNode, PropertyHelper.toKeyValueArray(edge.propertiesMap()));
        });
    }

    public void remove(Node node) {
        NodeRef nodeRef = this.getNodeRef(node);
        this.nodes.remove(nodeRef);
        this.indexManager.removeElement(nodeRef);
        this.storage.removeNode(node.id());
    }

    private NodeRef getNodeRef(Node node) {
        if (node instanceof NodeRef) {
            return (NodeRef)node;
        }
        return ((NodeDb)node).ref;
    }

    private NodeDb getNodeDb(Node node) {
        if (node instanceof NodeDb) {
            return (NodeDb)node;
        }
        return ((NodeRef)node).get();
    }

    public void persistLibraryVersion(String name, String version) {
        this.storage.persistLibraryVersion(name, version);
    }

    public ArrayList<Map<String, String>> getAllLibraryVersions() {
        return this.storage.getAllLibraryVersions();
    }
}

