package io.smallrye.graphql.execution.datafetcher.helper;

import static io.smallrye.graphql.SmallRyeGraphQLServerLogging.log;
import static io.smallrye.graphql.transformation.Transformer.shouldTransform;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.json.bind.Jsonb;
import javax.json.bind.JsonbException;

import graphql.schema.DataFetchingEnvironment;
import io.smallrye.graphql.execution.Classes;
import io.smallrye.graphql.json.InputFieldsInfo;
import io.smallrye.graphql.json.JsonBCreator;
import io.smallrye.graphql.schema.model.Argument;
import io.smallrye.graphql.schema.model.Field;
import io.smallrye.graphql.schema.model.MappingInfo;
import io.smallrye.graphql.schema.model.ReferenceType;
import io.smallrye.graphql.transformation.AbstractDataFetcherException;
import io.smallrye.graphql.transformation.TransformException;
import io.smallrye.graphql.transformation.Transformer;

/**
 * Help with the arguments when doing reflection calls
 *
 * Here we need to transform (if needed) the arguments, and then make sure we
 * get the in the correct class type as expected by the method we want to call.
 *
 * @author Phillip Kruger (phillip.kruger@redhat.com)
 */
public class ArgumentHelper extends AbstractHelper {

    private final List<Argument> arguments;

    /**
     * We need the modeled arguments to create the correct values
     *
     * @param arguments the arguments
     *
     */
    public ArgumentHelper(List<Argument> arguments) {
        this.arguments = arguments;
    }

    /**
     * This gets a list of arguments that we need to all the method.
     *
     * We need to make sure the arguments is in the correct class type and,
     * if needed, transformed
     *
     * @param dfe the Data Fetching Environment from graphql-java
     *
     * @return a (ordered) List of all argument values
     */
    public Object[] getArguments(DataFetchingEnvironment dfe) throws AbstractDataFetcherException {
        return getArguments(dfe, false);
    }

    public Object[] getArguments(DataFetchingEnvironment dfe, boolean excludeSource) throws AbstractDataFetcherException {
        List<Object> argumentObjects = new LinkedList<>();
        for (Argument argument : arguments) {
            if (!argument.isSourceArgument() || !excludeSource) {
                Object argumentValue = getArgument(dfe, argument);
                argumentObjects.add(argumentValue);
            }
        }
        return argumentObjects.toArray();
    }

    /**
     * Get one argument.
     *
     * As with above this argument needs to be transformed and in the correct class type
     *
     * @param dfe the Data Fetching Environment from graphql-java
     * @param argument the argument (as created while building the model)
     * @return the value of the argument
     */
    private Object getArgument(DataFetchingEnvironment dfe, Argument argument) throws AbstractDataFetcherException {
        // If this is a source argument, just return the source. The source does 
        // not need transformation and would already be in the correct class type
        if (argument.isSourceArgument()) {
            Object source = dfe.getSource();
            if (source != null) {
                return source;
            }
        }

        // Else, get the argument value as if is from graphql-java
        // graphql-java will also populate the value with the default value if needed.
        Object argumentValueFromGraphQLJava = dfe.getArgument(argument.getName());

        // return null if the value is null
        if (argumentValueFromGraphQLJava == null) {
            return null;
        }

        return super.recursiveTransform(argumentValueFromGraphQLJava, argument);
    }

    /**
     * By now this is a 'leaf' value, i.e not a collection of array, so we just transform if needed.
     * the result might be in the wrong format.
     *
     * @param argumentValue the value to transform
     * @param field the field as created while scanning
     * @return transformed value
     */
    @Override
    Object singleTransform(Object argumentValue, Field field) throws AbstractDataFetcherException {
        if (!shouldTransform(field)) {
            return argumentValue;
        } else {
            return Transformer.in(field, argumentValue);
        }
    }

    /**
     * By now this is a 'leaf' value, i.e not a collection of array, so we just map if needed.
     *
     * @param argumentValue the value to map
     * @param field the field as created while scanning
     * @return mapped value
     */
    @Override
    Object singleMapping(Object argumentValue, Field field) throws AbstractDataFetcherException {
        if (shouldApplyMapping(field)) {
            String expectedType = field.getReference().getClassName();
            Class<?> expectedClass = classloadingService.loadClass(expectedType);

            if (getCreate(field).equals(MappingInfo.Create.CONSTRUCTOR)) {
                // Try with contructor
                try {
                    Constructor<?> constructor = expectedClass.getConstructor(argumentValue.getClass());
                    return constructor.newInstance(argumentValue);
                } catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException
                        | IllegalArgumentException | InvocationTargetException ex) {
                    // TODO: Log to debug ?
                }
            } else if (getCreate(field).equals(MappingInfo.Create.SET_VALUE)) {
                // Try with setValue
                try {
                    // TODO: Maybe later allow annotation to indicate what method this should be ?
                    Method setValueMethod = expectedClass.getMethod("setValue", argumentValue.getClass());
                    Constructor<?> constructor = expectedClass.getConstructor();
                    Object instance = constructor.newInstance();
                    setValueMethod.invoke(instance, argumentValue);
                    return instance;
                } catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException
                        | IllegalArgumentException | InvocationTargetException ex) {
                    // TODO: Log to debug ?
                }
            } else if (getCreate(field).equals(MappingInfo.Create.STATIC_FROM)) {
                // Try with static from???
                try {
                    String simpleClassName = argumentValue.getClass().getSimpleName();
                    String staticMethodName = "from" + simpleClassName;
                    Method staticFromMethod = expectedClass.getMethod(staticMethodName, argumentValue.getClass());
                    Object instance = staticFromMethod.invoke(null, argumentValue);
                    return instance;
                } catch (NoSuchMethodException | SecurityException | IllegalAccessException
                        | IllegalArgumentException | InvocationTargetException ex) {
                    // TODO: Log to debug ?
                }
            }
        }
        // Fall back to the original value
        return argumentValue;
    }

    private boolean shouldApplyMapping(Field field) {
        return field.getReference().hasMappingInfo()
                && !field.getReference().getMappingInfo().getCreate().equals(MappingInfo.Create.NONE) ||
                field.hasMappingInfo() && !field.getMappingInfo().getCreate().equals(MappingInfo.Create.NONE);
    }

    private MappingInfo.Create getCreate(Field field) {
        if (field.getReference().hasMappingInfo()) {
            return field.getReference().getMappingInfo().getCreate();
        } else if (field.hasMappingInfo()) {
            return field.getMappingInfo().getCreate();
        }
        return MappingInfo.Create.NONE;
    }

    /**
     * Here we have the potential transformed input and just need to
     * get the correct type
     *
     * @param fieldValue the input from graphql-java, potentially transformed
     * @param field the field as created while scanning
     * @return the value to use in the method call
     */
    @Override
    protected Object afterRecursiveTransform(Object fieldValue, Field field) throws AbstractDataFetcherException {
        String expectedType = field.getReference().getClassName();
        String receivedType = fieldValue.getClass().getName();

        // No need to do anything, everyting is already correct
        if (expectedType.equals(receivedType)) {
            return fieldValue;
        } else if (Classes.isPrimitiveOf(expectedType, receivedType)) {
            //expected is a primitive, we got the wrapper
            return fieldValue;
        } else if (field.getReference().getType().equals(ReferenceType.ENUM)) {
            Class<?> enumClass = classloadingService.loadClass(field.getReference().getClassName());
            return Enum.valueOf((Class<Enum>) enumClass, fieldValue.toString());
        } else {
            return correctObjectClass(fieldValue, field);
        }
    }

    /**
     * Here we create a Object from the input.
     * This can be a complex POJO input, or a default value set by graphql-java or a Scalar (that is not a primitive, like Date)
     *
     * @param argumentValue the argument from graphql-java
     * @param field the field as created while scanning
     * @return the return value
     */
    private Object correctObjectClass(Object argumentValue, Field field) throws AbstractDataFetcherException {
        String receivedClassName = argumentValue.getClass().getName();

        if (Map.class.isAssignableFrom(argumentValue.getClass())) {
            return correctComplexObjectFromMap((Map) argumentValue, field);
        } else if (receivedClassName.equals(String.class.getName())) {
            // We got a String, but not expecting one. Lets bind to Pojo with JsonB
            // This happens with @DefaultValue and Transformable (Passthrough) Scalars
            return correctComplexObjectFromJsonString(argumentValue.toString(), field);
        } else {
            log.dontKnowHoToHandleArgument(field.getMethodName());
            return argumentValue;
        }
    }

    /**
     * If we got a map from graphql-java, this is a complex pojo input object
     *
     * We need to create a object from this using JsonB.
     * We also need to handle transformation of fields that is on this complex type.
     *
     * The transformation with JsonB annotation will happen when binding, and the transformation
     * with non-jsonb annotatin will happen when we create a json string from the map.
     *
     * @param m the map from graphql-java
     * @param field the field as created while scanning
     * @return a java object of this type.
     */
    private Object correctComplexObjectFromMap(Map m, Field field) throws AbstractDataFetcherException {
        String className = field.getReference().getClassName();

        // Let's see if there are any fields that needs transformation
        if (InputFieldsInfo.hasTransformationFields(className)) {
            Map<String, Field> transformationFields = InputFieldsInfo.getTransformationFields(className);

            for (Map.Entry<String, Field> entry : transformationFields.entrySet()) {
                String fieldName = entry.getKey();
                if (m.containsKey(fieldName)) {
                    Object valueThatShouldTransform = m.get(fieldName);
                    Field fieldThatShouldTransform = entry.getValue();
                    valueThatShouldTransform = recursiveTransform(valueThatShouldTransform, fieldThatShouldTransform);
                    m.put(fieldName, valueThatShouldTransform);
                }
            }
        }

        // Let's see if there are any fields that needs mapping
        if (InputFieldsInfo.hasMappingFields(className)) {
            Map<String, Field> mappingFields = InputFieldsInfo.getMappingFields(className);

            for (Map.Entry<String, Field> entry : mappingFields.entrySet()) {
                String fieldName = entry.getKey();
                if (m.containsKey(fieldName)) {
                    Object valueThatShouldMap = m.get(fieldName);
                    Field fieldThatShouldMap = entry.getValue();
                    valueThatShouldMap = recursiveMapping(valueThatShouldMap, fieldThatShouldMap);
                    m.put(fieldName, valueThatShouldMap);
                }
            }
        }

        // Create a valid jsonString from a map    
        String jsonString = JsonBCreator.getJsonB().toJson(m);
        return correctComplexObjectFromJsonString(jsonString, field);
    }

    /**
     * This is used once we have a valid jsonString, either from above or from complex default value from graphql-java
     *
     * @param jsonString the object represented as a json String
     * @param field the field as created while scanning
     * @return the correct object
     */
    private Object correctComplexObjectFromJsonString(String jsonString, Field field) throws AbstractDataFetcherException {
        Class ownerClass = classloadingService.loadClass(field.getReference().getClassName());
        try {
            Jsonb jsonb = JsonBCreator.getJsonB(field.getReference().getClassName());
            return jsonb.fromJson(jsonString, ownerClass);
        } catch (JsonbException jbe) {
            throw new TransformException(jbe, field, jsonString);
        }
    }
}
