package io.soffa.gradle.plugins.coverage

import org.gradle.api.Plugin
import org.gradle.api.Project

class JacocoCoveragePlugin implements Plugin<Project> {

    public static final String COVERAGE_CLASSES = "soffa.coverage.classes"
    public static final String COVERAGE_METHODS = "soffa.coverage.methods"
    public static final String COVERAGE_COMPLEXITY = "soffa.coverage.complexity"

    @Override
    void apply(Project project) {
        project.plugins.apply("jacoco")
        project.jacocoTestReport {
            reports {
                xml.enabled true
                html.enabled true
                html.destination project.file("${project.buildDir}/jacocoHtml")
            }
        }

        def classCoverage = project.findProperty(COVERAGE_CLASSES) ?: getDefaultClassesCoverage()
        def methodsCoverage = project.findProperty(COVERAGE_METHODS) ?: getDefaultMethodsCoverage()
        def complexityCoverage = project.findProperty(COVERAGE_COMPLEXITY) ?: getDefaultComplexityCoverage()

        if (classCoverage < 0 || classCoverage > 1) throw new Error("$COVERAGE_CLASSES must be a decimal value between 0.0 and 1.0")
        if (methodsCoverage < 0 || methodsCoverage > 1) throw new Error("$COVERAGE_METHODS must be a decimal value between 0.0 and 1.0")
        if (complexityCoverage < 0 || complexityCoverage > 10) throw new Error("$COVERAGE_COMPLEXITY must be an integer value between 0 and 10 (recommended)")

        project.jacocoTestCoverageVerification {
            violationRules {
                rule {
                    limit {
                        minimum = classCoverage
                        counter = "CLASS" // BRANCH CLASS COMPLEXITY INSTRUCTION  LINE  METHOD
                        value = "COVEREDRATIO" // COVEREDCOUNT COVEREDRATIO  MISSEDCOUNT  MISSEDRATIO  TOTALCOUNT
                    }
                    limit {
                        minimum = methodsCoverage
                        counter = "METHOD" // BRANCH CLASS COMPLEXITY INSTRUCTION  LINE  METHOD
                        value = "COVEREDRATIO" // COVEREDCOUNT COVEREDRATIO  MISSEDCOUNT  MISSEDRATIO  TOTALCOUNT
                    }
                    limit {
                        maximum = complexityCoverage
                        counter = "COMPLEXITY" // BRANCH CLASS COMPLEXITY INSTRUCTION  LINE  METHOD
                        value = "TOTALCOUNT" // COVEREDCOUNT COVEREDRATIO  MISSEDCOUNT  MISSEDRATIO  TOTALCOUNT
                    }
                }
            }
            dependsOn("jacocoTestReport")
        }

        project.check {
            dependsOn("jacocoTestCoverageVerification")
        }

    }

    Float getDefaultClassesCoverage() {
        return 0.1;
    }

    Float getDefaultMethodsCoverage() {
        return 0.1;
    }

    Float getDefaultComplexityCoverage() {
        return 8;
    }
}
