/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.front50.model;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.BucketVersioningConfiguration;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.HeadBucketRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.SSEAlgorithm;
import com.amazonaws.services.s3.model.SetBucketVersioningConfigurationRequest;
import com.amazonaws.services.s3.model.VersionListing;
import com.amazonaws.util.StringUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.netflix.spinnaker.front50.api.model.Timestamped;
import com.netflix.spinnaker.front50.api.model.pipeline.Pipeline;
import com.netflix.spinnaker.front50.jackson.mixins.PipelineMixins;
import com.netflix.spinnaker.front50.jackson.mixins.TimestampedMixins;
import com.netflix.spinnaker.front50.model.ObjectType;
import com.netflix.spinnaker.front50.model.ReadOnlyModeException;
import com.netflix.spinnaker.front50.model.StorageService;
import com.netflix.spinnaker.kork.web.exceptions.NotFoundException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import net.logstash.logback.argument.StructuredArguments;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3StorageService
implements StorageService {
    private static final Logger log = LoggerFactory.getLogger(S3StorageService.class);
    private final ObjectMapper objectMapper = new ObjectMapper().addMixIn(Timestamped.class, TimestampedMixins.class).addMixIn(Pipeline.class, PipelineMixins.class);
    private final AmazonS3 amazonS3;
    private final String bucket;
    private final String rootFolder;
    private final Boolean readOnlyMode;
    private final String region;
    private final Boolean versioning;
    private final Integer maxKeys;
    private final ServerSideEncryption serverSideEncryption;

    public S3StorageService(ObjectMapper objectMapper, AmazonS3 amazonS3, String bucket, String rootFolder, Boolean readOnlyMode, String region, Boolean versioning, Integer maxKeys, ServerSideEncryption serverSideEncryption) {
        this.amazonS3 = amazonS3;
        this.bucket = bucket;
        this.rootFolder = rootFolder;
        this.readOnlyMode = readOnlyMode;
        this.region = region;
        this.versioning = versioning;
        this.maxKeys = maxKeys;
        this.serverSideEncryption = serverSideEncryption;
    }

    public void ensureBucketExists() {
        HeadBucketRequest request = new HeadBucketRequest(this.bucket);
        try {
            this.amazonS3.headBucket(request);
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                if (StringUtils.isNullOrEmpty((String)this.region)) {
                    log.info("Creating bucket {} in default region", (Object)StructuredArguments.value((String)"bucket", (Object)this.bucket));
                    this.amazonS3.createBucket(this.bucket);
                } else {
                    log.info("Creating bucket {} in region {}", (Object)StructuredArguments.value((String)"bucket", (Object)this.bucket), (Object)StructuredArguments.value((String)"region", (Object)this.region));
                    this.amazonS3.createBucket(this.bucket, this.region);
                }
                if (this.versioning.booleanValue()) {
                    log.info("Enabling versioning of the S3 bucket {}", (Object)StructuredArguments.value((String)"bucket", (Object)this.bucket));
                    BucketVersioningConfiguration configuration = new BucketVersioningConfiguration().withStatus("Enabled");
                    SetBucketVersioningConfigurationRequest setBucketVersioningConfigurationRequest = new SetBucketVersioningConfigurationRequest(this.bucket, configuration);
                    this.amazonS3.setBucketVersioningConfiguration(setBucketVersioningConfigurationRequest);
                }
            }
            throw e;
        }
    }

    public boolean supportsVersioning() {
        return this.versioning;
    }

    public <T extends Timestamped> T loadObject(ObjectType objectType, String objectKey) throws NotFoundException {
        try {
            S3Object s3Object = this.amazonS3.getObject(this.bucket, this.buildS3Key(objectType.group, objectKey, objectType.defaultMetadataFilename));
            T item = this.deserialize(s3Object, objectType.clazz);
            item.setLastModified(Long.valueOf(s3Object.getObjectMetadata().getLastModified().getTime()));
            return item;
        }
        catch (AmazonS3Exception e) {
            if (e.getStatusCode() == 404) {
                throw new NotFoundException("Object not found (key: " + objectKey + ")");
            }
            throw e;
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to deserialize object (key: " + objectKey + ")", e);
        }
    }

    public void deleteObject(ObjectType objectType, String objectKey) {
        if (this.readOnlyMode.booleanValue()) {
            throw new ReadOnlyModeException();
        }
        this.amazonS3.deleteObject(this.bucket, this.buildS3Key(objectType.group, objectKey, objectType.defaultMetadataFilename));
        this.writeLastModified(objectType.group);
    }

    public void bulkDeleteObjects(ObjectType objectType, Collection<String> objectKeys) {
        if (this.readOnlyMode.booleanValue()) {
            throw new ReadOnlyModeException();
        }
        Lists.partition(new ArrayList<String>(objectKeys), (int)1000).forEach(keys -> this.amazonS3.deleteObjects(new DeleteObjectsRequest(this.bucket).withKeys(keys.stream().map(k -> new DeleteObjectsRequest.KeyVersion(this.buildS3Key(objectType.group, (String)k, objectType.defaultMetadataFilename))).collect(Collectors.toList()))));
    }

    public <T extends Timestamped> void storeObject(ObjectType objectType, String objectKey, T item) {
        if (this.readOnlyMode.booleanValue()) {
            throw new ReadOnlyModeException();
        }
        try {
            byte[] bytes = this.objectMapper.writeValueAsBytes(item);
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentLength((long)bytes.length);
            objectMetadata.setContentMD5(new String(Base64.encodeBase64((byte[])DigestUtils.md5((byte[])bytes))));
            if (this.serverSideEncryption != null && this.serverSideEncryption.equals((Object)ServerSideEncryption.AES256)) {
                objectMetadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            } else if (this.serverSideEncryption != null && this.serverSideEncryption.equals((Object)ServerSideEncryption.AWSKMS)) {
                objectMetadata.setSSEAlgorithm(SSEAlgorithm.KMS.getAlgorithm());
            }
            this.amazonS3.putObject(this.bucket, this.buildS3Key(objectType.group, objectKey, objectType.defaultMetadataFilename), (InputStream)new ByteArrayInputStream(bytes), objectMetadata);
            this.writeLastModified(objectType.group);
        }
        catch (JsonProcessingException e) {
            throw new IllegalStateException(e);
        }
    }

    public Map<String, Long> listObjectKeys(ObjectType objectType) {
        long startTime = System.currentTimeMillis();
        ObjectListing bucketListing = this.amazonS3.listObjects(new ListObjectsRequest(this.bucket, S3StorageService.buildTypedFolder(this.rootFolder, objectType.group), null, null, this.maxKeys));
        List summaries = bucketListing.getObjectSummaries();
        while (bucketListing.isTruncated()) {
            bucketListing = this.amazonS3.listNextBatchOfObjects(bucketListing);
            summaries.addAll(bucketListing.getObjectSummaries());
        }
        log.debug("Took {}ms to fetch {} object keys for {}", new Object[]{StructuredArguments.value((String)"fetchTime", (Object)(System.currentTimeMillis() - startTime)), summaries.size(), StructuredArguments.value((String)"type", (Object)objectType)});
        return summaries.stream().filter(s -> this.filterS3ObjectSummary((S3ObjectSummary)s, objectType.defaultMetadataFilename)).collect(Collectors.toMap(s -> this.buildObjectKey(objectType, s.getKey()), s -> s.getLastModified().getTime()));
    }

    public <T extends Timestamped> Collection<T> listObjectVersions(ObjectType objectType, String objectKey, int maxResults) throws NotFoundException {
        if (maxResults == 1) {
            ArrayList<T> results = new ArrayList<T>();
            results.add(this.loadObject(objectType, objectKey));
            return results;
        }
        try {
            VersionListing versionListing = this.amazonS3.listVersions(new ListVersionsRequest(this.bucket, this.buildS3Key(objectType.group, objectKey, objectType.defaultMetadataFilename), null, null, null, Integer.valueOf(maxResults)));
            return versionListing.getVersionSummaries().stream().map(s3VersionSummary -> {
                try {
                    S3Object s3Object = this.amazonS3.getObject(new GetObjectRequest(this.bucket, this.buildS3Key(objectType.group, objectKey, objectType.defaultMetadataFilename), s3VersionSummary.getVersionId()));
                    Object item = this.deserialize(s3Object, objectType.clazz);
                    item.setLastModified(Long.valueOf(s3Object.getObjectMetadata().getLastModified().getTime()));
                    return item;
                }
                catch (IOException e) {
                    throw new IllegalStateException(e);
                }
            }).collect(Collectors.toList());
        }
        catch (AmazonS3Exception e) {
            if (e.getStatusCode() == 404) {
                throw new NotFoundException(String.format("No item found with id of %s", objectKey.toLowerCase()));
            }
            throw e;
        }
    }

    public long getLastModified(ObjectType objectType) {
        try {
            Map lastModified = (Map)this.objectMapper.readValue((InputStream)this.amazonS3.getObject(this.bucket, S3StorageService.buildTypedFolder(this.rootFolder, objectType.group) + "/last-modified.json").getObjectContent(), Map.class);
            return (Long)lastModified.get("lastModified");
        }
        catch (Exception e) {
            return 0L;
        }
    }

    public long getHealthIntervalMillis() {
        return Duration.ofSeconds(2L).toMillis();
    }

    private void writeLastModified(String group) {
        if (this.readOnlyMode.booleanValue()) {
            throw new ReadOnlyModeException();
        }
        try {
            byte[] bytes = this.objectMapper.writeValueAsBytes(Collections.singletonMap("lastModified", System.currentTimeMillis()));
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentLength((long)bytes.length);
            objectMetadata.setContentMD5(new String(Base64.encodeBase64((byte[])DigestUtils.md5((byte[])bytes))));
            if (this.serverSideEncryption != null && this.serverSideEncryption.equals((Object)ServerSideEncryption.AES256)) {
                objectMetadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            } else if (this.serverSideEncryption != null && this.serverSideEncryption.equals((Object)ServerSideEncryption.AWSKMS)) {
                objectMetadata.setSSEAlgorithm(SSEAlgorithm.KMS.getAlgorithm());
            }
            this.amazonS3.putObject(this.bucket, S3StorageService.buildTypedFolder(this.rootFolder, group) + "/last-modified.json", (InputStream)new ByteArrayInputStream(bytes), objectMetadata);
        }
        catch (JsonProcessingException e) {
            throw new IllegalStateException(e);
        }
    }

    private <T extends Timestamped> T deserialize(S3Object s3Object, Class<T> clazz) throws IOException {
        return (T)((Timestamped)this.objectMapper.readValue((InputStream)s3Object.getObjectContent(), clazz));
    }

    private boolean filterS3ObjectSummary(S3ObjectSummary s3ObjectSummary, String metadataFilename) {
        return s3ObjectSummary.getKey().endsWith(metadataFilename);
    }

    private String buildS3Key(String group, String objectKey, String metadataFilename) {
        if (objectKey.endsWith(metadataFilename)) {
            return objectKey;
        }
        return (S3StorageService.buildTypedFolder(this.rootFolder, group) + "/" + objectKey.toLowerCase() + "/" + metadataFilename).replace("//", "/");
    }

    private String buildObjectKey(ObjectType objectType, String s3Key) {
        return s3Key.replaceAll(S3StorageService.buildTypedFolder(this.rootFolder, objectType.group) + "/", "").replaceAll("/" + objectType.defaultMetadataFilename, "");
    }

    private static String buildTypedFolder(String rootFolder, String type) {
        return (rootFolder + "/" + type).replaceAll("//", "/");
    }

    public static enum ServerSideEncryption {
        AWSKMS,
        AES256;

    }
}

