package io.streamlayer.utils;

import android.content.Context;
import android.content.res.TypedArray;
import android.os.Build;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.AbsSavedState;
import android.view.MotionEvent;
import android.view.VelocityTracker;
import android.view.View;
import android.view.ViewConfiguration;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.core.os.ParcelableCompat;
import androidx.core.os.ParcelableCompatCreatorCallbacks;
import androidx.core.view.MotionEventCompat;
import androidx.core.view.NestedScrollingChild;
import androidx.core.view.VelocityTrackerCompat;
import androidx.core.view.ViewCompat;
import androidx.customview.widget.ViewDragHelper;

import com.google.android.material.bottomsheet.BottomSheetBehavior;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;


/**
 * Modification of BottomSheetBehavior to work in a horizontal direction, starting from left.
 */
class LeftSheetBehavior<V extends View> extends BottomSheetBehavior<V> {

    private static final String TAG = "LeftSheetBehavior";

    private static final float HIDE_THRESHOLD = 0.5f;

    private static final float HIDE_FRICTION = 0.1f;

    private float mMaximumVelocity;

    private int mPeekWidth;

    private int mMinOffset;

    private int mMaxOffset;

    private boolean mHideable;

    private boolean mSkipCollapsed;

    @State
    private int mState = STATE_COLLAPSED;
    int oldState = mState;
    private ViewDragHelper mViewDragHelper;
    private boolean mIgnoreEvents;
    private int mLastNestedScrollDx;
    private boolean mNestedScrolled;
    private int mParentWidth;
    private WeakReference<V> mViewRef;
    private WeakReference<View> mNestedScrollingChildRef;
    private List<BottomSheetCallback> callbacks = new ArrayList<BottomSheetCallback>();
    private VelocityTracker mVelocityTracker;
    private int mActivePointerId;
    private int mInitialX;
    private boolean mTouchingScrollingChild;
    private final ViewDragHelper.Callback mDragCallback = new ViewDragHelper.Callback() {

        @Override
        public boolean tryCaptureView(View child, int pointerId) {
            if (mState == STATE_DRAGGING) {
                return false;
            }
            if (mTouchingScrollingChild) {
                return false;
            }
            if (mState == STATE_EXPANDED && mActivePointerId == pointerId) {
                View scroll = mNestedScrollingChildRef.get();
                if (scroll != null && ViewCompat.canScrollHorizontally(scroll, -1)) {
                    // Let the content scroll left
                    return false;
                }
            }
            return mViewRef != null && mViewRef.get() == child;
        }

        @Override
        public void onViewPositionChanged(View changedView, int left, int top, int dx, int dy) {
            dispatchOnSlide(left);
        }

        @Override
        public void onViewDragStateChanged(int state) {
            if (state == ViewDragHelper.STATE_DRAGGING) {
                setStateInternal(STATE_DRAGGING);
            }
        }

        @Override
        public void onViewReleased(View releasedChild, float xvel, float yvel) {
            int left;
            @State int targetState;
            if (xvel > 0) { // Moving left
                left = mMaxOffset;
                targetState = STATE_EXPANDED;
            } else if (mHideable && shouldHide(releasedChild, xvel)) {
                left = -mViewRef.get().getWidth();
                targetState = STATE_HIDDEN;
            } else if (xvel == 0.f) {
                int currentLeft = releasedChild.getLeft();
                if (Math.abs(currentLeft - mMinOffset) > Math.abs(currentLeft - mMaxOffset)) {
                    left = mMaxOffset;
                    targetState = STATE_EXPANDED;
                } else {
                    left = mMinOffset;
                    targetState = STATE_COLLAPSED;
                }
            } else {
                left = mMinOffset;
                targetState = STATE_COLLAPSED;
            }
            if (mViewDragHelper.settleCapturedViewAt(left, releasedChild.getTop())) {
                setStateInternal(STATE_SETTLING);
                ViewCompat.postOnAnimation(releasedChild,
                        new SettleRunnable(releasedChild, targetState));
            } else {
                setStateInternal(targetState);
            }
        }

        @Override
        public int clampViewPositionVertical(View child, int top, int dy) {
            return child.getTop();
        }

        @Override
        public int clampViewPositionHorizontal(View child, int left, int dx) {
            return constrain(left, mHideable ? -child.getWidth() : mMinOffset, mMaxOffset);
        }

        @Override
        public int getViewHorizontalDragRange(@NonNull View child) {
            if (mHideable) {
                return child.getWidth();
            } else {
                return mMaxOffset - mMinOffset;
            }
        }
    };

    /**
     * Default constructor for instantiating LeftSheetBehaviors.
     */
    public LeftSheetBehavior() {
    }

    /**
     * Default constructor for inflating LeftSheetBehaviors from layout.
     *
     * @param context The {@link Context}.
     * @param attrs   The {@link AttributeSet}.
     */
    public LeftSheetBehavior(Context context, AttributeSet attrs) {
        super(context, attrs);
        TypedArray a = context.obtainStyledAttributes(attrs,
                com.google.android.material.R.styleable.BottomSheetBehavior_Layout);
        setPeekWidth(a.getDimensionPixelSize(
                com.google.android.material.R.styleable.BottomSheetBehavior_Layout_behavior_peekHeight, 0));
        setHideable(a.getBoolean(com.google.android.material.R.styleable.BottomSheetBehavior_Layout_behavior_hideable, false));
        setSkipCollapsed(a.getBoolean(com.google.android.material.R.styleable.BottomSheetBehavior_Layout_behavior_skipCollapsed,
                false));
        a.recycle();
        ViewConfiguration configuration = ViewConfiguration.get(context);
        mMaximumVelocity = configuration.getScaledMaximumFlingVelocity();
    }

    /**
     * A utility function to get the {@link LeftSheetBehavior} associated with the {@code view}.
     *
     * @param view The {@link View} with {@link LeftSheetBehavior}.
     * @return The {@link LeftSheetBehavior} associated with the {@code view}.
     */
    @SuppressWarnings("unchecked")
    public static <V extends View> LeftSheetBehavior<V> from(V view) {
        ViewGroup.LayoutParams params = view.getLayoutParams();
        if (!(params instanceof CoordinatorLayout.LayoutParams)) {
            throw new IllegalArgumentException("The view is not a child of CoordinatorLayout");
        }
        CoordinatorLayout.Behavior behavior = ((CoordinatorLayout.LayoutParams) params)
                .getBehavior();
        if (!(behavior instanceof LeftSheetBehavior)) {
            throw new IllegalArgumentException(
                    "The view is not associated with LeftSheetBehavior");
        }
        return (LeftSheetBehavior<V>) behavior;
    }

    static int constrain(int amount, int low, int high) {
        return amount < low ? low : (amount > high ? high : amount);
    }

    static float constrain(float amount, float low, float high) {
        return amount < low ? low : (amount > high ? high : amount);
    }

    @Override
    public Parcelable onSaveInstanceState(CoordinatorLayout parent, V child) {
        return new SavedState(super.onSaveInstanceState(parent, child), mState);
    }

    @Override
    public void onRestoreInstanceState(CoordinatorLayout parent, V child, Parcelable state) {
        SavedState ss = (SavedState) state;
        super.onRestoreInstanceState(parent, child, ss.getSuperState());
        // Intermediate states are restored as collapsed state
        if (ss.state == STATE_DRAGGING || ss.state == STATE_SETTLING) {
            mState = STATE_COLLAPSED;
        } else {
            mState = ss.state;
        }
    }

    @Override
    public boolean onLayoutChild(CoordinatorLayout parent, V child, int layoutDirection) {
        if (ViewCompat.getFitsSystemWindows(parent) && !ViewCompat.getFitsSystemWindows(child)) {
            ViewCompat.setFitsSystemWindows(child, true);
        }
        int savedLeft = child.getLeft();
        // First let the parent lay it out
        parent.onLayoutChild(child, layoutDirection);
        // Offset the left sheet
        mParentWidth = parent.getWidth();
        mMinOffset = Math.max(-child.getWidth(), -(child.getWidth() - mPeekWidth));
        mMaxOffset = 0;
        if (mState == STATE_EXPANDED) {
            ViewCompat.offsetLeftAndRight(child, mMaxOffset);
        } else if (mHideable && mState == STATE_HIDDEN) {
            ViewCompat.offsetLeftAndRight(child, -child.getWidth());
        } else if (mState == STATE_COLLAPSED) {
            ViewCompat.offsetLeftAndRight(child, mMinOffset);
        } else if (mState == STATE_DRAGGING || mState == STATE_SETTLING) {
            ViewCompat.offsetLeftAndRight(child, savedLeft - child.getLeft());
        }
        if (mViewDragHelper == null) {
            mViewDragHelper = ViewDragHelper.create(parent, mDragCallback);
        }
        mViewRef = new WeakReference<>(child);
        mNestedScrollingChildRef = new WeakReference<>(findScrollingChild(child));
        return true;
    }

    @Override
    public boolean onInterceptTouchEvent(CoordinatorLayout parent, V child, MotionEvent event) {
        if (!child.isShown()) {
            return false;
        }
        int action = MotionEventCompat.getActionMasked(event);
        // Record the velocity
        if (action == MotionEvent.ACTION_DOWN) {
            reset();
        }
        if (mVelocityTracker == null) {
            mVelocityTracker = VelocityTracker.obtain();
        }
        mVelocityTracker.addMovement(event);
        switch (action) {
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                mTouchingScrollingChild = false;
                mActivePointerId = MotionEvent.INVALID_POINTER_ID;
                // Reset the ignore flag
                if (mIgnoreEvents) {
                    mIgnoreEvents = false;
                    return false;
                }
                break;
            case MotionEvent.ACTION_DOWN:
                int initialY = (int) event.getY();
                mInitialX = (int) event.getX();
                View scroll = mNestedScrollingChildRef.get();
                if (scroll != null && parent.isPointInChildBounds(scroll, mInitialX, initialY)) {
                    mActivePointerId = event.getPointerId(event.getActionIndex());
                    mTouchingScrollingChild = true;
                }
                mIgnoreEvents = mActivePointerId == MotionEvent.INVALID_POINTER_ID &&
                        !parent.isPointInChildBounds(child, mInitialX, initialY);
                break;
        }
        if (!mIgnoreEvents && mViewDragHelper.shouldInterceptTouchEvent(event)) {
            return true;
        }
        // We have to handle cases that the ViewDragHelper does not capture the left sheet because
        // it is not the left most view of its parent. This is not necessary when the touch event is
        // happening over the scrolling content as nested scrolling logic handles that case.
        View scroll = mNestedScrollingChildRef.get();
        return action == MotionEvent.ACTION_MOVE && scroll != null &&
                !mIgnoreEvents && mState != STATE_DRAGGING &&
                !parent.isPointInChildBounds(scroll, (int) event.getX(), (int) event.getY()) &&
                Math.abs(mInitialX - event.getX()) > mViewDragHelper.getTouchSlop();
    }

    @Override
    public boolean onTouchEvent(CoordinatorLayout parent, V child, MotionEvent event) {
        if (!child.isShown()) {
            return false;
        }
        int action = MotionEventCompat.getActionMasked(event);
        if (mState == STATE_DRAGGING && action == MotionEvent.ACTION_DOWN) {
            return true;
        }
        if (mViewDragHelper != null) {
            //no crash
            mViewDragHelper.processTouchEvent(event);
            // Record the velocity
            if (action == MotionEvent.ACTION_DOWN) {
                reset();
            }
            if (mVelocityTracker == null) {
                mVelocityTracker = VelocityTracker.obtain();
            }
            mVelocityTracker.addMovement(event);
            // The ViewDragHelper tries to capture only the left-most View. We have to explicitly tell it
            // to capture the left sheet in case it is not captured and the touch slop is passed.
            if (action == MotionEvent.ACTION_MOVE && !mIgnoreEvents) {
                if (Math.abs(mInitialX - event.getX()) > mViewDragHelper.getTouchSlop()) {
                    mViewDragHelper.captureChildView(child, event.getPointerId(event.getActionIndex()));
                }
            }
        }
        return !mIgnoreEvents;
    }

    @Override
    public boolean onStartNestedScroll(CoordinatorLayout coordinatorLayout, V child,
                                       View directTargetChild, View target, int nestedScrollAxes) {
        mLastNestedScrollDx = 0;
        mNestedScrolled = false;
        return (nestedScrollAxes & ViewCompat.SCROLL_AXIS_VERTICAL) != 0;
    }

    @Override
    public void onNestedPreScroll(CoordinatorLayout coordinatorLayout, V child, View target, int dx,
                                  int dy, int[] consumed) {
        View scrollingChild = mNestedScrollingChildRef.get();
        if (target != scrollingChild) {
            return;
        }
        int currentLeft = child.getLeft();
        int newLeft = currentLeft - dx;
        if (dx > 0) { // Right
            if (!ViewCompat.canScrollHorizontally(target, 1)) {
                if (newLeft >= mMinOffset || mHideable) {
                    consumed[1] = dx;
                    ViewCompat.offsetLeftAndRight(child, -dx);
                    setStateInternal(STATE_DRAGGING);
                } else {
                    consumed[1] = currentLeft - mMinOffset;
                    ViewCompat.offsetLeftAndRight(child, -consumed[1]);
                    setStateInternal(STATE_COLLAPSED);
                }
            }
        } else if (dx < 0) { // Left
            // Negative to check scrolling left, positive to check scrolling down
            if (newLeft < mMaxOffset) {
                consumed[1] = dx;
                ViewCompat.offsetLeftAndRight(child, -dx);
                setStateInternal(STATE_DRAGGING);
            } else {
                consumed[1] = currentLeft - mMaxOffset;
                ViewCompat.offsetLeftAndRight(child, -consumed[1]);
                setStateInternal(STATE_EXPANDED);
            }
        }
        dispatchOnSlide(child.getLeft());
        mLastNestedScrollDx = dx;
        mNestedScrolled = true;
    }

    @Override
    public void onStopNestedScroll(CoordinatorLayout coordinatorLayout, V child, View target) {
        if (child.getLeft() == mMaxOffset) {
            setStateInternal(STATE_EXPANDED);
            return;
        }
        if (target != mNestedScrollingChildRef.get() || !mNestedScrolled) {
            return;
        }
        int left;
        int targetState;
        if (mLastNestedScrollDx < 0) {
            left = mMaxOffset;
            targetState = STATE_EXPANDED;
        } else if (mHideable && shouldHide(child, getXVelocity())) {
            left = -child.getWidth();
            targetState = STATE_HIDDEN;
        } else if (mLastNestedScrollDx == 0) {
            int currentLeft = child.getLeft();
            if (Math.abs(currentLeft - mMinOffset) > Math.abs(currentLeft - mMaxOffset)) {
                left = mMaxOffset;
                targetState = STATE_EXPANDED;
            } else {
                left = mMinOffset;
                targetState = STATE_COLLAPSED;
            }
        } else {
            left = mMinOffset;
            targetState = STATE_COLLAPSED;
        }
        if (mViewDragHelper.smoothSlideViewTo(child, left, child.getTop())) {
            setStateInternal(STATE_SETTLING);
            ViewCompat.postOnAnimation(child, new SettleRunnable(child, targetState));
        } else {
            setStateInternal(targetState);
        }
        mNestedScrolled = false;
    }

    @Override
    public boolean onNestedPreFling(CoordinatorLayout coordinatorLayout, V child, View target,
                                    float velocityX, float velocityY) {
        return target == mNestedScrollingChildRef.get() &&
                (mState != STATE_EXPANDED ||
                        super.onNestedPreFling(coordinatorLayout, child, target,
                                velocityX, velocityY));
    }

    /**
     * Gets the width of the left sheet when it is collapsed.
     *
     * @return The width of the collapsed left sheet.
     * @attr ref com.google.android.material.R.styleable#RightSheetBehavior_Layout_behavior_peekWidth
     */
    public final int getPeekWidth() {
        return mPeekWidth;
    }

    /**
     * Sets the width of the left sheet when it is collapsed.
     *
     * @param peekWidth The width of the collapsed left sheet in pixels.
     * @attr ref com.google.android.material.R.styleable#LeftSheetBehavior_Params_behavior_peekWidth
     */
    public final void setPeekWidth(int peekWidth) {
        mPeekWidth = Math.max(0, peekWidth);
//        mMaxOffset = mParentWidth - peekWidth;
        if (mViewRef != null && mViewRef.get() != null) {
            mMinOffset = Math.max(-mViewRef.get().getWidth(), -(mViewRef.get().getWidth() - mPeekWidth));
        }
    }

    /**
     * Gets whether this left sheet can hide when it is swiped down.
     *
     * @return {@code true} if this left sheet can hide.
     * @attr ref com.google.android.material.R.styleable#RightSheetBehavior_Layout_behavior_hideable
     */
    public boolean isHideable() {
        return mHideable;
    }

    /**
     * Sets whether this left sheet can hide when it is swiped down.
     *
     * @param hideable {@code true} to make this left sheet hideable.
     * @attr ref com.google.android.material.R.styleable#RightSheetBehavior_Layout_behavior_hideable
     */
    public void setHideable(boolean hideable) {
        mHideable = hideable;
    }

    /**
     * Sets whether this left sheet should skip the collapsed state when it is being hidden
     * after it is expanded once.
     *
     * @return Whether the left sheet should skip the collapsed state.
     * @attr ref com.google.android.material.R.styleable#RightSheetBehavior_Layout_behavior_skipCollapsed
     */
    public boolean getSkipCollapsed() {
        return mSkipCollapsed;
    }

    /**
     * Sets whether this left sheet should skip the collapsed state when it is being hidden
     * after it is expanded once. Setting this to true has no effect unless the sheet is hideable.
     *
     * @param skipCollapsed True if the left sheet should skip the collapsed state.
     * @attr ref com.google.android.material.R.styleable#RightSheetBehavior_Layout_behavior_skipCollapsed
     */
    public void setSkipCollapsed(boolean skipCollapsed) {
        mSkipCollapsed = skipCollapsed;
    }

    /**
     * Sets a callback to be notified of sheet events.
     *
     * @param callback The callback to notify when bottom sheet events occur.
     * @deprecated use {@link #addBottomSheetCallback(BottomSheetCallback)} and {@link
     * #removeBottomSheetCallback(BottomSheetCallback)} instead
     */
    @Override
    @Deprecated
    public void setBottomSheetCallback(BottomSheetCallback callback) {
        Log.w(
                TAG,
                "BottomSheetBehavior now supports multiple callbacks. `setBottomSheetCallback()` removes"
                        + " all existing callbacks, including ones set internally by library authors, which"
                        + " may result in unintended behavior. This may change in the future. Please use"
                        + " `addBottomSheetCallback()` and `removeBottomSheetCallback()` instead to set your"
                        + " own callbacks.");
        callbacks.clear();
        if (callback != null) {
            callbacks.add(callback);
        }
    }

    /**
     * Adds a callback to be notified of sheet events.
     *
     * @param callback The callback to notify when sheet events occur.
     */
    @Override
    public void addBottomSheetCallback(@NonNull BottomSheetCallback callback) {
        if (!callbacks.contains(callback)) {
            callbacks.add(callback);
        }
    }

    /**
     * Removes a previously added callback.
     *
     * @param callback The callback to remove.
     */
    @Override
    public void removeBottomSheetCallback(@NonNull BottomSheetCallback callback) {
        callbacks.remove(callback);
    }

    /**
     * Gets the current state of the left sheet.
     *
     * @return One of {@link #STATE_EXPANDED}, {@link #STATE_COLLAPSED}, {@link #STATE_DRAGGING},
     * and {@link #STATE_SETTLING}.
     */
    @State
    public final int getState() {
        return mState;
    }

    /**
     * Sets the state of the left sheet. The left sheet will transition to that state with
     * animation.
     *
     * @param state One of {@link #STATE_COLLAPSED}, {@link #STATE_EXPANDED}, or
     *              {@link #STATE_HIDDEN}.
     */
    public final void setState(@State int state) {
        if (state == mState) {
            return;
        }
        if (mViewRef == null) {
            // The view is not laid out yet; modify mState and let onLayoutChild handle it later
            if (state == STATE_COLLAPSED || state == STATE_EXPANDED ||
                    (mHideable && state == STATE_HIDDEN)) {
                mState = state;
            }
            return;
        }
        V child = mViewRef.get();
        if (child == null) {
            return;
        }
        int left;
        if (state == STATE_COLLAPSED) {
            left = mMinOffset;
        } else if (state == STATE_EXPANDED) {
            left = mMaxOffset;
        } else if (mHideable && state == STATE_HIDDEN) {
            left = -child.getWidth();
        } else {
            throw new IllegalArgumentException("Illegal state argument: " + state);
        }
        setStateInternal(STATE_SETTLING);
        if (mViewDragHelper.smoothSlideViewTo(child, left, child.getTop())) {
            ViewCompat.postOnAnimation(child, new SettleRunnable(child, state));
        }
    }

    private void setStateInternal(@State int state) {
        if (state == LeftSheetBehavior.STATE_COLLAPSED || state == LeftSheetBehavior.STATE_EXPANDED) {
            oldState = state;
        }

        if (mState == state) {
            return;
        }
        mState = state;
        View leftSheet = mViewRef.get();
        if (leftSheet != null) {
            for (int i = 0; i < callbacks.size(); i++) {
                callbacks.get(i).onStateChanged(leftSheet, state);
            }
        }
    }

    private void reset() {
        mActivePointerId = ViewDragHelper.INVALID_POINTER;
        if (mVelocityTracker != null) {
            mVelocityTracker.recycle();
            mVelocityTracker = null;
        }
    }

    private boolean shouldHide(View child, float xvel) {
        if (child.getLeft() > mMinOffset) {
            // It should not hide, but collapse.
            return false;
        }
        final float newLeft = child.getLeft() + xvel * HIDE_FRICTION;
        return Math.abs(newLeft - mMinOffset) / (float) mPeekWidth > HIDE_THRESHOLD;
    }

    private View findScrollingChild(View view) {
        if (view instanceof NestedScrollingChild) {
            return view;
        }
        if (view instanceof ViewGroup) {
            ViewGroup group = (ViewGroup) view;
            for (int i = 0, count = group.getChildCount(); i < count; i++) {
                View scrollingChild = findScrollingChild(group.getChildAt(i));
                if (scrollingChild != null) {
                    return scrollingChild;
                }
            }
        }
        return null;
    }

    private float getXVelocity() {
        mVelocityTracker.computeCurrentVelocity(1000, mMaximumVelocity);
        return VelocityTrackerCompat.getXVelocity(mVelocityTracker, mActivePointerId);
    }

    private void dispatchOnSlide(int left) {
        View leftSheet = mViewRef.get();
        if (leftSheet != null && !callbacks.isEmpty()) {

            Boolean isOpening = oldState == LeftSheetBehavior.STATE_COLLAPSED;

            if (left < mMinOffset) {
                for (int i = 0; i < callbacks.size(); i++) {
                    callbacks.get(i).onSlide(leftSheet, (float) (left - mMinOffset) / mPeekWidth);
                }
            } else {
                for (int i = 0; i < callbacks.size(); i++) {
                    callbacks.get(i).onSlide(leftSheet, (float) (left - mMinOffset) / ((mMaxOffset - mMinOffset)));
                }
            }
        }
    }

    protected static class SavedState extends AbsSavedState {
        public static final Creator<SavedState> CREATOR = ParcelableCompat.newCreator(
                new ParcelableCompatCreatorCallbacks<SavedState>() {
                    @RequiresApi(api = Build.VERSION_CODES.N)
                    @Override
                    public SavedState createFromParcel(Parcel in, ClassLoader loader) {
                        return new SavedState(in, loader);
                    }

                    @Override
                    public SavedState[] newArray(int size) {
                        return new SavedState[size];
                    }
                });
        @State
        final int state;

        @RequiresApi(api = Build.VERSION_CODES.N)
        public SavedState(Parcel source) {
            this(source, null);
        }

        @RequiresApi(api = Build.VERSION_CODES.N)
        public SavedState(Parcel source, ClassLoader loader) {
            super(source, loader);
            //noinspection ResourceType
            state = source.readInt();
        }

        public SavedState(Parcelable superState, @State int state) {
            super(superState);
            this.state = state;
        }

        @Override
        public void writeToParcel(Parcel out, int flags) {
            super.writeToParcel(out, flags);
            out.writeInt(state);
        }
    }

    private class SettleRunnable implements Runnable {

        private final View mView;

        @State
        private final int mTargetState;

        SettleRunnable(View view, @State int targetState) {
            mView = view;
            mTargetState = targetState;
        }

        @Override
        public void run() {
            if (mViewDragHelper != null && mViewDragHelper.continueSettling(true)) {
                ViewCompat.postOnAnimation(mView, this);
            } else {
                setStateInternal(mTargetState);
            }
        }
    }
}